;+
; NAME:
;         p3d_misc_mean_smooth_small
;
;         $Id: p3d_misc_mean_smooth_small.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         For a two-dimensional input array (ARRAY) this routine creates a
;         collapsed version where pixel values along the dispersion direction
;         (DAXIS) have been averaged in bands, which are 2*SMOWIDTH+1 pixels
;         wide and are centered on specified pixels (POS). Additionally, POS is
;         returned as OUTPOS, where pixels have been filtered out which are <0
;         and larger than the dispersion dimension of ARRAY (s[DAXIS]).
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_mean_smooth_small,array,pos,smowidth,out,outpos, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         array           - A two-dimensional array of floating point values.
;         pos             - A one-dimensional array of floating point values,
;                           which specify pixels in the dispersion
;                           direction, for which the smoothing is made.
;         smowidth        - A scalar decimal value. The average value is
;                           calculated for a dispersion region of width
;                           2*smowidth+1.
;
; KEYWORD PARAMETERS:
;         daxis [1]       - Defines the dispersion direction dimension of
;                           IMAGE. The default, 1, is in the x-direction.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - A one-dimensional array of decimal values
;                           containing the cross-dispersion direction averaged
;                           spectrum values.
;         outpos          - A one-dimensional array of integers. This array is
;                           the same as POS, where negative values, and values
;                           which are larger than the maximum number of pixels
;                           in the dispersion direction of ARRAY are filtered
;                           out.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         06.10.2008 - Converted from the original routine mean_smo_small of
;                      Thomas Becker. /CS
;-
PRO p3d_misc_mean_smooth_small,array,pos,smowidth,out,outpos,daxis=daxis, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error, $
        debug=debug,help=help
  compile_opt hidden,IDL2

  error=0 & rname='p3d_misc_mean_smooth_small: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_misc_mean_smooth_small'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  if ~n_elements(daxis) then daxis=1L
  s=size(daxis)
  if s[s[0L]+2L] ne 1L or (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  sid=daxis?2L:1L

  s=size(array)
  if s[0L] ne 2L or (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='ARRAY must be a two-dimensional array of floating point type.'
    goto,error_handler
  endif

  sb=size(pos)
  if sb[0L] ne 1L or (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='POS must be set; to a one-dimensional array of floating point ' + $
           'values.'
    goto,error_handler
  endif
  pos=long(pos)
  pos2=pos

  sb=size(smowidth)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='SMOWIDTH must be set; to a decimal scalar value.'
    goto,error_handler
  endif

  outpos=where(pos ge 0L and pos le s[daxis]-1L,count)
  if count ne 0L then begin
    outpos=pos2[outpos]
  endif else begin
    out=array
    msg='No valid reference position given [POS]!'
    error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
        verbose=verbose ge 1)
    return
  endelse
  n=n_elements(outpos)

  ;;========================================------------------------------
  ;; Calculating the average values:

  mtmp=float(s[daxis]-1L)

  if daxis then begin

    out=fltarr(n,s[sid])
    for k=0L,n-1L do begin
      lower=outpos[k]-smowidth>0.0
      upper=outpos[k]+smowidth<mtmp
      out[k,*]=total(array[lower:upper,*],daxis)/(upper-lower+1L)
    endfor

  endif else begin

    out=fltarr(s[sid],n)
    for k=0L,n-1L do begin
      lower=outpos[k]-smowidth>0.0
      upper=outpos[k]+smowidth<mtmp
      out[*,k]=total(array[*,lower:upper],daxis)/(upper-lower+1L)
    endfor

  endelse ;; daxis

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_misc_mean_smooth_small
