;+
; NAME:
;         p3d_display_tele
;
;         $Id: p3d_display_tele.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         This routine is a replacement for TV that can also enlarge the image
;         and assign the data a color table.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: display routine
;
; CALLING SEQUENCE:
;         p3d_display_tele,array,lcut,hcut,xsize,ysize,xpos=,ypos=,cubic=, $
;             interp=,minus_one=,true=,bottom=, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         array           - A 2D-array, which must be compatible with float.
;         lcut            - float, color=  0 is set to the data value lcut.
;         hcut            - float, color=255 is set to the data value hcut.
;
; OPTINAL INPUTS:
;         xsize [1-Dim. of array] - x-size of the displayed image
;         ysize [2-Dim. of array] - y-size of the displayed image
;
; KEYWORD PARAMETERS:
;         xpos [0]        - A scalar integer; defines the x-position of the
;                           lower left corner.
;         ypos [0]        - A scalar integer; defines the y-position of the
;                           lower left corner.
;         cubic [0]       - A scalar float -1<=cubic<=0. Determines if cubic
;                           interpolation is to be used.
;         interp [0]      - Uses linear interpolation if set.
;         minus_one [0]   - Interpolates on image borders when set.
;         true [0]        - Sets the TRUE keyword of TV (0<=TRUE<=3).
;         bottom [0]      - The lower index used in the color table.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done
;         error           - Returns an error code if set
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         04.09.2008 - Converted from original routine tele.pro of
;                      Thomas Becker. /CS
;-
PRO p3d_display_tele,array,lcut,hcut,xsize,ysize,xpos=xpos,ypos=ypos, $
        cubic=cb,interp=ip,minus_one=mo,true=true,bottom=bottom, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error, $
        debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_display_tele: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_display_tele'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking input arguments:

  if n_params() gt 5L or ~n_params() then begin
    errmsg='The number of input parameters P must satisfy 0<P<5.'
    goto,error_handler
  endif else begin

    if ~n_elements(true) then true=0L
    s=size(true)
    if s[s[0L]+2L] gt 1L or $
      (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
      errmsg='TRUE must be of integer, type; 0<=TRUE<=3.'
      goto,error_handler
    endif
    if true lt 0L or true gt 3L then begin
      errmsg='TRUE must be of integer, type; 0<=TRUE<=3.'
      goto,error_handler
    endif

    s=size(array)
    if (s[0L] ne 2L and s[0L] ne 1L and ~true) or $
       (s[0L] ne 3L and true ge 1L) or $
       ((s[1L+s[0L]] ge 6L and s[1L+s[0L]] le 11L) or ~s[1L+s[0L]]) then begin
      errmsg='ARRAY must have 1-2 dim. (true=0) or 3 dim. (true>=1),' + $
             'and have 1<=type<=5, or type>=12.'
      goto,error_handler
    endif else begin
      if n_params() lt 5L then if s[0L] eq 1L then ysize=1L else ysize=s[2L]
      if n_params() lt 4L then xsize=s[1L]
      x=size(xsize) & y=size(ysize) 
      if x[0L] ne 0L or y[0] ne 0L or $
         x[1L+x[0L]] lt 2L or x[1L+x[0L]] gt 3L or $
         y[1L+y[0L]] lt 2L or y[1L+y[0L]] gt 3L then error=-1 else begin
        if xsize lt 0 or ysize lt 0 then begin
          errmsg='XSIZE and/or YSIZE <0. This is not allowed.'
          goto,error_handler
        endif
      endelse
    endelse
  endelse

  if ~n_elements(lcut) or ~n_elements(hcut) or $
     (size(lcut,/type) ge 6L and size(lcut,/type) le 11L) or $
     (size(hcut,/type) ge 6L and size(hcut,/type) le 11L) then begin
    errmsg='LCUT and HCUT must be set to a floating point or integer value.'
    goto,error_handler
  endif
  lcut_=float(lcut)
  hcut_=float(hcut)

  if ~n_elements(xpos) then xpos=0L
  if (size(xpos,/type) ge 4L and size(xpos,/type) le 11L) or $
     (size(ypos,/type) ge 4L and size(ypos,/type) le 11L) then begin
    errmsg='XPOS and YPOS must be of integer type.'
    goto,error_handler
  endif
  if ~n_elements(ypos) then ypos=0L
  if ~n_elements(cb) then cb=0L
  if size(cb,/type) ge 6L and size(cb,/type) ne 14L then begin
    errmsg='CUBIC must be of float, or signed integer, type.'
    goto,error_handler
  endif
  if ~keyword_set(ip) then ip=0L
  if ~keyword_set(mo) then mo=0L

  if ~n_elements(bottom) then bottom=0L
  if bottom gt !d.table_size-1L then begin
    errmsg='BOTTOM ['+strtrim(bottom,2L)+'] is larger than !D.TABLE_SIZE ['+ $
        strtrim(!d.table_size,2L)+'].'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Rescaling:

  image=true ge 1L? $
        array:(float(array)-lcut_)/(hcut_-lcut_)*(!d.table_size-bottom)+bottom

  ;; Changing to indexed colors, if necessary:
  if ~(!d.name eq 'PS') then begin
    device,get_decomposed=decomposed
    if decomposed then device,decomposed=0L
  endif

  tmp=congrid(image,xsize,ysize,true,cubic=cb,interp=ip,minus_one=mo)
  tmp__=true ge 1L?tmp:tmp>bottom<(!d.table_size-1L)
  tv,tmp__,xpos,ypos,true=true

  ;; Changing back to the entry color model:
  if ~(!d.name eq 'PS') then begin
    if decomposed then device,decomposed=decomposed
  endif

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_display_tele
