;+
; NAME:
;         p3d_misc_findspec
;
;         $Id: p3d_misc_findspec.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         For a defined input value this routine checks if this value exists in
;         a list of defined dead/non-used fiber values. If it does then the
;         value is decremented/incremented until a value is found that is not
;         in the list. The found value is then returned.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_findspec,specnum,maxnum,deadfibers,logstr=,topwid=, $
;             logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         specnum         - Initial spectrum to look for an emission line in
;                           (in the cross-dispersion direction). This value is
;                           checked against the values in the DEADFIBERS array.
;                           If it is present there then the value is decreased
;                           until a value is found that is not in that array.
;                           If the value 0 is reached then the search is
;                           started over by increasing the value, starting with
;                           the initial value, until a non-dead number is
;                           found. The maximum value is given by MAXNUM.
;                           The following -must- be satisfied:
;                             0<=SPECNUM<MAXNUM.
;         maxnum          - A scalar integer specifying the maximum allowed
;                           number+1 of the return value.
;         deadfibers      - A one-dimensional array of integers specifying
;                           which fiber positions are to be interpolated
;                           instead of fitted. If DEADFIBERS is not specified
;                           then the input value SPECNUM is returned as is.
;
; KEYWORD PARAMETERS:
;         logstr          - Returns a scalar string with the outcome of the
;                           value search for logging purposes.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         return_value    - A scalar integer with the number of the
;                           spectrum to use.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         16.11.2009 - Routine introduced. /CS
;-
FUNCTION p3d_misc_findspec,specnum,maxnum,deadfibers_,logstr=logstr, $
    topwid=topwid,logunit=logunit,verbose=verbose,error=error, $
    debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_misc_findspec: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_misc_findspec'
    return,0
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return,error
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  if ~n_elements(deadfibers_) then return,specnum

  spc=size(maxnum)
  if spc[spc[0L]+2L] ne 1L or $
    (spc[spc[0L]+1L] ge 6L and spc[spc[0L]+1L] le 11L) then begin
    errmsg='MAXNUM [2] must be set to a scalar integer; MAXNUM>0.'
    goto,error_handler
  endif
  if maxnum le 0L then begin
    errmsg='MAXNUM [2] must be set to a scalar integer; MAXNUM>0.'
    goto,error_handler
  endif

  spc=size(specnum)
  if spc[spc[0L]+2L] ne 1L or $
    (spc[spc[0L]+1L] ge 6L and spc[spc[0L]+1L] le 11L) then begin
    errmsg='SPECNUM [1] must be set; to a scalar value of integer type.'
    goto,error_handler
  endif
  if specnum lt 0L or specnum ge maxnum then begin
    errmsg='0<=SPECNUM (='+strtrim(specnum,2L)+')<'+strtrim(maxnum,2L)+ $
           ' does not hold.'
    goto,error_handler
  endif
  specnum_=specnum

  spc=size(deadfibers_)
  if ~spc[spc[0L]+2L] or $
     (spc[spc[0L]+1L] ge 4L and spc[spc[0L]+1L] le 11L) then begin
    errmsg='DEADFIBERS [3] must be set to an array of integer type.'
    goto,error_handler
  endif
  if min(deadfibers_) lt 0L or max(deadfibers_) gt maxnum then begin
    errmsg='DEADFIBERS [3] must be set to an array of integer type.'
    goto,error_handler
  endif
  deadfibers=deadfibers_-1L

  ;;========================================------------------------------
  ;; Finding a suitable number:

  iterate=1L & negloop=1L & logstr=''
  while iterate do begin
    tmp=where(deadfibers eq specnum,count)
    if ~count then begin
      iterate=0L
    endif else begin
      if ~specnum then begin
        negloop=0L
        specnum=specnum_
      endif
      if specnum eq maxnum-1L and ~negloop then iterate=0L else $
         specnum=negloop?specnum-1L:specnum+1L
      logstr=' ['+(negloop?'decremented':'incremented')+ $
             ' due to dead fibers]'
    endelse
  endwhile ;; iterate

  return,specnum

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return,error
END ;;; function: p3d_misc_findspec
