;+
; NAME:
;         p3d_misc_mmodhdr (Multi block Modify HeaDeR)
;
;         $Id: p3d_misc_mmodhdr.pro 115 2010-03-19 12:17:32Z christersandin $
;
; PURPOSE:
;         For detectors that are read out to several files (blocks) this file
;         adds some block-specific header keywords to the input fits header
;         variable.
;
;         It is for these detectors, moreover, likely an issue that the gain
;         is different for the separate files. This creates a problem since the
;         extracted spectra are stored in the ADU, and it is not possible to
;         know what the gain of every pixel is after the data has been reduced
;         (The error can still be calculated properly).
;
;         This routine solves this issue by first calculating a mean value on
;         the gain (of all blocks). Thereafter the data (not the prescan and
;         overscan regions though) are normalized to the same value of the
;         gain.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_mmodhdr,nblocks,hdr,data,gain,detsec=,userparfile=, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         nblocks         - A scalar integer specifying the number of blocks
;                           on the detector.
;
; INPUT/OUTPUTS:
;         hdr             - A string array that contains a fits file header.
;         data            - A two-dimensional image of floating point type that
;                           at input uses the gain values of GAIN for the
;                           different blocks. On output DATA has been
;                           normalized to use GAIN instead across the entire
;                           detector.
;
; KEYWORD PARAMETERS:
;         detsec          - A four-element (columns) -by- number of blocks
;                           (rows) integer array that specifies the detector
;                           region to use on the CCD for each block. For each
;                           row the first two elements are used with the
;                           x-axis, and the second two elements with the
;                           y-axis.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
;
;   These keywords are passed over directly to p3d_misc_getinformation:
;         filename
;         kwrdlist
;         userparfile
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
;-
PRO p3d_misc_mmodhdr,nblocks,filename,kwrdlist,hdr,data,detsec=detsec, $
        userparfile=userparfile,topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_misc_mmodhdr: '
  if ~n_elements(verbose) then verbose=0
  if ~n_elements(topwid) then topwid=0L
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_misc_mmodhdr'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(nblocks) & n=s[s[0L]+2L]
  if ~n or (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='NBLOCKS [1] must be set to a scalar integer.'
    goto,error_handler
  endif

  ;; Nothing is done if there is only one element in GAINARR:
  if nblocks eq 1L then return

  s=size(filename)
  if s[0L] ne 2L or s[s[0L]+1L] ne 7L then begin
    errmsg='FILENAME must be set to a two-dimensional array of strings.'
    goto,error_handler
  endif

  s=size(kwrdlist)
  if ~s[s[0L]+2L] or s[s[0L]+1L] ne 7L then begin
    errmsg='KWRDLIST must be set to a string array.'
    goto,error_handler
  endif

  s=size(hdr)
  if ~s[s[0L]+2L] or s[s[0L]+1L] ne 7L then begin
    errmsg='HDR must be set to a string array.'
    goto,error_handler
  endif

  s=size(data)
  if s[0L] ne 2L or (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='DATA [2] must be set to a decimal array.'
    goto,error_handler
  endif

  s=size(detsec)
  if ~s[s[0L]+2L] then begin
    errmsg='DETSEC must be specified.'
    goto,error_handler
  endif
  if s[0L] ne 2L or $
    (s[0L] eq 2L and (s[1L] ne 4L or s[2L] ne nblocks)) or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='DETSEC must be set to a two-dimensional array with 4,NBLOCKS='+ $
           strtrim(nblocks,2L)+' columns,rows (not '+strtrim(s[1L],2L)+','+ $
           strtrim(s[2L,2L])+').'
    goto,error_handler
  endif
  if min(detsec) lt 0L then begin
    errmsg='DETSEC contains elements which are <0.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Including block-related information to the header:

  gainarr=dblarr(nblocks) & rdn=gainarr
  for j=0L,nblocks-1L do begin
    dtmp='['+strtrim(detsec[0L,j]+1L,2L)+ $
         ':'+strtrim(detsec[1L,j]+1L,2L)+ $
         ','+strtrim(detsec[2L,j]+1L,2L)+ $
         ':'+strtrim(detsec[3L,j]+1L,2L)+']'
    tmp=strtrim(j+1L,2L)+'/'+strtrim(nblocks,2L)+'.'
    fxaddpar,hdr,'CETSEC'+strtrim(j+1L,2L),dtmp,' DETSEC: block '+tmp

    p3d_misc_getinformation,filename[0L,j],kwrdlist, $
        gain=sgain,rdnoise=rdnoise,userparfile=userparfile,index=j+1L, $
        kwgain=kwgain,topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug
    if error ne 0 then return
    gainarr[j]=sgain & rdn[j]=rdnoise
  endfor ;; j=0L,nblocks-1L

  for j=0L,nblocks-1L do begin
    tmp=strtrim(j+1L,2L)+'/'+strtrim(nblocks,2L)+'.'
    fxaddpar,hdr,'CRDNOI'+strtrim(j+1L,2L),rdn[j],' RDNOISE block '+tmp
  endfor ;; j=0L,nblocks-1L

  ;;========================================------------------------------
  ;; Calculating a mean gain, and calculating correction factors for the
  ;; different blocks:

  gain=total(gainarr,/double)/nblocks
  carr=gainarr/gain

  ;; Replacing the header gain keyword with the mean value:
  fxaddpar,hdr,kwgain,gain,' e-/ADU gain (mean of all blocks)'

  ;;========================================------------------------------
  ;; Normalizing the data:

  xmin=min(min(detsec[0L:1L,*],dimension=1L),minjx)
  xmax=max(max(detsec[0L:1L,*],dimension=1L),maxjx)
  tmp=min(detsec[0L:1L,maxjx])-max(detsec[0L:1L,minjx])-1L
  xoff=~tmp?0L:tmp
  xmax-=xoff

  ymin=min(min(detsec[2L:3L,*],dimension=1L),minjy)
  ymax=max(max(detsec[2L:3L,*],dimension=1L),maxjy)
  tmp=min(detsec[2L:3L,maxjy])-max(detsec[2L:3L,minjy])-1L
  yoff=~tmp?0L:tmp
  ymax-=yoff

  xmint=xmin
  ymint=ymin
  for j=0L,nblocks-1L do begin
    xmin=~(min(detsec[0L:1L,j])-xmint)?xmint:(max(detsec[0L:1L,minjx])+1L)
    xmax=xmin+max(detsec[0L:1L,j])-min(detsec[0L:1L,j])

    ymin=~(min(detsec[2L:3L,j])-ymint)?ymint:(max(detsec[2L:3L,minjy])+1L)
    ymax=ymin+max(detsec[2L:3L,j])-min(detsec[2L:3L,j])

    data[xmin:xmax,ymin:ymax]*=carr[j]
  endfor ;; i=0L,nblocks-1L

  ;; Logging the operations:
  msg=['Got '+strtrim(nblocks,2L)+' input values on the gain for the diff' + $
       'erent detector blocks:', $
       '  '+strjoin(string(gainarr,format='(f6.4)'),', ',/single), $
       'Calculated a mean value on the gain to replace these values: '+ $
       string(format='(f6.4)',gain), $
       'Normalized all blocks of the data to use the same gain value.']
  error=p3d_misc_logger(msg,logunit,rname=rname, $
      topwid=topwid,verbose=verbose ge 1L)
  if error ne 0 then return

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
            verbose=verbose,/error)
  return
END ;;; procedure: p3d_misc_mmodhdr
