;+
; NAME:
;         p3d_extract_optimal_mox
;
;         $Id: p3d_extract_optimal_mox.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         This routine extracts spectra from a two-dimensional spectrum image
;         using the modified optimal extraction of
;         Sandin et al. 2010, A&A, in press.
;         Optimal extraction weights are calculated according to the approach
;         of Horne, K. 1986, PASP, 98, 609-617 (all equations in the comments
;         refer to this paper).
;
;         The spectra are extracted using (pre-calculated) input profiles (over
;         the cross-dispersion axis) to sum up the flux at every wavelength
;         bin. The input image is turned into an extracted spectrum image,
;         where every spectrum is placed in an individual row.
;
;         In order to correct for cross-talk the extraction is iterated;
;         thereby recalculating the fraction every profile occupies of the
;         total profile at every pixel. The total profile is the sum of all
;         profiles in the cross-dispersion direction.
;
;         This method also allows for removal of cosmic rays, although note
;         that it may be difficult to remove them efficiently. Also note that
;         the flux is just removed if this method is used, no flux replacement
;         or interpolation is used.
;
;         The following parameters are read from the user parameter file:
;          eliminate_crays ['no'] ::
;               If this keyword is set then the extraction is iterated in order
;               to eliminate probable cosmic ray hits across the cross-
;               dispersion profile of every spectrum. The maximum number of
;               iterations is CR_MAXIT, and the threshold that determines if
;               the data of a pixel is a cosmic ray hit is defined by CR_SIGMA.
;                Note! Cosmic rays are only removed using this approach if no
;                      cosmic ray mask has been specified (CRMASK).
;          correct_crosstalk ['no'] ::
;               If this keyword is set then the cross-dispersion profiles are
;               rescaled in an iteration procedure in order to correct for
;               fiber-to-fiber cross-talk before the spectra are finally
;               extracted. The condition that is used to determine when the
;               iterations can be finished is if the maximum number of
;               iterations has been reached (CTALK_MAXIT) or if the maximum
;               change in the scaling factors of two consecutive iterations is
;               smaller than CTALK_EPS.
;          cr_sigma [10] ::
;               A scalar decimal value specifying the threshold that is used to
;               see which pixels in the data of a profile are cosmic ray
;               events.
;          cr_maxit [2] ::
;               A scalar integer specifying the maximum number of iterations
;               that are performed during the iterations to eliminate cosmic
;               ray events in the profiles.
;          ctalk_eps [1d-5] ::
;               A scalar decimal value specifying the maximum allowed value of
;               the difference between the integrated spectrum of two
;               consecutive spectra.
;          ctalk_maxit [15] ::
;               A scalar integer specifying the maximum number of iterations
;               that are performed during the iterations to correct for fiber-
;               to-fiber cross-talk.
;          ctalk_secit [1] ::
;               A scalar integer specifying the maximum number of iterations
;               that are performed in the secondary loop (where the raw data
;               isn't used anymore when calculating the variance). This keyword
;               is only used if crosstalk correction is used (secit is
;               always==1 otherwise).
;          ctalk_reiterate ['no'] ::
;               If this keyword is set then the full spectrum extraction
;               procedure is iterated two times. The purpose of the first
;               iteration is to calculate a first version fractional profile
;               for every spectrum. Pixels may, in this process, be incorrectly
;               masked as cosmic rays. In particular this is the case if
;               spectra are tightly packed (in which case neighboring spectra
;               are found to be cosmic rays. In the second iteration the first
;               version fractional profile is used initially (instead of 1.0);
;               this will decrease the number of elements which are incorrectly
;               masked as cosmic rays.
;          pmultfac [10] ::
;               A scalar integer specifying a subsampling factor that is used
;               when calculating the profile.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: spectrum extraction
;
; CALLING SEQUENCE:
;         p3d_extract_optimal_mox,im,dim,lprofs,out,dout,profwidth=, $
;             detsec=,gain=,rdnoise=,proffun=,/ecalc,userparfile=,stawid=, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         im              - A two-dimensional array, that holds the spectra
;                           that will be extracted here. The dispersion axis
;                           must be the x-axis.
;         dim             - A two-dimensional array of the same dimensions as
;                           IM, with the errors of IM. DIM must be present.
;                           The variance of IM is DIM.
;         lprofs [sp=size(lprofs)]
;                         - A three-dimensional array, that for every spectrum
;                           bin, of every spectrum, holds the fitting
;                           parameters of a cross-dispersion profile.
;
; KEYWORD PARAMETERS:
;         profwidth       - 2*PROFWIDTH+1 is the total pixel-width of the
;                           interval where the flux is integrated.
;         detsec          - A four-element (columns) -by- number of blocks
;                           (rows) integer array that specifies the detector
;                           region to use on the CCD for each block. RDNOISE
;                           must have as many elements as DETSEC has rows.
;         gain            - The data gain factor [e-/ADU]; a scalar decimal
;                           value.
;         rdnoise         - A decimal scalar or decimal array that specifies
;                           the readout noise. RDNOISE must have as many
;                           elements as DETSEC has rows.
;         proffun         - A scalar string with the name of the function to
;                           use when (re-)calculating the line profile.
;         usecrmask [0]   - If this keyword is unset then the MOX-method cosmic
;                           ray removal option can be activated. Otherwise it
;                           is assumed cosmic rays are already treated.
;         ecalc [0]       - If this keyword is set then output errors are
;                           calculated.
;         userparfile     - A scalar string specifying the name of an optional
;                           user parameter file, that could contain any of the
;                           keywords that are described in the routine
;                           description.
;         stawid          - If set, then various messages are written to the
;                           p3d GUI status line (this must be the widget id of
;                           that label widget).
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - A two-dimensional array of the extracted spectra
;                           with the same dimensions as the first two
;                           dimensions of LPROFS.
;         dout            - A two-dimensional array of the extracted spectra
;                           with the same dimensions as OUT. This is the error
;                           of OUT. DOUT is onlyt present if ECALC is set.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
;-
PRO p3d_extract_optimal_mox,im,dim_,lprofs,out,dout,profwidth=profwidth_, $
        detsec=detsec,gain=gain,rdnoise=rdnoise_,proffun=proffun_, $
        usecrmask=usecrmask,ecalc=ecalc,userparfile=userparfile, $
        stawid=stawid,topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_extract_optimal_mox: '
  if ~n_elements(verbose) then verbose=0
  usestawid=~n_elements(stawid)?0L:widget_info(stawid,/valid_id)
  debug=keyword_set(debug)
  loglevel=~n_elements(logunit)?0L:logunit[1L]

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_extract_optimal_mox'
    return
  endif ;; keyword_set(help) or ~n_params()

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(im)
  if ~s[s[0L]+2L] or s[0L] ne 2L or $
     (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='IM [1] must be set to a two-dimensional array of decimal type.'
    goto,error_handler
  endif
  ncim=s[2L]

  se=size(dim_)
  if ~se[se[0L]+2L] or (se[se[0L]+1L] ge 6L and se[se[0L]+1L] le 11L) or $
     (se[se[0L]+2L] ge 1L and (se[0L] ne s[0L] or se[1L] ne s[1L] or $
                               se[se[0L]+2L] ne s[s[0L]+2L])) then begin
    errmsg='DIM [2] {'+strtrim(se[se[0L]+1L],2L)+',['+strtrim(se[1L],2L)+ $
           ','+strtrim(se[2L],2L)+']} must be of the same ' + $
           'dimensions as IM {'+strtrim(s[s[0L]+1L],2L)+',['+ $
           strtrim(s[1L],2L)+','+strtrim(s[2L],2L)+']}.'
    goto,error_handler
  endif
  dim=dim_

  sp=size(lprofs)
  if ~sp[sp[0L]+2L] or sp[0L] ne 3L or $
     (sp[sp[0L]+1L] ge 6L and sp[sp[0L]+1L] le 11L) or $
     (sp[0L] eq 3L and (sp[1L] ne s[1L])) then begin
    errmsg='LPROFS [3] must be set to a three-dimensional array of decimal' + $
           ' type, with as any elements in the first dimension as IM.'
    goto,error_handler
  endif
  nwl =sp[1L]
  nprf=sp[2L]

  sz=size(profwidth_)
  if sz[sz[0L]+2L] ne 1L or $
    (sz[sz[0L]+1L] ge 6L and sz[sz[0L]+1L] le 11L) then begin
    errmsg='PROFWIDTH must be set to a scalar decimal value; PROFWIDTH>=0.'
    goto,error_handler
  endif
  if profwidth_ le 0d0 then begin
    errmsg='PROFWIDTH must be set to a scalar decimal value; PROFWIDTH>=0.'
    goto,error_handler
  endif
  profwidth=profwidth_

  tmp='scalar'
  nblocks=1L
  if n_elements(detsec) ne 0L then begin
    sd=size(detsec)
    nblocks=(sd[0L] eq 1L)?1L:sd[2L]
    if (sd[0L] eq 1L and sd[1L] ne 4L) or $
       (sd[0L] eq 2L and sd[1L] ne 4L) or $
       (sd[sd[0L]+1L] ge 4L and sd[sd[0L]+1L] le 11L) then begin
      errmsg='DETSEC must have the dimensions [4,NBLOCKS], not ['+ $
             strtrim(sd[1L],2L)+','+strtrim(sd[2L],2L)+'].'
      goto,error_handler
    endif
    tmp='4x'+strtrim(nblocks,2L)+'-element array'
  endif ;; nblocks gt 1L

  sb=size(gain)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='GAIN must be set to a decimal scalar; GAIN>0.'
    goto,error_handler
  endif
  if gain lt 0d0 then begin
    errmsg='GAIN must be set to a decimal scalar; GAIN>0.'
    goto,error_handler
  endif

  sb=size(rdnoise_)
  if ~sb[sb[0L]+2L] or $
     (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='RDNOISE must be set to a decimal '+tmp+'; RDNOISE>0.'
    goto,error_handler
  endif
  if min(rdnoise_) lt 0d0 or sb[sb[0L]+2L] ne nblocks then begin
    errmsg='RDNOISE must be set to a decimal '+tmp+'; RDNOISE>0.'
    goto,error_handler
  endif

  sb=size(proffun_)
  if sb[sb[0L]+2L] ne 1L or sb[sb[0L]+1L] ne 7L then begin
    errmsg='PROFFUN must be set to a scalar string with the name of the fu' + $
           'nction to use.'
    goto,error_handler
  endif

  proffun=strlowcase(proffun_)
  case proffun of
    'gaussian':      iidx=2L
    'lorentzian':    iidx=2L
    'gauss/lorentz': iidx=3L
    'doublegauss':   iidx=4L
    else: begin
      errmsg=['PROFFUN must be one of the four options:', $
              '  "gaussian", "lorentzian", "gauss/lorentz", "doublegauss"', $
              ' PROFFUN="'+proffun_+'" is not a valid option.']
      goto,error_handler
    end
  endcase ;; proffun

  if n_elements(userparfile) ne 0L then begin
    sp=size(userparfile)
    if sp[sp[0L]+2L] ne 1L or sp[sp[0L]+1L] ne 7L then begin
      errmsg='USERPARFILE must, if set, be a scalar string.'
      goto,error_handler
    endif
    if userparfile[0L] ne '' then begin
      if ~file_test(userparfile,/regular,/read) then begin
        errmsg='USERPARFILE, cannot read the file "'+userparfile+'".'
        goto,error_handler
      endif

      ;; Reading the user parameter file data:
      uparname_='' & uparvalue_=''
      readcol,userparfile,uparname_,uparvalue_,format='a,a',comment=';', $
          silent=verbose lt 3,delimiter=' '
      if n_elements(uparname_) ne 0L then begin
        uparname=uparname_ & uparval=uparvalue_
      endif ;; nelements(uparname_) ne 0L
    endif ;; userparfile[0L] ne ''
  endif ;; n_elements(userparfile) ne 0L

  ecalc=keyword_set(ecalc)

  ;;========================================------------------------------
  ;;========================================------------------------------
  ;; Reading various MOX-related parameters:

  ;;====================---------------
  ;; Profile subsampling factor:

  p3d_misc_read_params,uparname,uparval,'pmultfac',pmultfac,/upo,/a0, $
      type='integer',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(pmultfac) then pmultfac=10L

  sp=size(pmultfac)
  if ~sp[sp[0L]+2L] or $
     (sp[sp[0L]+1L] ge 4L and sp[sp[0L]+1L] le 11L) then begin
    errmsg='PMULTFAC must be set to a scalar integer; PMULTFAC>0.'
    goto,error_handler
  endif
  if pmultfac le 0L then begin
    errmsg='PMULTFAC must be set to a scalar integer; PMULTFAC>0.'
    goto,error_handler
  endif

  ;;====================---------------
  ;; Eliminate cosmic rays:

  eliminate_crays=0L
  if ~keyword_set(usecrmask) then begin
    p3d_misc_read_params,uparname,uparval,'eliminate_crays',elcr,/upo,/a0, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return
    if n_elements(elcr) ne 0L then $
       eliminate_crays=strlowcase(elcr) eq 'yes'?1L:0L
  endif ;; ~keyword_set(usecrmask)

  ;;====================---------------
  ;; Cross-talk correction:

  p3d_misc_read_params,uparname,uparval,'correct_crosstalk',cocr,/upo,/a0, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(cocr) then correct_crosstalk=0L
  if  n_elements(cocr) ne 0L then $
     correct_crosstalk=strlowcase(cocr) eq 'yes'?1L:0L

  ;;====================---------------
  ;; Eliminate-cosmic-rays threshold:

  p3d_misc_read_params,uparname,uparval,'cr_sigma',cr_sigma,/upo,/a0, $
      type='float',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(cr_sigma) then cr_sigma=10d0

  sp=size(cr_sigma)
  if ~sp[sp[0L]+2L] or $
     (sp[sp[0L]+1L] ge 6L and sp[sp[0L]+1L] le 11L) then begin
    errmsg='CR_SIGMA must be set to a scalar decimal value; CR_SIGMA>0.'
    goto,error_handler
  endif
  if cr_sigma le 0d0 then begin
    errmsg='CR_SIGMA must be set to a scalar decimal value; CR_SIGMA>0.'
    goto,error_handler
  endif

  ;;====================---------------
  ;; Eliminate-cosmic-rays maximum number of iterations:

  p3d_misc_read_params,uparname,uparval,'cr_maxit',cr_maxit,/upo,/a0, $
      type='integer',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(cr_maxit) then cr_maxit=2L

  sp=size(cr_maxit)
  if ~sp[sp[0L]+2L] or $
     (sp[sp[0L]+1L] ge 4L and sp[sp[0L]+1L] le 11L) then begin
    errmsg='CR_MAXIT must be set to a scalar integer; CR_MAXIT>0.'
    goto,error_handler
  endif
  if cr_maxit le 0L then begin
    errmsg='CR_MAXIT must be set to a scalar integer; CR_MAXIT>0.'
    goto,error_handler
  endif

  ;;====================---------------
  ;; Cross-talk maximum difference between two consecutive iterations:

  p3d_misc_read_params,uparname,uparval,'ctalk_eps',ctalk_eps,/upo,/a0, $
      type='float',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(ctalk_eps) then ctalk_eps=1d-5

  sp=size(ctalk_eps)
  if ~sp[sp[0L]+2L] or $
     (sp[sp[0L]+1L] ge 6L and sp[sp[0L]+1L] le 11L) then begin
    errmsg='CTALK_EPS must be set to a scalar decimal value; CTALK_EPS>0.'
    goto,error_handler
  endif
  if ctalk_eps le 0d0 then begin
    errmsg='CTALK_EPS must be set to a scalar decimal value; CTALK_EPS>0.'
    goto,error_handler
  endif

  ;;====================---------------
  ;; Cross-talk maximum number of iterations:

  p3d_misc_read_params,uparname,uparval,'ctalk_maxit',ctalk_maxit,/upo,/a0, $
      type='integer',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(ctalk_maxit) then ctalk_maxit=15L

  sp=size(ctalk_maxit)
  if ~sp[sp[0L]+2L] or $
     (sp[sp[0L]+1L] ge 4L and sp[sp[0L]+1L] le 11L) then begin
    errmsg='CTALK_MAXIT must be set to a scalar integer; CTALK_MAXIT>0.'
    goto,error_handler
  endif
  if ctalk_maxit le 0L then begin
    errmsg='CTALK_MAXIT must be set to a scalar integer; CTALK_MAXIT>0.'
    goto,error_handler
  endif

  ;;====================---------------
  ;; Cross-talk secondary iteration loop maximum number of iterations:

  p3d_misc_read_params,uparname,uparval,'ctalk_secit',ctalk_secit,/upo,/a0, $
      type='integer',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(ctalk_secit) then ctalk_secit=1L

  sp=size(ctalk_secit)
  if ~sp[sp[0L]+2L] or $
     (sp[sp[0L]+1L] ge 4L and sp[sp[0L]+1L] le 11L) then begin
    errmsg='CTALK_SECIT must be set to a scalar integer; CTALK_SECIT>0.'
    goto,error_handler
  endif
  if ctalk_secit le 0L then begin
    errmsg='CTALK_SECIT must be set to a scalar integer; CTALK_SECIT>0.'
    goto,error_handler
  endif

  ;;====================---------------
  ;; Cross-talk reiteration flag:

  p3d_misc_read_params,uparname,uparval,'ctalk_reiterate',/a0, $
      ctalk_reiterate,/upo,topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  if n_elements(ctalk_reiterate) ne 0L then $
     ctalk_reiterate=strlowcase(ctalk_reiterate) eq 'yes'?1L:0L
  if ~n_elements(ctalk_reiterate) then ctalk_reiterate=0L

  sb=size(ctalk_reiterate)
  if sb[sb[0L]+1L] ge 4L and sb[sb[0L]+1L] le 11L then begin
    errmsg='CTALK_REITERATE must be set to an integer scalar; ' + $
           'CTALK_REITERATE=0||1.'
    goto,error_handler
  endif
  if ctalk_reiterate ne 0L and ctalk_reiterate ne 1L then begin
    errmsg='CTALK_REITERATE must be set to an integer scalar; ' + $
           'CTALK_REITERATE=0||1.'
    goto,error_handler
  endif
  if ~eliminate_crays or ~correct_crosstalk then ctalk_reiterate=0L


  ;;========================================------------------------------
  ;;========================================------------------------------
  ;; Preparing the calculations

  c=reform(lprofs[*,*,0L])
  n=2L*profwidth+1L
  c=round(c) & clo=c-profwidth

  out=dblarr(nwl,nprf)
  if ecalc then dout=out

  ;; The array that will hold the normalized line profiles:
  prof=dblarr(nwl,nprf,n)

  ;; The array that will hold the normalized line profiles:
  x0=rebin(dindgen(n)-profwidth,n,pmultfac)
  x1=rebin(dindgen(1L,pmultfac)/pmultfac,n,pmultfac)
  x=x0+x1

  ;;========================================------------------------------
  ;; Resizing RDNOISE to full frame arrays in order to easily sum up
  ;; values of spectra which extend across several CCD blocks:

  rdnoise=dblarr(nwl,ncim)
  case nblocks of
    1: rdnoise+=rdnoise_
    else: begin
      xmint=min(detsec[0L:1L,*])
      ymint=min(detsec[2L:3L,*])
      for j=0L,nblocks-1L do begin
        xoff=min(detsec[0L:1L,j]) ne xmint?xmint:0L
        yoff=min(detsec[2L:3L,j]) ne ymint?ymint:0L

        xmin=min(detsec[0L:1L,j])-xmint-xoff
        xmax=max(detsec[0L:1L,j])-xmint-xoff
        ymin=min(detsec[2L:3L,j])-ymint-yoff
        ymax=max(detsec[2L:3L,j])-ymint-yoff

        rdnoise[xmin:xmax,ymin:ymax]=rdnoise_[j]
      endfor ;; j=0L,nblocks-1L
    end ;; else
  endcase ;; nblocks

  ;;========================================------------------------------
  ;; Logging the used parameters:

  msg='Using the extraction method: Modified Horne'
  msg=[msg,'Using the following extraction parameters:',' profwidth='+ $
       string(profwidth,format='(f9.3)')+' :: spectrum extraction half w' + $
      'idth.']
  msg=[msg,'  Profile subsampling factor='+ $
       string(format='(i9)',pmultfac)+'.']
  tmpstr=eliminate_crays?'yes':' no'
  msg=[msg,'   Eliminating cosmic ray events: '+tmpstr+'.']
  if eliminate_crays then begin
    if n_elements(cr_sigma) eq 1L then $
       msg=[msg,'  Cosmic rays, sigma='+string(format='(e9.2)',cr_sigma)+'.']
    if n_elements(cr_maxit) eq 1L then $
       msg=[msg,'  Cosmic rays, maxit='+string(format='(i9)',cr_maxit)+'.']
  endif

  tmpstr=correct_crosstalk?'yes':' no'
  msg=[msg,'  Applying cross-talk correction: '+tmpstr+'.']
  if correct_crosstalk then begin
    if n_elements(ctalk_eps) eq 1L then $
       msg=[msg,'  Cross-talk,   eps='+string(format='(e9.2)',ctalk_eps)+'.']
    if n_elements(ctalk_maxit) eq 1L then $
       msg=[msg,'  Cross-talk, maxit='+string(format='(i9)',ctalk_maxit)+'.']
    if n_elements(ctalk_reiterate) eq 1L then $
       msg=[msg,'  Cross-talk, reiterate='+(ctalk_reiterate?'yes':' no')+'.']
  endif

  if nblocks eq 1L then begin
    msg=[msg, $
         '         gain='+string(gain,format='(f9.3)')+' :: [e-/ADU].', $
         '      rdnoise='+string(rdnoise_,format='(f9.3)')+' :: [ADU].']
  endif else begin
    msg=[msg,'         gain='+string(gain,format='(f9.3)')+' :: [e-/ADU].']
    for j=0L,nblocks-1L do begin
      jstr=strtrim(j+1L,2L)+'/'+strtrim(nblocks,2L)
      msg=[msg,'    rdnoise '+jstr+' = '+ $
           string(rdnoise_[j],format='(f9.3)')+' [ADU].']
    endfor ;; j=0L,nblocks-1L
  endelse ;; nblocks eq 1L
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  ;;========================================------------------------------
  ;; Extracting pixel-based cross-dispersion profiles for every spectrum and
  ;; wavelength bin:

  wlstr='/'+strtrim(nwl,2L)
  i=-1L & while ++i lt nwl do begin

    if usestawid and ~(i mod 100L) then begin
      tmp=strtrim(i+1L,2L)+'-'+strtrim((i+100L)<nwl,2L)
      msg='[Optimal spectrum extraction] extracting pixel-based profiles;'+ $
          ' w.l.bins '+tmp+wlstr
      widget_control,stawid,set_value=msg
    endif ;; usestawid and ~(i mod 100L)

    j=-1L & while ++j lt nprf do begin
      f=p3d_misc_profunction(x+c[i,j],lprofs[i,j,*],proffun=proffun, $
            /nobg,topwid=topwid,error=error,verbose=verbose)
      if error ne 0 then return

      if pmultfac gt 1L then f=total(f,2L)/pmultfac

      ;;==============================--------------------
      ;; Normalizing every pixel-based cross-dispersion profile:

      if lprofs[i,j,iidx] ne 0d0 then prof[i,j,*]=f/total(f)

    endwhile ;; ++j lt nprf
  endwhile ;; ++i lt nwl

  ;;========================================------------------------------
  ;; Extracting spectra using modified optimal extraction:
  ;;   o according to the approach of Horne (1986)
  ;;   o additionally correcting for fiber-to-fiber cross-talk according to
  ;;     Sandin et al. 2010, A&A, in press

  spc=dblarr(nprf,n)
  var=dblarr(nprf,n)

  fprof=dblarr(nwl,nprf,n)+1d0 ;; fractional profile
  sfactor=dblarr(nwl,nprf)+1d0 ;; cross-disp. profile scaling factor

  ;;==============================--------------------
  ;; Making a reiterating loop - if requested:

  ipx=lindgen(n) & itimes=dblarr(nwl)

  for kk=0L,ctalk_reiterate do begin

    ;; Reiteration logging:
    if ctalk_reiterate then begin 
      tmp=kk?'Reiterating...using the 1st version fractional profile.': $
          'Calculating a 1st version fractional profile.'
      msg=' Iterating the spectrum extraction [step '+strtrim(kk+1L,2L)+ $
          '/2] :: '+tmp
      error=p3d_misc_logger(msg,logunit,loglevel=2,rname=rname, $
          verbose=verbose ge 2)
    endif ;; ctalk_reiterate

    ;;==============================--------------------
    ;; Looping over all wavelength bins:

    i=-1L & while ++i lt nwl do begin
      itimes[i]=systime(1L)

      mask=lonarr(nprf,n)+1L

      if usestawid and ~(i mod 50L) then begin
        tmp=strtrim(i+1L,2L)+'-'+strtrim((i+50L)<nwl,2L)
        tmp_=ctalk_reiterate?strtrim(kk+1L,2L)+'/2 ':''
        msg='[Optimal spectrum extraction] '+tmp_+'w.l.bins '+tmp+wlstr
        if i gt 0L then begin
          tmp=total(itimes[0L:i-1L])
          tmp=strtrim(round(tmp*(double(nwl)/i-1d0)),2L)
          msg+=' :: remaining time: ~'+strtrim(tmp,2L)+'s.'
        endif else msg+='.'
        widget_control,stawid,set_value=msg
      endif

      j=-1L & while ++j lt nprf do $
         spc[j,*]=im[i,clo[i,j]+ipx]*reform(fprof[i,j,*],1L,n)

      ;;==============================--------------------
      ;; Iterating the solution; iterating <=CTALK_MAXIT (twice) times with
      ;; (without) cr.-talk corr.:

      cit=0L & rit=0L & csec=0L & crmasked=0L
      done=correct_crosstalk and eliminate_crays?1L:(eliminate_crays?1L:2L)

      repeat begin ;; done ge 4L

        ;;==============================--------------------
        ;; Correcting for cross-talk - calculating fractional profiles:

        if correct_crosstalk and (~done or ~(done-2L) or ~(done-3L)) then begin

          ;; Calculating a total profile:
          if ~(kk and ~cit) then begin
            tprof=dblarr(s[2L])
            j=-1L & while ++j lt nprf do $      ;; looping over all spectra
               tprof[clo[i,j]+ipx]+=sfactor[i,j]*reform(prof[i,j,*])
          endif ;; ~(kk and ~cit)

          ;; Calculating a profile scaling factor:
          j=-1L & while ++j lt nprf do begin    ;; looping over all spectra

            ;; Skipping profiles with an intensity=0.0:
            if lprofs[i,j,iidx] eq 0d0 then continue

            ;; Calculating a fractional profile:
            if ~(kk and ~cit) then begin
              fprof[i,j,*]=sfactor[i,j]*prof[i,j,*]/ $
                           reform(tprof[clo[i,j]+ipx],1L,1L,n)
            endif ;; ~(kk and ~cit)

            ;; Setting the image array, anew:
            spc[j,*]=im[i,clo[i,j]+ipx]*reform(fprof[i,j,*],1L,n)

          endwhile ;; ++j lt nprf

        endif ;;  correct_crosstalk and  (~done or ~(done-2L))

        ;;==============================--------------------
        ;; Calc. the variance (and hence the optimal extraction weights):

        case done of
          3L: begin ;; Iterating the error estimate; eq.(13)
            j=-1L & while ++j lt nprf do begin ;; looping over all spectra
              var[j,*]=rdnoise[i,clo[i,j]+ipx]^2 + $
                       abs(out[i,j])*reform(prof[i,j,*],1L,n)/gain
            endwhile ;; ++j lt nprf
          end ;; 3L
          else: begin  ;; This is the initial error estimate (would be eq.12)
            j=-1L & while ++j lt nprf do begin ;; looping over all spectra
              tnx=clo[i,j]+ipx
              var[j,*]=dim[i,clo[i,j]+ipx]^2 ;*reform(fprof[i,j,*],1L,n)
            endwhile ;; ++j lt nprf
          end ;; 0L
        endcase ;; done

        ;;==============================--------------------
        ;; Extracting optimal spectra:

        idx=where(reform(lprofs[i,*,iidx]) gt 0d0,nidx)
        if nidx gt 0L then begin

          ;; Weights; eq.(7):
          weights=mask[idx,*]*(reform(prof[i,idx,*],nidx,n)^2/ $
                               (var[idx,*]*reform(fprof[i,idx,*],nidx,n)))

          ;; Summed weights; eq.(9):
          sweight=1d0/total(weights,2L)

          ;; Error of optimal spectrum; Table 1-8b:
          if ecalc then $
             dout[i,idx]=sqrt(total(mask[idx,*]* $
                                    reform(prof[i,idx,*],nidx,n),2L)*sweight)

          ;; Optimal spectrum; Table 1-8a:
          out[i,idx]=sweight*reform(total(spc[idx,*]*weights/ $
                         reform(prof[i,idx,*],nidx,n),2L),1L,nidx)

          ;;==============================--------------------
          ;; Finalizing secondary iteration round:

          if ~(done-3L) then begin
            done_sec=1L
            if correct_crosstalk then begin
              psfactor=sfactor[i,*]
              sfactor[i,idx]=out[i,idx]
              facarr=abs(sfactor[i,*]/psfactor[0L,*])-1d0
              midx=0L & sspec=max(facarr[idx],midx)
              done_sec=sspec lt ctalk_eps or ++csec gt ctalk_secit-1L?1L:0L
            endif
            if done_sec then done++
          endif

          ;;==============================--------------------
          ;; Elimination of CRs; Table 1-7, also Section II.E., p.614:

          done_cr=0L
          if eliminate_crays and ~(done-1L) then begin
            svar=mask[idx,*]*((spc[idx,*]- $
                               rebin(reform(out[i,idx]),nidx,n)* $
                               reform(prof[i,idx,*]))^2-var[idx,*]*cr_sigma^2)
            changed=0L
            j=-1L & while ++j lt nidx do begin    ;; looping over all spectra
;             max=max(svar[idx[j],*],index)
              max=max(svar[j,*],index)
              if max gt 0d0 then begin
                mask[idx[j],index]=0L & changed++ & crmasked++
              endif ;; max gt 0d0
            endwhile ;; ++j lt nidx

            done_cr=++rit gt cr_maxit-1L or ~changed?1L:0L
          endif ;; eliminate_crays and ~(done-1L)

          ;;==============================--------------------
          ;; Finalizing the cross-talk correction procedure:

          if correct_crosstalk and (~done or ~(done-2L)) then begin
            psfactor=sfactor[i,*] ;; Storing the current scaling factor array
            sfactor[i,idx]=out[i,idx] ;; calculating the scaling factor anew

            facarr=abs(sfactor[i,*]/psfactor[0L,*])-1d0
            midx=0L
            dspec=~cit and ~kk?1d0:max(facarr[idx],midx)

            if ~done then done++ else begin
              done_ct=(dspec lt ctalk_eps or ++cit gt ctalk_maxit-1L)?1L:0L
              if done_ct then $
                 if ctalk_reiterate and ~kk then done+=2L else done++
            endelse ;; ~done then done++

          endif else if ~correct_crosstalk and ~(done-2L) then begin
            done++
          endif ;; correct_crosstalk and ~done or ~(done-2L)

          if eliminate_crays and done_cr and ~(done-1L) then $
             done=correct_crosstalk?2L:3L

        endif ;; nidx gt 0L
      endrep until done ge 4L

      ;;========================================-----------------------------
      ;; Logging information about the cosmic ray elimination:

      if eliminate_crays then begin
        tmp=string(format='(i4,"/",i4)',i+1L,nwl)
        msg='  wavelength bin '+tmp+' :: #CR-elimination iterations=' + $
            string(format='(i3)',rit)+', masked totally ' + $
            strtrim(crmasked,2L)+' pixels.'
        error=p3d_misc_logger(msg,logunit,loglevel=2,rname=rname, $
            verbose=verbose ge 2)
      endif ;; eliminate_crays

      ;;========================================-----------------------------
      ;; Logging information about the crosstalk-correction:

      tmpstr=string(replicate(32b,26))
      if correct_crosstalk then begin
        tmp=eliminate_crays?tmpstr: $
            string(format='("  wavelength bin ",i4,"/",i4)',i+1L,nwl)
        msg=tmp+' ::     #Cross-talk iterations='+ $
            string(format='(i3)',cit)+ $
            ', max(delta(scale factor))='+string(dspec,format='(e12.5)')+'.'
        error=p3d_misc_logger(msg,logunit,loglevel=2,rname=rname, $
            verbose=verbose ge 2)
      endif ;; correct_crosstalk

      ;;========================================-----------------------------
      ;; Logging information about the secondary iterations:

      if csec gt 0L then begin
        tmp=eliminate_crays or correct_crosstalk?tmpstr: $
            string(format='("  wavelength bin ",i4,"/",i4)',i+1L,nwl)
        msg=tmp+' ::      #Secondary iterations='+ $
            string(format='(i3)',csec)+ $
            ', max(delta(scale factor))='+string(sspec,format='(e12.5)')+'.'
        error=p3d_misc_logger(msg,logunit,loglevel=2,rname=rname, $
            verbose=verbose ge 2)
      endif ;;  csec gt 0L

      itimes[i]=systime(1L)-itimes[i]

    endwhile ;;  ++i lt nwl

  endfor ;; kk=0L,reiterate

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_extract_optimal_mox
