;+
; NAME:
;         p3d_wavecal_dispmask_gui
;
;         $Id: p3d_wavecal_dispmask_gui.pro 155 2010-04-16 13:50:27Z christersandin $
;
; PURPOSE: 
;         This is an interactive program for creating a dispersion mask. That
;         is, a file listing the best-fit dispersion solution parameters for
;         each spectrum in the input image. The GUI layout displays this input
;         RSS image in the top panel. Below that a plot of an individual
;         spectrum ("reference spectrum") is shown.
;
;         The initial dispersion mask is a set of straight lines given by the
;         input linelist.  This mask is overlaid on the RSS image. The user can
;         modify the initial mask as follows:
;           o Match the shape of the mask lines to the curvature of the
;             emission lines in the RSS image.
;           o Shift and centroid the mask so that its spatially aligned with
;             the emission lines in the RSS image.
;           o Optionally delete mask lines from the input line list.
;           o Inspect the best-fit dispersion solution for the reference
;             spectrum and change the order of the best-fit polynomial if
;             necessary.
;           o When the match between the mask and the RSS image is satisfactory
;             the output dispersion mask is created and saved as a FITS file.
;
;         Format of files with line lists:
;           o Lines which begin with the character ';' are comments.
;           o Individual lines are specified with the wavelength (given in
;             Angstroms []) in the first column of the file. Columns are
;             separated with white space.
;           o Only data in the first column is read. The data must be of
;             floating point type, as it subsequently is converted to doubles.
;
; AUTHOR:
;         Christer Sandin and Joris Gerssen
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: wavelength calibration
;
; CALLING SEQUENCE:
;         p3d_wavecal_dispmask_gui,imagefile,linelist,gratingfile,ofilename, $
;             linewidth=,refdist=,nrows=,kwrdlist=,cdelt=,xdeg=,ydeg=,saved=, $
;             colortable=,cindex=,cindv=,bottom=,residualcut=,fwhm=,postable=,$
;             deadfibers=,sdeadfibers=,/removecalfibers,userparfile=, $
;             imageorder=,imagesplitbin=,/dflip,/compress,dbin=,track=,$
;             daxis=,topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         imagefile       - One, or two, filename(s) of an exposure, which was
;                           taken using one, or several, arc lamps. If
;                           IMAGEFILE is actually two image files, then they
;                           are combined using the values of IMAGEORDER and
;                           IMAGESPLITBIN:
;                            IMAGEORDER==0:: The used image uses the wavelength
;                              bins 0L:IMAGESPLITBIN-2L from IMAGEFILE[0L]
;                              and IMAGESPLITBIN: from IMAGEFILE[1L]. 
;                            IMAGEORDER==1:: The used image uses the wavelength
;                              bins 0L:IMAGESPLITBIN-2L from IMAGEFILE[1L]
;                              and IMAGESPLITBIN: from IMAGEFILE[0L]. 
;                           The lowermost value of IMAGESPLITBIN is 1.
;         linelist        - The filename of a file specifying which wavelengths
;                           are emission lines of the arc lamp.
;         gratingfile     - The filename of an instrument-specific file that
;                           provides grating-related parameters.
;         ofilename       - The filename of the output dispersion mask.
;
; KEYWORD PARAMETERS:
;         linewidth [4]   - A scalar integer that specifies the half-pixel-
;                           width that is used when searching for maxima of
;                           emission lines in the spectrum image
;                           (of IMAGEFILE).
;         refdist [1]     - A scalar integer specifying if any spectrum is to
;                           be skipped. If REFDIST==1 then every spectrum is
;                           used, if REFDIST==2 then every second spectrum
;                           is used, etc. (cf. REFDIST in the file
;                           p3d_wavecal_fit_maskwd).
;         nrows [0]       - A scalar integer specifying how many adjacent
;                           spectra should be used in the fitting procedure
;                           (the actual number is 2*NROWS+1).
;         kwrdlist        - The name of a file, which contains a two-column
;                           list of parameters to use with p3d for the
;                           instrument in question.
;         cdelt           - The dispersion (only used when PMAS==0).
;         xdeg [3]        - The degree of the polynomial which is used to fit
;                           wavelengths of emission lines to the whole
;                           spectrum.
;         ydeg [-1]       - The degree of the polynomial which is used to fit
;                           wavelengths of emission lines in the cross-
;                           dispersion direction; YDEG=-1 skips this fitting.
;         colortable [-1] - A scalar integer or string. If COLORTABLE is a
;                           string, then this string must point to a color
;                           table file that can be read by loadct. If it is an
;                           integer then the integer must be given in the range
;                           -1,0,...,40. While -1 loads the 'Sauron'
;                           colortable, the other values use the respective
;                           colormap as defined by loadct.
;         cindex          - An array of integers specifying the color indices
;                           that are used as reserved colors. CINDEX ideally
;                           has 5 elements, P3D_RSS uses 5 reserved colors. If
;                           the number of elements is smaller, then the upper
;                           range of reserved colors will use the largest
;                           index available (also see CINDV).
;         cindv           - An array of 5 integers specifying which of the
;                           indices in CINDEX are used as the reserved colors:
;                           If CINDEX has 5 elements then:
;                              cindv=[CINDEX[0],CINDEX[1],CINDEX[2],
;                                     CINDEX[3],CINDEX[4]]
;                           If CINDEX has 3 elements then:
;                              cindv=[CINDEX[0],CINDEX[1],CINDEX[2],
;                                     CINDEX[2],CINDEX[2]]
;         bottom          - A scalar integer that specifies the bottom
;                           colormap index to use with the data. Note,
;                           0<=BOTTOM<=!D.TABLE_SIZE-1.
;         residualcut [-1]- A scalar decimal value. During the polynomial
;                           fitting, to get the dispersion solution,
;                           residuals>RESIDUALCUT will not be used in a second
;                           fitting.
;         fwhm            - A scalar decimal value with the instrumental
;                           line width in the dispersion dimension.
;         method          - A scalar string with the name of the method that is
;                           used when centering lines: ['Gaussian'] and
;                           'Weighted' are implemented,
;                           cf. p3d_wavecal_correct_maskpos.
;         postable        - A scalar string that contains the name of a
;                           position table file. This file is, if present, used
;                           to remove calibration elements in the data before
;                           the tool is setup.
;         deadfibers      - An array of integers specifying dead fibers, which
;                           should not be considered. The values must be
;                           1-based, and the maximum number must be smaller
;                           than the maximum number of spectra in the input
;                           image file.
;         sdeadfibers     - A string array with as many elements as DEADFIBERS
;                           specifying the type of fiber in DEADFIBER (dead,
;                           low transmission, unused,...).
;         removecalfibers [1] - n/a.
;                           Calibration fibers are removed from the data before
;                           handling the data if this keyword is set.
;         userparfile     - A scalar string specifying the name of an optional
;                           user parameter file, that could contain the keyword
;                           'polynomialorder'. If it does then the value of
;                           that keyword is used instead of XDEG.
;                           If there are several 'polynomialorder'-lines in the
;                           file then only the first is used.
;         imageorder [0]  - A scalar integer that is used if the number of
;                           files specified in IMAGEFILE is equal to two. For
;                           the use, see the description of the input parameter
;                           IMAGEFILE.
;         imagesplitbin [0.5*# pixels in dispersion direction of IMAGEFILE[0L]]
;                         - A scalar integer that is used if the number of
;                           files specified in IMAGEFILE is equal to two. For
;                           the use, see the description of the input parameter
;                           IMAGEFILE.
;         dflip [0]       - Spectrum order; if dflip is set then it is assumed
;                           that the wavelength range has been flipped.
;         compress [0]    - If this keyword is set then the output data file is
;                           compressed (using gzip).
;         dbin [1]        - This parameter determines if the input data is re-
;                           binned on the dispersion axis (DBIN=2||3), or if
;                           the data is kept as is (DBIN=1). A good reason to
;                           rebin the data is if all pixels should fit on the
;                           screen. The different values allowed are:
;                             DBIN=1: The data is not rebinned.
;                             DBIN=2: The data is rebinned by a factor 2.
;                             DBIN=3: The data is rebinned by a factor 2 until
;                                     all pixels fit on the screen.
;         track           - A keyword specifying if hints are to be shown about
;                           what is being done, and what the function of the
;                           various widgets is.
;         daxis [1]       - Defines the dispersion direction dimension
;                           of the data of IMAGEFILE. The default, 1, is in
;                           the x-direction.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         17.06.2009 - Updated widget layout ('tabs'), added check best-fit
;                      option, added  RMS diagnostics and updated
;                      PURPOSE description, /JAG
;         21.10.2008 - Converted from the original routine p3d_dispmask_gui of
;                      Thomas Becker. /CS
;-
PRO p3d_wavecal_dispmask_gui_calc_scalegeometry,igeo,ddsp,rcol,rdsp,rrow, $
        orcol,nrcol,ogeo
  compile_opt hidden,IDL2

  ;;========================================------------------------------
  ;; Rescaling the cross-dispersion line curvature array:

  p3d_wavecal_dispmask_gui_calc_lineardisp,rcol,rdsp,0L,ddsp,orcol,ordsp
  p3d_wavecal_dispmask_gui_calc_lineardisp,rcol,rdsp,0L,ddsp,nrcol,nrdsp
  ogeo=igeo*nrdsp/ordsp & ogeo-=ogeo[rrow]

  return
END ;;; procedure: p3d_wavecal_dispmask_gui_calc_scalegeometry


PRO p3d_wavecal_dispmask_gui_calc_linepositions,state
  compile_opt hidden,IDL2

  ;;========================================------------------------------
  ;; Calculating positions for all spectra of each entry in the line mask:

  ptr_free,(*state).linepix
  (*state).linepix=ptr_new(dblarr((*state).ncalwl))

  ptr_free,(*state).lines
  (*state).lines=ptr_new(dblarr((*state).ncalwl,(*state).si[2L])-1d0)
  (*state).slines=size(*(*state).lines)

  ;;===============================--------------------
  ;; Finding the "minimum wavelength" in the input array [[if a large part of
  ;; the lower wavelength range was fitted without any present lines then the
  ;; fit might actually be non-monotonic => this check is necessary]]:

  tmp=min((*state).lambda,mdx)
  marr=lindgen(n_elements((*state).lambda)-mdx)+mdx

  ;;===============================--------------------
  ;; Looping over all lines in the line mask:

  for k=0L,(*state).ncalwl-1L do begin
    pos=where((*state).lambda[marr] ge (*(*state).calwl)[k],count)

    if count ge 1L then (*(*state).linepix)[k]=pos[0L]+mdx

    if pos[0L]+mdx ge 0L then (*(*state).lines)[k,*]=pos[0L]+mdx + $
       (*state).geo*(*state).disp[pos[0L]+mdx]/ $
       (*state).disp[(*state).refcol]+(*state).ceo*(pos[0L]+mdx-(*state).crf)
  endfor ;; k=0L,(*state).ncalwl-1L

  return
END ;;; procedure: p3d_wavecal_dispmask_gui_calc_linepositions


PRO p3d_wavecal_dispmask_gui_calc_linearlambda,state
  compile_opt hidden,IDL2
  
  ;;========================================------------------------------
  ;; Calculating -linear- dispersion and wavelength arrays using the following
  ;; scalars defining the reference spectrum and reference wavelength bin:
  ;;   refdisp :: Dispersion
  ;;   refcol  :: Wavelength bin number
  ;;   refwave :: Wavelength
  ;;   ddisp   :: Change of dispersion between bins

  rd=(*state).refdisp
  tmp=lindgen((*state).npix)-(*state).refcol
  (*state).disp=rd+(*state).ddisp*tmp
  (*state).lambda=(*state).refwave+tmp*rd+tmp*(abs(tmp)+1L)/2L*(*state).ddisp

  return
END ;;; procedure: p3d_wavecal_dispmask_gui_calc_linearlambda


PRO p3d_wavecal_dispmask_gui_calc_lineardisp, $
        refcol,refdisp,refwave,ddisp,pix,pixdisp,pixwave
  compile_opt hidden,IDL2

  ;;========================================------------------------------
  ;; Calculating -linear- values of the dispersion and the wavelength at a
  ;; specified pixel using the following scalars:
  ;;  refdisp :: Dispersion
  ;;  refcol  :: Wavelength bin number
  ;;  refwave :: Wavelength
  ;;  ddisp   :: Change of dispersion between bins

  n=pix-refcol & an=abs(n)
  if arg_present(pixdisp) then pixdisp=refdisp+an*ddisp
  if arg_present(pixwave) then pixwave=refwave+n*refdisp+(n*(an+1L))/2L*ddisp

  return
END ;;; procedure: p3d_wavecal_dispmask_gui_calc_lineardisp


FUNCTION p3d_wavecal_dispmask_gui_imcombine,filenames,isplitbin, $
             imageorder,daxis
  compile_opt hidden,IDL2

  im1=readfits(filenames[0L],hdr1,/no_unsigned,/silent) & si=size(im1)
  im2=readfits(filenames[1L],hdr2,/no_unsigned,/silent)

  out=im1
  if ~imageorder then begin
    if daxis then begin
      out[       0L:isplitbin-1L,*]=im1[       0L:isplitbin-1L,*]
      out[isplitbin:si[daxis]-1L,*]=im2[isplitbin:si[daxis]-1L,*]
    endif else begin
      out[*,       0L:isplitbin-1L]=im1[*,       0L:isplitbin-1L]
      out[*,isplitbin:si[daxis]-1L]=im2[*,isplitbin:si[daxis]-1L]
    endelse
  endif else begin
    if daxis then begin
      out[       0L:isplitbin-1L,*]=im2[       0L:isplitbin-1L,*]
      out[isplitbin:si[daxis]-1L,*]=im1[isplitbin:si[daxis]-1L,*]
    endif else begin
      out[*,       0L:isplitbin-1L]=im2[*,       0L:isplitbin-1L]
      out[*,isplitbin:si[daxis]-1L]=im1[*,isplitbin:si[daxis]-1L]
    endelse
  endelse ;; ~imageorder

  return,daxis eq 1L?out:transpose(out)
END ;;; function: p3d_wavecal_dispmask_gui_imcombine


PRO p3d_wavecal_dispmask_gui_plot,state,xtra_x,xtra_y,image=image,rpos=rpos, $
        imxtra=imxtra,wavrange=wavrange,wall=wall,spectrum=spectrum,nox=nox, $
        plotfit=plotfit,tplotfit=tplotfit,final=final, $
        residuals=residuals,tresiduals=tresiduals,erase=erase,error=error
  compile_opt hidden,IDL2

  error=0
  relbin=1d0/(*state).tvbin

  ;;========================================------------------------------
  ;; Changing the decomposed state, if necessary:

  device,get_decomposed=decomposed
  if decomposed then device,decomposed=0L

  ;;========================================------------------------------
  ;; Setting required plotting parameters:

  tmppm=!p.multi
  !p.multi=0L ;; Only one diagram per figure

  xof=(*state).xoffset

  ;;========================================------------------------------
  ;; Saving the current colors, loading the used ones:

  tvlct,red,green,blue,/get
  tvlct,(*state).clr.red,(*state).clr.green,(*state).clr.blue

  ;;========================================------------------------------
  ;; This routine does all the drawing in the three draw widgets:

  if keyword_set(image) then begin

    ;; Redrawing the spectrum image:
    wset,(*state).wid.ima

    ;; clear image area
    if keyword_set(erase) then erase

    p3d_display_tele,*(*state).tvimage,(*state).colmin,(*state).colmax, $
        bottom=(*state).cbottom,topwid=(*state).top,logunit=(*state).logunit, $
        verbose=(*state).verbose,error=error
    if error ne 0 then return

    ;; Drawing a vertical line (which is a curve before the mask has been
    ;; created) for every calibration line:
    if (*state).slines[0L] ge 2L then begin
      for k=0L,(*state).slines[1L]-1L do begin
        for L=0L,(*state).slines[2L]-1L do $
           plots,[(*(*state).lines)[k,L]*relbin, $
                  (*(*state).lines)[k,L]*relbin],[L,L], $
                 /device,color=(*state).cindex[(*state).cindv[1L]]
      endfor
    endif ;; (*state).slines[0L] ge 2L

    plotsym,0,1.5,/fill

    ;; Should an extra reference line be drawn?
    if keyword_set(imxtra) then begin
      color=(*state).cindex[(*state).cindv[4L]]
      plots,[xtra_x*relbin],xtra_y,psym=8,color=color,/device
      plots,[xtra_x*relbin,xtra_x*relbin], $
            [0L,(*state).stvi[2L]-1L],/device,color=color
    endif

    ;; Plotting the reference line:
    if n_elements(rpos) eq 2L then begin
      x=[rpos[0L]]
      y=[rpos[1L]]
    endif else begin
      x=[(*state).refcol]
      y=[(*state).refrow]
    endelse
    x*=relbin

    color=(*state).cindex[(*state).cindv[2L]]
    plots,x,y,psym=8,color=color,/device
    plots,[x,x],[0L,(*state).stvi[2L]-1L],/device,color=color

  endif ;; keyword_set(image)


  ;;========================================------------------------------
  ;; Printing the wavelength range; on the left hand side, in the center, and
  ;; on the right hand side:

  if keyword_set(wavrange) then begin
    wset,(*state).wid.wav

    wall=keyword_set(wall)

    erase
    color =(*state).cindex[(*state).cindv[1L]]
    color2=(*state).cindex[(*state).cindv[3L]]
    offs=70L

    lamv=strtrim(round([(*state).lambda[0L], $
                        (*state).lambda[(*state).npix/2L], $
                        (*state).lambda[(*state).npix-1L]]),2L)

    ;; Plotting the lower marker:
    if xof eq 0L or wall then begin
      xyouts,0L,2L,lamv[0L],/device,alignment=0.0,color=color
    endif else begin
      if (*state).tvpix/2L ge xof+offs then begin
        xarr=[0,10,10,25,25,10,10]
        yarr=[7,0,4,4,10,10,15]
        polyfill,xarr,yarr,/device,color=color
        xyouts,30L,2L,lamv[0L],/device,alignment=0.0,color=color
      endif
    endelse ;; xof eq 0L

    ;; Plotting the middle marker:
    if wall or ((*state).tvpix/2L le xof-30L+(*state).winxsize and $
                (*state).tvpix/2L ge xof+30L) then begin
      xyouts,(*state).tvpix/2L-xof,2L,lamv[1L], $
             /device,alignment=0.5,color=color2
    endif else begin
      if (*state).tvpix/2L le xof+offs then begin
        xarr=[0,10,10,25,25,10,10]
        xpos=30L
        alignment=0.0
      endif else begin
        xarr=(*state).winxsize-[0,10,10,25,25,10,10]-5L
        xpos=(*state).winxsize-30L
        alignment=1.0
      endelse
      yarr=[7,0,4,4,10,10,15]
      polyfill,xarr,yarr,/device,color=color2
      xyouts,xpos,2L,lamv[1L],/device,alignment=alignment,color=color2
    endelse

    ;; Plotting the upper marker:
    if wall or (*state).winxsize+xof ge (*state).tvpix then begin
      xyouts,(*state).winxsize,2L,lamv[2L],/device,alignment=1.0,color=color
    endif else begin
      if (*state).tvpix/2L le xof-offs+(*state).winxsize then begin
        xarr=(*state).winxsize-[0,10,10,25,25,10,10]-3L
        yarr=[7,0,4,4,10,10,15]
        polyfill,xarr,yarr,/device,color=color
        xyouts,(*state).winxsize-38L,2L,lamv[2L], $
               /device,alignment=1.0,color=color
      endif
    endelse

  endif ;; keyword_set(wavrange)


  ;;========================================------------------------------
  ;; Plotting the current spectrum:

  if keyword_set(spectrum) then begin
    wset,(*state).wid.plo

    color=(*state).cindex[(*state).cindv[1L]]

    ;; Extracting the spectrum to plot:
    pspec=(*(*state).image)[*,(*state).refrow]

    ;; Checking plots always use the full wavelength range for the spectrum:
    xrange=keyword_set(tplotfit) or keyword_set( plotfit) or $
           keyword_set(residuals)?[0L,(*state).tvpix/relbin]:(*state).xrange

    plot,(*state).plotlambda,pspec,/xstyle,/ystyle,/device,color=color, $
         yrange=[(*state).colmin,(*state).colmax],xrange=xrange, $
         position=[0L,(*state).ypmin,(*state).winxsize,(*state).ypmax]

    ;; Plotting an 'x' for every calibration line:
    if ~keyword_set(nox) and max(*(*state).plotlines) gt 0.0 then begin
      tmp=dblarr((*state).slines[1L])+((*state).colmin+(*state).colmax)/2L
      oplot,*(*state).plotlines,tmp,psym=7,color=color
    endif

  endif ;; keyword_set(spectrum)


  ;;========================================------------------------------
  ;; Plot the reference wavelength positions as crosses:

  if keyword_set(plotfit) then begin
    wset,(*state).wid.ima
    color=(*state).cindex[(*state).cindv[1L]]

    okline=where(*(*state).linepix gt 0L,count)
    if count ne 0L then begin
      inline=okline
      inwave=(*(*state).calwl)[okline]
      inpix =(*(*state).linepix )[okline]

      xlo=min((*state).plotlambda,max=xhi)
      ylo=min(inwave)*0.97
      yhi=max(inwave)*1.03

      plot,inpix,inwave,/xstyle,/ystyle,/device,symsize=1.5, $
           color=color,psym=7,xrange=[xlo,xhi],yrange=[ylo,yhi], $
           position=[xof,(*state).ypmin, $
                     xof+(*state).winxsize,(*state).winysize-5L]
    endif ;; count ne 0L
  endif ;; keyword_set(plotfit)

  ;;========================================------------------------------
  ;; Plot the reference wavelength positions as crosses (total plot):

  if keyword_set(tplotfit) then begin
    wset,(*state).wid.ima
    color=(*state).cindex[(*state).cindv[1L]]

    okline=where(*(*state).linepix gt 0L,count)
    if count ne 0L then begin
      inline=okline
      inwave=(*(*state).calwl)[okline]
      inpix =(*(*state).cimlinep)[okline,(*state).slval]

      xlo=0L & xhi=(*state).si[1L]
      ylo=min(inwave,max=yhi)

      plot,inpix,inwave,/xstyle,/ystyle,/device,symsize=1.5, $
           color=color,psym=7,xrange=[xlo,xhi],yrange=[ylo,yhi], $
           position=[xof+70L,(*state).ypmin, $
                     xof+(*state).winxsize-15L,(*state).winysize-5L]
    endif ;; count ne 0L
  endif ;; keyword_set(tplotfit)


  ;;========================================------------------------------
  ;; Plot the residuals between points of the fitted polynomial and the
  ;; reference wavelengths:

  if keyword_set(residuals) then begin
     wset,(*state).wid.ima
     color=(*state).cindex[(*state).cindv[1L]]

     inwave=*(*state).calwl

     xvec=residuals.centroid
     yvec=residuals.wave-residuals.fitwave

     xlo=min((*state).plotlambda,max=xhi)
     ylo=min(yvec)*1.2
     yhi=max(yvec)*1.2

     plot,xvec,yvec,/xstyle,ystyle=5,/device,symsize=1.5, $
       color=color,psym=7,xrange=[xlo,xhi],yrange=[ylo,yhi], $
       position=[xof,(*state).ypmin,xof+(*state).winxsize,(*state).winysize-5L]

     oplot,[xlo,xhi],[0.,0.],linestyle=1

     ;; Plotting the y-axis:
     x0=0.02*(xhi-xlo)
     axis, x0,0,yaxis=1L,/ystyle,yrange=[ylo,yhi]
     axis,xhi,0,yaxis=1L,/ystyle,yrange=[ylo,yhi],charsize=0.01
  endif ;; keyword_set(residuals)

  ;;========================================------------------------------
  ;; Plot the residuals between points of the fitted polynomial and the
  ;; reference wavelengths:

  if keyword_set(tresiduals) then begin
     wset,(*state).wid.wav
     erase

     wset,(*state).wid.plo
     color=(*state).cindex[(*state).cindv[1L]]

     idx=where(*(*state).linepix gt 0L,count)
     inwave=(*(*state).calwl)[idx]
     inpix=reform((*(*state).cimlinep)[idx,(*state).slval])
     yfit=dblarr(n_elements(inwave))
     for L=0L,(*state).lxdeg do $
        yfit+=(*(*state).fitpar)[(*state).slval,L]*inpix^L

     xvec=inpix
     yvec=inwave-yfit

     xlo=0L & xhi=(*state).si[1L]
     ylo=min(yvec,max=yhi)
     ylo=ylo<0.0 & yhi=yhi>0.0
     if yhi eq ylo then yhi=1.0 ;; for safety

     plot,xvec,yvec,/xstyle,/ystyle,/ynozero,/device,symsize=1.5, $
          color=color,psym=7,xrange=[xlo,xhi],yrange=[ylo,yhi], $
          position=[70L,(*state).ypmin,(*state).winxsize-15L,(*state).ypmax]

     oplot,[xlo,xhi],[0.,0.],linestyle=1

     infostring='['+strtrim((*state).slval+1L,2L)+'] max(|residual|)='+ $
                string(max(abs(yvec)),format='(e9.2)')
     xpos=(!x.crange[1L]-!x.crange[0L])*0.05+!x.crange[0L]
     ypos=(!y.crange[1L]-!y.crange[0L])*0.85+!y.crange[0L]
     xyouts,xpos,ypos,infostring,charsize=1.5,color=color

  endif ;; keyword_set(tresiduals)


  ;;========================================------------------------------
  ;; Restoring the colors:

  tvlct,red,green,blue

  ;;========================================------------------------------
  ;; Resetting various plotting variables:

  !p.multi=tmppm

  ;;========================================------------------------------
  ;; Returning to the entry decomposed state, if necessary:

  if decomposed then device,decomposed=decomposed

  ;;========================================------------------------------
  ;; Re-sensitizing the colortable menu button if xloadct is closed:

  if ~xregistered('xloadct') then $
     widget_control,(*state).wid.bcolo,sensitive=1L

  return
END ;;; procedure: p3d_wavecal_dispmask_gui_plot


PRO p3d_wavecal_dispmask_gui_fitdisp,state,residuals=residuals,error=error
  compile_opt hidden,IDL2

  ;; Fitting the positions and wavelength in the reference spectrum with a
  ;; poylnomial:

  idx=where(*(*state).linepix gt 0L,count) ;; number of 'valid' lines
  if ~count then begin
    infostr='[fitdisp] There are no valid lines to plot.'
    if (*state).track then $
      widget_control,(*state).wid.lstat,set_value=infostr
  endif else begin

    ;; Saving the current colors, loading the used ones:
    tvlct,red,green,blue,/get
    tvlct,(*state).clr.red,(*state).clr.green,(*state).clr.blue

    p3d_wavecal_dispmask_gui_plot,state,/spectrum,/plotfit, $
        /wall,/wavrange,error=error
    if error ne 0 then return

    inwave=(*(*state).calwl)[idx]    ;; array with line wavelengths
    inpix =(*(*state).linepix )[idx] ;; array with line centroids
    sidx=sort(inpix)
    degree=(*state).lxdeg            ;; polynomial degree

    bestpar=poly_fit(inpix,inwave,degree,chisq=chisq,status=status,yfit=yfit)
    if ~status then begin
      yfit=dblarr(n_elements(inwave))
      for i=0L,degree do yfit+=bestpar[i]*inpix^i

      color=(*state).cindex[(*state).cindv[1L]]
      oplot,inpix[sidx],yfit[sidx],color=color
      rms=sqrt(total((yfit-inwave)^2)/n_elements(yfit))

      infostring='Polynomial order: '+strtrim(degree,2L) + $
        ' :: poly_fit status: '+string(status,format='(i1)') + $
        ' :: !4v!x!u2!n:'+string(chisq,format='(e10.3)') + $
        ' :: RMS:'+ string(rms, format='(e10.3)')
      xpos=((*state).xoffset+0.05*(*state).winxsize)/(*state).tvpix
      xyouts,xpos,0.85,/normal,infostring,charsize=1.5,color=color
    endif else begin
      msg='[fitdisp] POLY_FIT: status='+strtrim(status,2L)+' '
      case status of
        1: msg+='REFPIX is singluar.'
        2: msg+='A small pivot element was used, accuracy is likely lost.'
        3: msg+='Undefined error estimate (NaN) was encountered.'
      endcase
      if (*state).track then $
        widget_control,(*state).wid.lstat,set_value=msg
      return
    endelse ; ~status

    ;;========================================------------------------------
    ;; Restoring the colors:

    tvlct,red,green,blue

    ;;========================================------------------------------
    ;; Saving the residuals:

    residuals={centroid:inpix,wave:inwave,fitwave:yfit}

  endelse ;; ~count

  return
END ;;; procedure: p3d_wavecal_dispmask_gui_fitdisp


PRO p3d_wavecal_dispmask_gui_plotfits,state,error=error
  compile_opt hidden,IDL2

  ;; This routine shows fitted lines and residuals for all spectra:

  idx=where(*(*state).linepix gt 0L,count) ;; number of 'valid' lines
  if ~count then begin
    infostr='[plotfits] There are no valid lines to plot.'
    if (*state).track then $
      widget_control,(*state).wid.lstat,set_value=infostr
  endif else begin

    ;; Saving the current colors, loading the used ones:
    tvlct,red,green,blue,/get
    tvlct,(*state).clr.red,(*state).clr.green,(*state).clr.blue

    p3d_wavecal_dispmask_gui_plot,state,/tplotfit,/wavrange,/wall,error=error
    if error ne 0 then return

    ;; Retrieving the current line fit:
    x=dindgen((*state).si[1L])
    spec=dblarr((*state).si[1L])
    for L=0L,(*state).lxdeg do spec+=(*(*state).fitpar)[(*state).slval,L]*x^L

    color=(*state).cindex[(*state).cindv[1L]]
    oplot,x,spec,color=color

    tmpstr=''
    if (*state).usedeadfibers then begin
      tmp=where((*state).deadfibers eq (*state).slval+1L,count)
      if count eq 1L then tmpstr=' {'+(*state).sdeadfibers[tmp]+'}'
    endif ;; (*state).usedeadfibers

    infostring='['+strtrim((*state).slval+1L,2L)+'] Pol. order: '+ $
        strtrim((*state).lxdeg,2L)+' :: !4v!x!u2!n:'+ $
        string((*(*state).chisq)[(*state).slval],format='(e10.3)')+tmpstr
    xpos=(!x.crange[1L]-!x.crange[0L])*0.05+!x.crange[0L]
    ypos=(!y.crange[1L]-!y.crange[0L])*0.85+!y.crange[0L]
    xyouts,xpos,ypos,infostring,charsize=1.5,color=color

    ;; Residuals:
    p3d_wavecal_dispmask_gui_plot,state,/tresiduals,error=error
    if error ne 0 then return

    ;;========================================------------------------------
    ;; Restoring the colors:

    tvlct,red,green,blue

  endelse ;; ~count

  return
END ;;; procedure: p3d_wavecal_dispmask_gui_plotfits


PRO p3d_wavecal_dispmask_gui_pairplot,state,xarr,oarr,xi
  compile_opt hidden,IDL2

  loop=1L & i=0L
  if n_elements(xi) ne 0L then begin & i=xi & loop=0L & endif

  relbin=1d0/(*state).tvbin
  colorred=(*state).cindex[(*state).cindv[2L]]
  colorgrn=(*state).cindex[(*state).cindv[3L]]

  ;; Y positions:
  ypo1=(*state).ypmin+0.67*((*state).ypmax-(*state).ypmin)
  ypo2=(*state).ypmin+0.73*((*state).ypmax-(*state).ypmin)
  ypos=(*state).ypmin+0.70*((*state).ypmax-(*state).ypmin)
  ypoa=[ypo1,ypo2,ypos]
  ypob=[ypo1,ypo2]

  xof=(*state).xoffset

  repeat begin
    index=strtrim(i+1L,2L)
    fac=(*state).tvpix/relbin*0.01
    xp=xarr[i]+fac*(oarr[i] gt xarr[i]?1L:-1L)
    polyfill,([xarr[i],xarr[i],xp]-xof)*relbin,ypoa,/device,color=colorred
    plots,([xarr[i],oarr[i]]-xof)*relbin,[ypos,ypos],/device,color=colorred
    plots,([oarr[i],oarr[i]]-xof)*relbin,ypob,/device,color=colorgrn
    xyouts,(oarr[i]-xof)*relbin,ypo2,index,/device,color=colorgrn
    if ++i eq n_elements(xarr) and loop then loop=0L
  endrep until ~loop

  return
END ;;; procedure: p3d_wavecal_dispmask_gui_pairplot


PRO p3d_wavecal_dispmask_gui_xloadct,data=state
  compile_opt hidden,IDL2

  ;; Loading the colors to use:
  tvlct,red,green,blue,/get
  (*state).clr={red:red,green:green,blue:blue}

  ctab=widget_info((*state).wid.bconp,/tab_current)
  if ctab ne 1L and ctab ne 2L then $
    p3d_wavecal_dispmask_gui_plot,state,/image,/wavrange,/spectrum

  return
END ;;; procedure: p3d_wavecal_dispmask_gui_xloadct


PRO p3d_wavecal_dispmask_gui_view_linelist,state,text
  compile_opt hidden,IDL2

  tmp1=*(*state).calwl & tmp2=*(*state).linepix
  idx=sort(tmp1) & tmp1=tmp1[idx] & tmp2=tmp2[idx]
  text=strarr(n_elements(tmp1)+2L)
  text[0L]='  n wavelength [pixelpos]'
  text[1L]='-------------------------'
  for i=0L,n_elements(tmp1)-1L do $
     text[i+2L]=string(i+1L,tmp1[i],tmp2[i],format='(i3,1x,f10.4,2x,f9.3)')

  return
END ;;; procedure: p3d_wavecal_dispmask_gui_view_linelist


PRO p3d_wavecal_dispmask_gui_event,event
  compile_opt hidden,IDL2

  rname='p3d_wavecal_dispmask_gui_event: '

  ;;========================================------------------------------
  ;; Getting the state structure:

  widget_control,event.top,get_uvalue=state

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=event.top
      catch,/cancel
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Handling tracking events first:

  tmpstrname=strlowcase(tag_names(event,/structure_name))
  if tmpstrname eq 'widget_tracking' then begin
    if event.enter then begin  
      infostr=widget_info(event.id,/uname)
      widget_control,(*state).wid.lstat,set_value=infostr
    endif else widget_control,(*state).wid.lstat,set_value=''

    ;; Nothing more to do, returning:
    return
  endif

  if tmpstrname eq 'widget_kbrd_focus' then if event.enter then return

  ;;========================================------------------------------
  ;; Getting the user value of the event id:

  widget_control,event.id,get_uvalue=uval

  ;; Extracting a widget "index" for those widgets which have defined one:
  index=-1L
  if n_elements(uval) ne 0L  then begin
    tmppos=strpos(uval,'_',/reverse_search)
    if tmppos ne -1L then begin
      index=float(strmid(uval,tmppos+1L))
      uval=strmid(uval,0L,tmppos)
    endif
  endif

  ;;========================================------------------------------
  ;; Changing the decomposed state, if necessary:

  device,get_decomposed=decomposed
  if decomposed then device,decomposed=0L

  relbin=1d0/(*state).tvbin

  ;;========================================------------------------------
  ;; Viewport events:

  widget_control,(*state).wid.dima,get_draw_view=dview
  if (keyword_set(tplotfit) or keyword_set(plotfit)) and $
     dview[0L] ne (*state).xoffset then begin
    widget_control,(*state).wid.dima, $
                   set_draw_view=[(*state).xoffset,dview[1L]]
  endif

  if strlowcase(tag_names(event,/structure_name)) eq 'widget_draw' then begin
    if event.type eq 3L then begin
      widget_control,(*state).wid.dima,get_draw_view=dview
      (*state).xoffset=dview[0L]
      if (*state).maskdone then begin
        (*state).xrange=(*state).plotlambda[ $
                        ([0d0,(*state).winxsize]+(*state).xoffset)/relbin]
      endif else begin
        (*state).xrange=([0d0,(*state).winxsize]+(*state).xoffset)/relbin
      endelse
      p3d_wavecal_dispmask_gui_plot,state,/image,/wavrange,/spectrum, $
          error=error
    endif ;; event.type eq 3L
  endif ;; strlowcase(tag_names(event,/structure_name)) eq 'widget_draw'

  ;;========================================------------------------------
  ;; Remaining events:

  case strlowcase(uval) of

    ;;===============================--------------------
    ;; Delete a line from the list of calibration lines:

    'maskremoveline': begin

      ;; Un-sensitizing the control panel:
      widget_control,(*state).wid.bmact,sensitive=0L
      widget_control,(*state).wid.bconp,sensitive=0L

      (*state).draww.mode=(*state).draww.maskremoveline

      ;; Updating the status widget:
      if (*state).track then widget_control,(*state).wid.lstat, $
          set_value='[Delete line(s) in the mask] Click the left button in' + $
          ' the vicinity of (red) lines of the line list; finish with the ' + $
          'right mouse button'
    end ;; case: 'maskremoveline'

    ;;===============================--------------------
    ;; Define an initial dispersion solution manually:

    'dispersionmanualset': begin

      ;; Un-sensitizing the control panel:
      widget_control,(*state).wid.bmact,sensitive=0L
      widget_control,(*state).wid.bconp,sensitive=0L

      (*state).draww.mode=(*state).draww.dispmanual

      ;; Updating the status widget:
      if (*state).track then widget_control,(*state).wid.lstat, $
          set_value='[Initial dispersion sol.] {define set} :: Left: set p' + $
          'ixel position of current line.'
    end ;; case: 'dispersionmanualset'


    ;;===============================--------------------
    ;; Spectrum plot window:

    'spectrumplot': begin

      case (*state).draww.mode of

        ;;===============================--------------------
        ;;===============================--------------------
        ;;===============================--------------------
        ;; Removing a line in the line mask:

        (*state).draww.maskremoveline: begin
          wset,(*state).wid.plo

          if event.press eq 4b then begin
            ;; Not accepting more events of this kind:
            (*state).draww.mode=0L

            ;; Pressed right button, stopping deletion of lines in the mask:

            if (*state).draww.tempi[0L] ne -1L then begin
              ;; If the line, which was marked for deletion, was never deleted,
              ;; then the position of this line is redrawn:

              tmpz= (*state).ypmin*0.37+(*state).ypmax*0.63

              tmp=(*state).draww.tempi[0L]
              tmpx=(*(*state).plotlines)[tmp]
              tmpstr=strtrim((*(*state).calwl)[tmp],2L)
              oplot,[tmpx],[(*state).colmin+(*state).colmax]/2L, $
                  psym=7,color=(*state).cindex[(*state).cindv[1L]]
              xyouts,(tmpx-(*state).xoffset)*relbin,tmpz,tmpstr,/device, $
                  alignment=0.5,color=(*state).cindex[(*state).cindv[0L]]

              ;; Redrawing the spectrum plot:
              p3d_wavecal_dispmask_gui_plot,state,/spectrum,error=error
            endif ;; (*state).draww.tempi[0L] ne -1L

            ;;===============================--------------------
            ;; Restoring the colors:

            tvlct,(*state).rclr.red,(*state).rclr.green,(*state).rclr.blue

            ;;===============================--------------------
            ;; Re-sensitizing the control panel:

            widget_control,(*state).wid.bsavm,sensitive=0L ;; save & exit
            widget_control,(*state).wid.bsave,sensitive=0L ;; -"-
            widget_control,(*state).wid.brmas,sensitive=0L ;; reset
            widget_control,(*state).wid.bmact,sensitive=1L
            widget_control,(*state).wid.bconp,sensitive=1L

            (*state).draww.active  =0b

          endif else begin

            if ~(*state).draww.active then begin
              ;; Redrawing the spectrum plot:
              p3d_wavecal_dispmask_gui_plot,state,/spectrum,error=error
              if error ne 0 then return

              (*state).draww.tempi[0L]=-1L

              ;; Saving the current colors, loading the ones used by this tool:
              tvlct,red,green,blue,/get
              (*state).rclr.red  =red
              (*state).rclr.green=green
              (*state).rclr.blue =blue
              tvlct,(*state).clr.red,(*state).clr.green,(*state).clr.blue

              (*state).draww.active  =1b
            endif ;; ~(*state).draww.active

            if event.x ge 0L and event.x lt (*state).winxsize and $
               event.y ge 0L and max(*(*state).linepix) gt 0.0 then begin

              tmpy=[(*state).ypmin*0.45+(*state).ypmax*0.55, $
                    (*state).ypmin*0.40+(*state).ypmax*0.60]
              tmpz= (*state).ypmin*0.37+(*state).ypmax*0.63

              event.x+=(*state).xoffset

              y=event.y<((*state).slines[2L]-1L)
              tmp=min(abs((*(*state).lines)[*,y]-event.x/relbin),delline)

              if ~event.press and $
                delline ne (*state).draww.tempi[0L] then begin

                ;; Motion events, marking the line closest to the mouse
                ;; pointer (but only if it wasn't already marked):

                if (*state).draww.tempi[0L] ne -1L then begin
                  tmp=(*state).draww.tempi[0L]
                  tmpx=(*(*state).plotlines)[tmp]
                  tmpstr=strtrim((*(*state).calwl)[tmp],2L)
                  oplot,[tmpx],[(*state).colmin+(*state).colmax]/2L, $
                      psym=7,color=(*state).cindex[(*state).cindv[1L]]
                  xyouts,(tmpx-(*state).xoffset)*relbin,tmpz,tmpstr,/device, $
                      alignment=0.5,color=(*state).cindex[(*state).cindv[0L]]
                endif ;; (*state).draww.tempi[0L] ne -1L

                ;; Redrawing the spectrum plot:
                p3d_wavecal_dispmask_gui_plot,state,/spectrum,error=error

                tmpx=(*(*state).plotlines)[delline]
                tmpstr=strtrim((*(*state).calwl)[delline],2L)
                oplot,[tmpx],[(*state).colmin+(*state).colmax]/2L,psym=7, $
                      color=(*state).cindex[(*state).cindv[2L]]
                xyouts,(tmpx-(*state).xoffset)*relbin,tmpz,tmpstr,/device, $
                    alignment=0.5,color=(*state).cindex[(*state).cindv[2L]]

                (*state).draww.tempi[0L]=delline
              endif else if event.press ge 1b then begin ;; ~event.press

                ;; A button has been pressed, the line closest to the mouse
                ;; pointer will be removed:

                pos=lonarr((*state).slines[1L])+1L
                pos[delline]=0L

                wpos=where(pos,count)
                if ~count then begin
                  ;; No lines left:
                  if ptr_valid((*state).lines)     then ptr_free,(*state).lines
                  if ptr_valid((*state).plotlines) then $
                    ptr_free,(*state).plotlines
                  if ptr_valid((*state).linepix)   then $
                    ptr_free,(*state).linepix
                  if ptr_valid((*state).calwl)     then ptr_free,(*state).calwl

                  (*state).lines    =ptr_new([-1L]) ;; avoiding array resize
                  (*state).plotlines=ptr_new(-1L)
                  (*state).linepix  =ptr_new(-1L)
                  (*state).calwl    =ptr_new(-1L)
                  (*state).ncalwl   =0L
                endif else begin
                  ;; Re-allocating the 'used'-lines arrays:
                  tmplin=*(*state).lines
                  tmppix=*(*state).linepix
                  tmpcal=*(*state).calwl

                  if ptr_valid((*state).lines)     then $
                    ptr_free,(*state).lines
                  if ptr_valid((*state).plotlines) then $
                    ptr_free,(*state).plotlines
                  if ptr_valid((*state).linepix)   then $
                    ptr_free,(*state).linepix
                  if ptr_valid((*state).calwl)     then ptr_free,(*state).calwl

                  (*state).lines    =ptr_new(tmplin[wpos,*])
                  (*state).plotlines=ptr_new(tmplin[wpos,(*state).refrow])
                  (*state).linepix  =ptr_new(tmppix[wpos])
                  (*state).calwl    =ptr_new(tmpcal[wpos])
                  (*state).ncalwl   =n_elements(*(*state).calwl)
                endelse ;; ~count

                if widget_info((*state).lltid,/valid_id) then begin
                  p3d_wavecal_dispmask_gui_view_linelist,state,tmp
                  widget_control,(*state).lltid,set_value=tmp
                endif

                (*state).slines=size(*(*state).lines)

                ;; Redrawing all (changed) plots:
                p3d_wavecal_dispmask_gui_plot,state,/image,/spectrum, $
                    error=error
                if error ne 0 then return

                (*state).draww.tempi[0L]=-1L
              endif ;; ~event.press

            endif ;; event.x ge 0L and event.x lt (*state).winxsize and $
                  ;; event.y ge 0L and event.y lt (*state).stvi[2L] and $
                  ;; max(*(*state).linepix) gt 0.0
          endelse ;; event.press eq 4b
        end ;; case: (*state).draww.maskremoveline


        ;;===============================--------------------
        ;;===============================--------------------
        ;;===============================--------------------
        ;; Defining an initial dispersion solution manually:

        (*state).draww.dispmanual: begin
          if (*(*state).calwl)[0L] eq -1L then return
          wset,(*state).wid.plo

          if ~event.press and ~event.release and $
             ~(*state).monopressed then begin

            ;; Marking the line that lies closest to the current position:

            x=(((event.x<((*state).winxsize-1L))>0L)+(*state).xoffset)/relbin
            y=(*state).refrow
            tmp=min(abs((*(*state).lines)[*,y]-x),minps)
            tmpz= (*state).ypmin*0.37+(*state).ypmax*0.63

            ;; Motion events, marking the line closest to the mouse
            ;; pointer (but only if it wasn't already marked):

            if (*state).draww.tempi[0L] ne -1L then begin
              tmp=(*state).draww.tempi[0L]
              tmpx=(*(*state).plotlines)[tmp]
              tmpstr=strtrim((*(*state).calwl)[tmp],2L)
              oplot,[tmpx],[(*state).colmin+(*state).colmax]/2L, $
                    psym=7,color=(*state).cindex[(*state).cindv[1L]]
              xyouts,(tmpx-(*state).xoffset)*relbin,tmpz,tmpstr,/device, $
                   alignment=0.5,color=(*state).cindex[(*state).cindv[0L]]
            endif ;; (*state).draww.tempi[0L] ne -1L

            ;; Redrawing the spectrum plot:
            p3d_wavecal_dispmask_gui_plot,state,/spectrum,error=error

            tmpx=(*(*state).plotlines)[minps]
            tmpstr=strtrim((*(*state).calwl)[minps],2L)
            oplot,[tmpx],[(*state).colmin+(*state).colmax]/2L,psym=7, $
                color=(*state).cindex[(*state).cindv[2L]]
            xyouts,(tmpx-(*state).xoffset)*relbin,tmpz,tmpstr,/device, $
                alignment=0.5,color=(*state).cindex[(*state).cindv[1L]]

            ;; Drawing indicator lines for the current pair:
            if ptr_valid((*state).draww.storage) then begin
              s1=(*(*state).draww.storage).s1
              s2=(*(*state).draww.storage).s2
              if n_elements(s1) ge 1L then $
                 p3d_wavecal_dispmask_gui_pairplot,state,s1,s2
            endif ;; ptr_valid((*state).draww.storage)

            (*state).draww.tempi[0L]=minps

          endif else if event.press eq 4b then begin

            ;; Right mouse-button - not accepting more events of this kind:
            tmp=(*state).draww.pressed

            (*state).monopressed=0L
            (*state).draww.mode=0L
            (*state).draww.active =0b
            (*state).draww.pressed=0L
            (*state).showline=-1L

            if tmp lt 4L then begin
              if ptr_valid((*state).draww.storage) then $
                 ptr_free,(*state).draww.storage

              msg='[Initial dispersion sol.] Canceled.'
              if (*state).track then $
                 widget_control,(*state).wid.lstat,set_value=msg

              p3d_wavecal_dispmask_gui_plot,state,/spectrum,error=error
              if error ne 0 then return

              widget_control,(*state).wid.bmact,sensitive=1L
              widget_control,(*state).wid.bconp,sensitive=1L

              return
            endif ;; tmp lt 4L

            ;;===============================--------------------
            ;; Restoring the colors:

            tvlct,(*state).rclr.red,(*state).rclr.green,(*state).rclr.blue

            ;;===============================--------------------
            ;; Recalculating the dispersion solution:

            xt=(*(*state).draww.storage).s2
            yw=(*(*state).calwl)[(*(*state).draww.storage).ld]
            idx=sort(xt) & inx=xt[idx] & iny=yw[idx]

            if ptr_valid((*state).draww.storage) then $
                ptr_free,(*state).draww.storage

            lambda=spline(inx,iny,dindgen((*state).npix))
            disp=lambda[1L:(*state).npix-1L]-lambda[0L:(*state).npix-2L]
            (*state).disp=disp
            (*state).refdisp=  disp[(*state).refcol]
            (*state).lambda =lambda
            (*state).refwave=lambda[(*state).refcol]

            ;; Recalculating pixel positions of all lines:
            p3d_wavecal_dispmask_gui_calc_linepositions,state

            ptr_free,(*state).plotlines
            (*state).plotlines=ptr_new((*(*state).lines)[*,(*state).refrow])

            ;;===============================--------------------
            ;; Redrawing the spectrum image and plot:

            p3d_wavecal_dispmask_gui_plot,state, $
                /image,/wavrange,/spectrum,error=error
            if error ne 0 then return

            ;; Re-sensitizing the control panel:
            widget_control,(*state).wid.bsavm,sensitive=0L   ;; save & exit
            widget_control,(*state).wid.bsave,sensitive=0L   ;; -"-
            widget_control,(*state).wid.bshim,sensitive=0L   ;; shift mask
            widget_control,(*state).wid.bcdin,sensitive=1L   ;; reset disp.
            widget_control,(*state).wid.bcdir,sensitive=1L   ;; -"-
            widget_control,(*state).wid.brmas,sensitive=0L   ;; reset
            widget_control,(*state).wid.bmact,sensitive=1L
            widget_control,(*state).wid.bconp,sensitive=1L

            msg='The initial dispersion solution has been re-evaluated.'
            if (*state).track then $
                widget_control,(*state).wid.lstat,set_value=msg

          endif else if event.press eq 2b then begin

            ;;====================---------------
            ;; Clicked middle mouse button; removing the last entry:

            ;; Only removing a pair if a complete pair has been defined:
            if ptr_valid((*state).draww.storage) and $
               ~((*state).draww.pressed mod 2L) then begin

              n=n_elements((*(*state).draww.storage).s1)-1L
              if n eq 0L then begin
                ptr_free,(*state).draww.storage
                (*state).draww.pressed=0L
                (*state).monopressed=0L

                ;; Redrawing the spectrum plot:
                p3d_wavecal_dispmask_gui_plot,state,/spectrum,error=error

                msg='[Initial dispersion sol.] {define set} :: Left: set p' + $
                    'ixel position of current line.'
                if (*state).track then $
                   widget_control,(*state).wid.lstat,set_value=msg

              endif else if n ge 1L then begin
                ;; Removing the last entry in the list of pairs:
                s1=(*(*state).draww.storage).s1[0L:n-1L]
                s2=(*(*state).draww.storage).s2[0L:n-1L]
                ld=(*(*state).draww.storage).ld[0L:n-1L]
                ptr_free,(*state).draww.storage
                (*state).draww.storage=ptr_new({s1:s1,s2:s2,ld:ld})
                (*state).draww.pressed-=2L

                ;; Redrawing the spectrum plot:
                p3d_wavecal_dispmask_gui_plot,state,/spectrum,error=error

                ;; Drawing indicator lines for the current pair:
                if n ge 1L then p3d_wavecal_dispmask_gui_pairplot,state,s1,s2

                tmpstr=n ge 2L?'recalculate dispersion.':'cancel.'
                msg='[Initial dispersion sol.] {define set, pair '+ $
                    strtrim(n,2L)+' defined} :: Left: select another line ' + $
                    ':: Middle: delete last entry :: Right: '+tmpstr
                if (*state).track then $
                   widget_control,(*state).wid.lstat,set_value=msg

              endif ;; n ge 0L
            endif ;; ptr_valid((*state).draww.storage) and $
                  ;;                       ~((*state).draww.pressed mod 2L)

          endif else if event.press eq 1b then begin

            ;;====================---------------
            ;; Clicked left mouse button; selecting a line or a pixel position:

            ;; Determining if we're selecting a line, or a pixel position:
            mode=~((*state).draww.pressed mod 2L)?1L:2L
            (*state).monopressed=mode
            butmode=event.press
            if ~(*state).draww.active then begin
              (*state).draww.pressed=0L
              if ptr_valid((*state).draww.storage) then $
                  ptr_free,(*state).draww.storage

              ;; Saving the current colors, loading the ones used by this tool:
              tvlct,red,green,blue,/get
              (*state).rclr.red  =red
              (*state).rclr.green=green
              (*state).rclr.blue =blue
              tvlct,(*state).clr.red,(*state).clr.green,(*state).clr.blue
            endif ;; ~(*state).draww.active

            x=(((event.x<((*state).winxsize-1L))>0L)+(*state).xoffset)/relbin
            y=(*state).refrow

            if (*state).monopressed eq 1L then begin

              ;;====================---------------
              ;; Selecting a line in the line mask:

              (*state).draww.active=1b ;; index indicating a set is defined
              (*state).draww.pressed++

              tmp=min(abs((*(*state).lines)[*,y]-x),minps)
              (*state).showline=minps ;; index of currently selected line

              msg='[Initial dispersion sol.] {define set, pair '+ $
                  strtrim((*state).draww.pressed/2L+1L,2L)+'} :: line# ' + $
                  strtrim(minps,2L)+' :: Left: set pixel position of curre' + $
                  'nt line.'
              if (*state).track then $
                 widget_control,(*state).wid.lstat,set_value=msg

            endif else if (*state).monopressed eq 2L then begin

              ;;====================---------------
              ;; Selecting a pixel position where the selected line should be:

              originx=(*(*state).lines)[(*state).showline,y]
              targetx=(event.x+(*state).xoffset)/relbin

              ;; Storing the pixel positions of the current pair:
              (*state).draww.pressed++

              if ~ptr_valid((*state).draww.storage) then begin
                (*state).draww.storage=ptr_new({s1:[originx],s2:[targetx], $
                    ld:(*state).showline})
              endif else begin
                s1=(*(*state).draww.storage).s1 & s1=[s1,originx]
                s2=(*(*state).draww.storage).s2 & s2=[s2,targetx]
                ld=(*(*state).draww.storage).ld & ld=[ld,(*state).showline]
                ptr_free,(*state).draww.storage
                (*state).draww.storage=ptr_new({s1:s1,s2:s2,ld:ld})
              endelse

              ;; Drawing indicator lines for the current pair:
              i=((*state).draww.pressed-1L)/2L+1L
              p3d_wavecal_dispmask_gui_pairplot,state, $
                  (*(*state).draww.storage).s1, $
                  (*(*state).draww.storage).s2,i-1L

              tmpstr=i ge 2L?'recalculate dispersion.':'cancel.'
              msg='[Initial dispersion sol.] {define set, pair '+ $
                  strtrim(i,2L)+' defined} :: Left: select another line ::' + $
                  ' Middle: delete last entry :: Right: '+tmpstr
              if (*state).track then $
                 widget_control,(*state).wid.lstat,set_value=msg

              (*state).monopressed=0L

            endif ;; (*state).monopressed eq 1L

          endif ;; event.press eq 4b

        end ;; case :(*state).draww.dispmanual

        else: begin

          if (*(*state).calwl)[0L] eq -1L then return

          if event.press eq 1L or event.press eq 2L then begin
            (*state).monopressed=1L & butmode=event.press
          endif ;; event.press eq 1L or event.press eq 2L

          if event.press   eq 4L then (*state).monopressed=0L
          if event.release ne 0L and (*state).monopressed then $
            black=1L else black=0L
          if event.release ne 0L then (*state).monopressed=0L

          tmpy=[(*state).ypmin*0.45+(*state).ypmax*0.55, $
                (*state).ypmin*0.40+(*state).ypmax*0.60]
          tmpz= (*state).ypmin*0.37+(*state).ypmax*0.63

          ;; Saving the current colors, loading the ones used by this tool:
          tvlct,red,green,blue,/get
          (*state).rclr.red  =red
          (*state).rclr.green=green
          (*state).rclr.blue =blue
          tvlct,(*state).clr.red,(*state).clr.green,(*state).clr.blue

          if (*state).monopressed then begin
            x=event.x & x=(((x<((*state).winxsize-1L))>0L)+ $
                           (*state).xoffset)/relbin
            y=(*state).refrow

            oldshowline=(*state).showline

            tmp=min(abs((*(*state).lines)[*,y]-x),minps)
            (*state).showline=minps

            wset,(*state).wid.plo
            if oldshowline ne -1L and $
              oldshowline ne (*state).showline then begin
              tmpx=(*(*state).plotlines)[oldshowline]
              tmpstr=strtrim((*(*state).calwl)[oldshowline],2L)
              pspec=(*(*state).image)[*,(*state).refrow]

              color=(*state).cindex[(*state).cindv[0L]]
              plots,([tmpx,tmpx]-(*state).xoffset)*relbin,tmpy, $
                    /device,color=color
              xyouts,(tmpx-(*state).xoffset)*relbin,tmpz,tmpstr,/device, $
                  alignment=0.5,color=color

              color=(*state).cindex[(*state).cindv[1L]]
              plot,(*state).plotlambda,pspec, $
                   /xstyle,/ystyle,/noerase,/device,color=color, $
                   position=[0L,(*state).ypmin, $
                             (*state).winxsize,(*state).ypmax], $
                   yrange=[(*state).colmin,(*state).colmax], $
                   xrange=(*state).xrange
            endif ;; oldshowline ne -1L

            tmpx=(*(*state).plotlines)[(*state).showline]
            tmpstr=strtrim((*(*state).calwl)[(*state).showline],2L)
            color=(*state).cindex[(*state).cindv[1L]]
            plots,([tmpx,tmpx]-(*state).xoffset)*relbin,tmpy, $
                  /device,color=color
            xyouts,(tmpx-(*state).xoffset)*relbin,tmpz,tmpstr,/device, $
                   alignment=0.5,color=color

          endif else begin ;; (*state).monopressed

            if black then begin
              tmpx=(*(*state).plotlines)[(*state).showline]
              tmpstr=strtrim((*(*state).calwl)[(*state).showline],2L)
              pspec=(*(*state).image)[*,(*state).refrow]

              color=(*state).cindex[(*state).cindv[0L]]
              plots,([tmpx,tmpx]-(*state).xoffset)*relbin,tmpy, $
                    /device,color=color
              xyouts,(tmpx-(*state).xoffset)*relbin,tmpz,tmpstr,/device, $
                     alignment=0.5,color=color

              color=(*state).cindex[(*state).cindv[1L]]
              plot,(*state).plotlambda,pspec, $
                   /xstyle,/ystyle,/noerase,/device,color=color, $
                   position=[0L,(*state).ypmin, $
                            (*state).winxsize,(*state).ypmax], $
                   yrange=[(*state).colmin,(*state).colmax], $
                   xrange=(*state).xrange
            endif ;; black
            (*state).showline=-1L

          endelse ;; (*state).monopressed

          ;;===============================--------------------
          ;; Restoring the colors:

          tvlct,(*state).rclr.red,(*state).rclr.green,(*state).rclr.blue

        end ;; case: else
      endcase ;; (*state).draww.mode
    end ;; case: 'spectrumplot'

    ;;===============================--------------------
    ;; Match the mask curvature with that of the spectrum image:

    'matchcurvature': begin

      ;; Un-sensitizing the control panel:
      widget_control,(*state).wid.bmact,sensitive=0L
      widget_control,(*state).wid.bconp,sensitive=0L

      (*state).draww.mode=(*state).draww.maskcurvature

      ;; Updating the status widget:
      if (*state).track then widget_control,(*state).wid.lstat, $
         set_value='[Match curvature] Left- (1 line) or right-click (2 l' + $
         'ines) near to a brighter line in order to match the mask curva' + $
         'ture with the data'
    end ;; case: 'maskcurvature'


    ;;===============================--------------------
    ;; Shift the dispersion mask along the dispersion direction:

    'waverangeshift': begin

      ;; Un-sensitizing the control panel:
      widget_control,(*state).wid.bmact,sensitive=0L
      widget_control,(*state).wid.bconp,sensitive=0L

      (*state).draww.mode=(*state).draww.waverangeshift

      ;; Updating the status widget:
      if (*state).track then widget_control,(*state).wid.lstat, $
          set_value='[Shift the line mask] Click the left button and drag ' + $
          'the line mask across the spectrum image; finish by releasing th' + $
          'e button.'
    end ;; case: 'waverangeshift'


    ;;===============================--------------------
    ;; Set the reference point interactively:

    'imagerefchange': begin

      ;; Un-sensitizing the control panel:
      widget_control,(*state).wid.bmact,sensitive=0L
      widget_control,(*state).wid.bconp,sensitive=0L

      (*state).draww.mode=(*state).draww.refposchange

      ;; Updating the status widget:
      if (*state).track then widget_control,(*state).wid.lstat, $
          set_value='[Change pos.] Click the left button at the new refere' + $
          'nce position [wavelength bin,spectrum]; finish with the right b' + $
          'utton.'
    end ;; case: 'imagerefchange'


    ;;===============================--------------------
    ;; Change the dispersion interactively:

    'dispersioninteractivechange': begin

      ;; Un-sensitizing the control panel:
      widget_control,(*state).wid.bmact,sensitive=0L
      widget_control,(*state).wid.bconp,sensitive=0L

      (*state).draww.mode=(*state).draww.dispchange

      ;; Updating the status widget:
      if (*state).track then widget_control,(*state).wid.lstat, $
         set_value='[Change the dispersion] Click in order to stretch the' + $
                   ' dispersion interactively in the spectrum image.'
    end ;; case: 'dispersioninteractivechange'


    ;;===============================--------------------
    ;; Spectrum image motion and button events:

    'spectrumimage': begin

      case (*state).draww.mode of
        0L: return ;; The default is to do nothing

        ;;===============================--------------------
        ;;===============================--------------------
        ;;===============================--------------------
        ;; Setting the mask curvature:

        (*state).draww.maskcurvature: begin

          if event.press eq 1b and ~(*state).draww.pressed then begin

            ;;====================----------
            ;; Left mouse button - using only one line:
            ;;====================----------

            if (*state).usedeadfibers then deadfibers=(*state).deadfibers
            p3d_wavecal_calculate_mcurv,*(*state).image,event.x/relbin, $
                event.y,tmp,deadfibers=deadfibers,topwid=event.top, $
                logunit=(*state).logunit, $
                verbose=(*state).verbose,error=error,debug=(*state).debug
            if error ne 0 then return

            (*state).geo=tmp
            ;; Normalizing the line geometry array with the value of the
            ;; reference spectrum:
            (*state).geo-=(*state).geo[(*state).refrow]

            p3d_wavecal_dispmask_gui_calc_linepositions,state

            if ptr_valid((*state).plotlines) then ptr_free,(*state).plotlines
            (*state).plotlines=ptr_new((*(*state).lines)[*,(*state).refrow])

            if (*state).track then widget_control,(*state).wid.lstat, $
                set_value='[Match curvature] Done!'

            ;; Redrawing the image and the spectrum:
            p3d_wavecal_dispmask_gui_plot,state,/image,/spectrum,error=error
            if error ne 0 then return

            ;; Re-sensitizing the control panel:
            widget_control,(*state).wid.bmact,sensitive=1L
            widget_control,(*state).wid.bconp,sensitive=1L

            ;; Not accepting more events of this kind:
            (*state).draww.mode=0L
          endif else if event.press eq 4b then begin ;; event.press eq 1b

            ;;====================----------
            ;; Right mouse button - using two lines:
            ;;====================----------

            if ~(*state).draww.pressed then begin

              ;; A first line has been selected:
              if (*state).usedeadfibers then deadfibers=(*state).deadfibers
              p3d_wavecal_calculate_mcurv,*(*state).image, $
                  event.x/relbin,event.y,tmp,deadfibers=deadfibers, $
                  topwid=event.top,logunit=(*state).logunit, $
                  verbose=(*state).verbose,error=error,debug=(*state).debug
              if error ne 0 then return

              ;; Normalizing the line geometry array with the value of the
              ;; reference spectrum:
	      (*state).draww.tempf[0L]=tmp[(*state).refrow]
              if ptr_valid((*state).draww.storage) then $
                  ptr_free,(*state).draww.storage
              (*state).draww.storage=ptr_new(tmp)

              if (*state).track then widget_control,(*state).wid.lstat, $
                 set_value='[Match curvature] Please right-click near to a' + $
                 ' second brighter line; that is not too close to the firs' + $
                 't one.'
              (*state).draww.pressed=1L

            endif else begin ;; ~(*state).draww.pressed

              ;; A second line has been selected:
              if (*state).usedeadfibers then deadfibers=(*state).deadfibers
              p3d_wavecal_calculate_mcurv,*(*state).image, $
                  event.x/relbin,event.y,tmp,deadfibers=deadfibers, $
                  topwid=event.top,logunit=(*state).logunit, $
                  verbose=(*state).verbose,error=error,debug=(*state).debug
              if error ne 0 then return
	      tmp1=tmp[(*state).refrow]
              tmp-=tmp1

              if abs(tmp1-(*state).draww.tempf[0L]) $
                 le 1d-1*(*state).si[1L] then begin
                if (*state).track then widget_control,(*state).wid.lstat, $
                   set_value='[Match curvature] The two selected lines are' + $
                   ' too close to each other!'

                ;; Re-sensitizing the control panel:
                widget_control,(*state).wid.bmact,sensitive=1L
                widget_control,(*state).wid.bconp,sensitive=1L

                ;; Not accepting more events of this kind:
                (*state).draww.mode=0L
                (*state).draww.pressed=0L
                if ptr_valid((*state).draww.storage) then $
                    ptr_free,(*state).draww.storage

                return
              endif ;; abs(tmp1-(*state).draww.tempf[0L]) le
                    ;;     0.1*((*state).si[1L])

              if ~ptr_valid((*state).draww.storage) then print,'Error!!'

              (*state).geo=*(*state).draww.storage-(*state).draww.tempf[0L]
              if ptr_valid((*state).draww.storage) then $
                  ptr_free,(*state).draww.storage
              (*state).crf=(*state).draww.tempf[0L]
	      (*state).ceo=(tmp-(*state).geo)/(tmp1-(*state).draww.tempf[0L])

              p3d_wavecal_dispmask_gui_calc_linepositions,state

              if ptr_valid((*state).plotlines) then ptr_free,(*state).plotlines
              (*state).plotlines=ptr_new((*(*state).lines)[*,(*state).refrow])

              if (*state).track then widget_control,(*state).wid.lstat, $
                 set_value='[Match curvature] Done!'

              ;; Redrawing the image and the spectrum:
              p3d_wavecal_dispmask_gui_plot,state,/image,/spectrum,error=error
              if error ne 0 then return

              ;; Re-sensitizing the control panel:
              widget_control,(*state).wid.bmact,sensitive=1L
              widget_control,(*state).wid.bconp,sensitive=1L

              ;; Not accepting more events of this kind:
              (*state).draww.mode=0L
              (*state).draww.pressed=0L
            endelse ;; ~(*state).draww.pressed
          endif ;; event.press eq 1b
        end ;; case: (*state).draww.maskcurvature

        ;;===============================--------------------
        ;;===============================--------------------
        ;;===============================--------------------
        ;; Shifting the line mask:

        (*state).draww.waverangeshift: begin

          if event.release eq 1b then begin
            ;; Not accepting more events of this kind:
            (*state).draww.mode=0L
            (*state).draww.active =0b
            (*state).draww.pressed=0L

            if ptr_valid((*state).draww.storage) then $
                ptr_free,(*state).draww.storage

            ;; Re-sensitizing the control panel:
            widget_control,(*state).wid.bsavm,sensitive=0L ;; save & exit
            widget_control,(*state).wid.bsave,sensitive=0L ;; -"-
            widget_control,(*state).wid.brmas,sensitive=0L ;; reset
            widget_control,(*state).wid.bmact,sensitive=1L
            widget_control,(*state).wid.bconp,sensitive=1L

          endif else if event.press eq 1b or (*state).draww.active then begin

            if ~(*state).draww.pressed then begin
              if ptr_valid((*state).draww.storage) then $
                  ptr_free,(*state).draww.storage
              (*state).draww.storage=ptr_new({refdisp0:(*state).refdisp, $
                                              refwave0:(*state).refwave, $
                                                  geo0:(*state).geo})

              (*state).draww.tempf[0L]=event.x
              (*state).draww.pressed =1L
              (*state).draww.active  =1b
            endif ;; ~(*state).draww.pressed

            if event.x lt 0L or event.x ge (*state).stvi[1L] then return
            if event.y lt 0L or event.y ge (*state).stvi[2L] then return

            ;; Recalculate and replot until the mouse button has been released:
            tvshft=event.x-(*state).draww.tempf[0L]
            (*state).cshift=tvshft/relbin

            ;; Recalculating various properties:
            p3d_wavecal_dispmask_gui_calc_lineardisp,(*state).refcol, $
                (*(*state).draww.storage).refdisp0, $
                (*(*state).draww.storage).refwave0, $
                (*state).ddisp,(*state).refcol-(*state).cshift,tmpdisp,tmpwave

            (*state).refdisp=tmpdisp & (*state).crefdisp=tmpdisp
            (*state).refwave=tmpwave & (*state).crefwave=tmpwave

            p3d_wavecal_dispmask_gui_calc_linearlambda,state

            p3d_wavecal_dispmask_gui_calc_scalegeometry, $
                (*(*state).draww.storage).geo0, $
                (*state).ddisp,(*state).refcol,(*state).refdisp, $
                (*state).refrow,(*state).refcol+(*state).cshift, $
                (*state).refcol,tmpgeo
            (*state).geo=tmpgeo

            p3d_wavecal_dispmask_gui_calc_linepositions,state

            if ptr_valid((*state).plotlines) then ptr_free,(*state).plotlines
            (*state).plotlines=ptr_new((*(*state).lines)[*,(*state).refrow])

            tmp=strtrim((*state).refwave,2L)
            widget_control,(*state).wid.tdisw,set_value=tmp
            tmp=strtrim((*state).refdisp,2L)
            widget_control,(*state).wid.tdisp,set_value=tmp

            ;; Redrawing all plots:
            p3d_wavecal_dispmask_gui_plot,state,/image,/wavrange,/spectrum, $
                error=error
            if error ne 0 then return
          endif ;; event.release eq 1b
        end ;; case: (*state).draww.waverangeshift

        ;;===============================--------------------
        ;;===============================--------------------
        ;;===============================--------------------
        ;; Changing the dispersion:

        (*state).draww.dispchange: begin
          wset,(*state).wid.ima

          ;; Maybe the following properties should use .tempf instead of .tempi
          ;;   (*state).draww.tempi[0L] :: xold
          ;;   (*state).draww.tempi[1L] :: xprevious
          ;;   (*state).draww.tempi[4L] :: x0
          ;;   (*state).draww.tempi[5L] :: y0

          if event.release eq 1b then begin
            ;; Not accepting more events of this kind:
            (*state).draww.mode=0L
            (*state).draww.active  =0b
            (*state).draww.pressed =0L

            if ptr_valid((*state).draww.storage) then $
                ptr_free,(*state).draww.storage

            ;;===============================--------------------
            ;; Restoring the colors:

            tvlct,(*state).rclr.red,(*state).rclr.green,(*state).rclr.blue

            ;;===============================--------------------
            ;; Redrawing the spectrum image:

            p3d_wavecal_dispmask_gui_plot,state,/image,/spectrum,error=error
            if error ne 0 then return

            ;; Re-sensitizing the control panel:
            widget_control,(*state).wid.bsavm,sensitive=0L   ;; save & exit
            widget_control,(*state).wid.bsave,sensitive=0L   ;; -"-
            widget_control,(*state).wid.bcdin,sensitive=1L   ;; reset disp.
            widget_control,(*state).wid.bcdir,sensitive=1L   ;; -"-
            widget_control,(*state).wid.brmas,sensitive=0L   ;; reset
            widget_control,(*state).wid.bmact,sensitive=1L
            widget_control,(*state).wid.bconp,sensitive=1L

            msg='The dispersion-related values have been changed manuall' + $
                'y with the mouse in the spectrum image.'
            if (*state).track then $
                widget_control,(*state).wid.lstat,set_value=msg

          endif else if event.press eq 1b or (*state).draww.active then begin

            ;; Stage 0:
            if ~(*state).draww.pressed then begin

              ;;===============================--------------------
              ;; Saving the current colors, loading the ones used by this tool:

              tvlct,red,green,blue,/get
              (*state).rclr.red  =red
              (*state).rclr.green=green
              (*state).rclr.blue =blue
              tvlct,(*state).clr.red,(*state).clr.green,(*state).clr.blue

              if ptr_valid((*state).draww.storage) then $
                  ptr_free,(*state).draww.storage

              (*state).draww.storage=ptr_new({ disp0:(*state).disp, $
                                              ddisp0:(*state).ddisp, $
                                                geo0:(*state).geo})

              (*state).draww.pressed=1L
              (*state).draww.active =1b
            endif ;; ~(*state).draww.pressed

            ;; Stage 1:
            if (*state).draww.pressed eq 1L then begin

               ;;===============================--------------------
               ;; Selecting a starting wavel. bin ((*state).draww.tempi[4L]):

               (*state).draww.tempi[4L]=event.x/relbin-1L
               if (*state).draww.tempi[4L] eq (*state).refcol then begin
                 msg='[Change the dispersion] The selected wavelength bin ' + $
                     'must be different from that of the reference point. ' + $
                     'Please try again.'
                 if (*state).track then $
                   widget_control,(*state).wid.lstat,set_value=msg

                 (*state).draww.active=0b
                 return
               endif ;; (*state).draww.tempi[4L] eq (*state).refcol

               msg='[Change the dispersion] Selected wavelength bin number' + $
                   ' '+strtrim(long((*state).draww.tempi[4L]),2L)+ $
                   '; release button to finish.'
               if (*state).track then $
                  widget_control,(*state).wid.lstat,set_value=msg

               (*state).draww.tempi[5L]=event.y
               x=(*state).draww.tempi[4L]

               (*state).draww.tempi[2L]= $
                 ((*state).draww.tempi[4L]-(*state).refcol) gt 0L

               ;; Marking the starting position:
               color=(*state).cindex[(*state).cindv[4L]]

               plotsym,0,1.5,/fill
               plots,([event.x]-(*state).xoffset)*relbin, $
                     (*state).draww.tempi[5L],psym=8,color=color,/device
               plots,([event.x,event.x]-(*state).xoffset)*relbin, $
                     [0L,(*state).stvi[2L]-1L],/device,color=color

               (*state).draww.tempi[0L]=x
               (*state).draww.tempi[1L]=x

               (*state).draww.pressed=2L
            endif ;; (*state).draww.pressed eq 1L

            if (*state).draww.pressed eq 2L then begin
               (*state).ok=1L

               x=event.x/relbin-1L

               xprevious=(*state).draww.tempi[1L]
               if (x-(*state).refcol) gt 0L ne (*state).draww.tempi[2L] then $
                  (*state).ok=0L
               if ~(*state).ok and xprevious ne x then begin
                  msg='In order to set a new dispersion please '
                  msg+=(*state).draww.tempi[2L]? $
                       'click right to the reference point!': $
                       'click left to the reference point!'
                  msg+=' Try again!'
                  if (*state).track then $
                     widget_control,(*state).wid.lstat,set_value=msg
               endif

               (*state).draww.tempi[1L]=x

               if (*state).ok then (*state).draww.pressed=3L
            endif ;; (*state).draww.pressed eq 2L

            ;; Change the dispersion:
            if (*state).draww.pressed eq 3L then begin
              if x ne (*state).draww.tempi[0L] then begin
                fac=double(x-(*state).refcol)/ $
                    ((*state).draww.tempi[4L]-(*state).refcol)
                (*state).refdisp= $
                   (*(*state).draww.storage).disp0[(*state).refcol]/fac
                (*state).ddisp=(*(*state).draww.storage).ddisp0/fac^2

                p3d_wavecal_dispmask_gui_calc_linearlambda,state

                (*state).geo= $
                   (*(*state).draww.storage).geo0* $
                   (*(*state).draww.storage).disp0[(*state).refcol]/ $
                   (*state).disp[(*state).refcol]

                p3d_wavecal_dispmask_gui_calc_linepositions,state

                ptr_free,(*state).plotlines
                (*state).plotlines= $
                   ptr_new((*(*state).lines)[*,(*state).refrow])

                ;; Updating the dispersion-related widgets:
                tmp=strtrim((*state).refwave,2L)
                widget_control,(*state).wid.tdisw,set_value=tmp
                tmp=strtrim((*state).refdisp,2L)
                widget_control,(*state).wid.tdisp,set_value=tmp
                tmp=strtrim((*state).ddisp,2L)
                widget_control,(*state).wid.tdisc,set_value=tmp

                ;; Redrawing all plots:
                p3d_wavecal_dispmask_gui_plot,state, $
                    x,(*state).draww.tempi[5L],/image,/imxtra,/wavrange, $
                    /spectrum,error=error
                if error ne 0 then return
              endif ;; event.x ne (*state).draww.tempi[0L]

              (*state).draww.tempi[0L]=x
              (*state).draww.tempi[1L]=x
              (*state).draww.pressed=2L
            endif ;; (*state).draww.pressed eq 3L
          endif ;; event.release eq 1b

        end ;; case: (*state).draww.dispchange


        ;;===============================--------------------
        ;;===============================--------------------
        ;;===============================--------------------
        ;; Setting the reference spectrum and wavelength bin:

        (*state).draww.refposchange: begin

          case event.press of
            0b: begin
              if (*state).draww.pressed and event.release then $
                 (*state).draww.pressed=0L
              return ;; Doing nothing with only motion events
            end ;; case: 0b
            1b: begin ;; Setting the Only react when the button is pressed
              ;; Nothing is done if the button has already been pressed
              if (*state).draww.pressed then return

              if event.x-1L ge 0L and event.x-1L lt (*state).tvpix and $
                 event.y-1L ge 0L and event.y-1L lt (*state).stvi[2L] $
                 then begin

                if event.x/relbin-1L ne (*state).refcol or $
                   event.y-1L ne (*state).refrow then begin
                  ;; Redrawing the spectrum image:
                  p3d_wavecal_dispmask_gui_plot,state,/image, $
                      rpos=[event.x/relbin,event.y],error=error
                  if error ne 0 then return
                endif

                if ptr_valid((*state).plotlines) then $
                    ptr_free,(*state).plotlines
                (*state).plotlines=(*state).maskdone? $
                    ptr_new(*(*state).calwl): $
                    ptr_new((*(*state).lines)[*,(*state).refrow])

                oldrefcol=(*state).refcol
                if (*state).usedeadfibers then deadfibers=(*state).deadfibers
                (*state).refrow=p3d_misc_findspec(long(event.y), $
                    (*state).stvi[2L],deadfibers,topwid=(*state).top, $
                    logunit=(*state).logunit,verbose=(*state).verbose, $
                    error=error,debug=(*state).debug)
                if error ne 0 then return

                tmp=strtrim((*state).refrow+1L,2L)
                (*state).slval=(*state).refrow
                widget_control,(*state).wid.bfftx,set_value=tmp
                widget_control,(*state).wid.bffsl,set_value=(*state).refrow+1L
                sensitive=(*state).refrow+1L gt 1L
                widget_control,(*state).wid.bffdc,sensitive=sensitive
                sensitive=(*state).refrow+1L lt (*state).stvi[2L]
                widget_control,(*state).wid.bffic,sensitive=sensitive

                (*state).refcol=long(event.x/relbin)-1L
                (*state).refwave=(*state).lambda[(*state).refcol]
                (*state).refdisp=(*state).disp  [(*state).refcol]
                (*state).crefdisp=(*state).refdisp
                (*state).crefwave=(*state).refwave

                p3d_wavecal_dispmask_gui_calc_scalegeometry,(*state).geo, $
                    (*state).ddisp,(*state).refcol,(*state).refdisp, $
                    (*state).refrow,oldrefcol,(*state).refcol,tmpgeo
                (*state).geo=tmpgeo

                ;; Normalizing the geometry array:
                (*state).geo-=(*state).geo[(*state).refrow]

                ;; Text widgets:
                tmp=strtrim((*state).refrow+1L,2L)
                widget_control,(*state).wid.trefr,set_value=tmp
                tmp=strtrim((*state).refcol+1L,2L)
                widget_control,(*state).wid.trefc,set_value=tmp
                tmp=strtrim((*state).refwave,2L)
                widget_control,(*state).wid.tdisw,set_value=tmp
                tmp=strtrim((*state).refdisp,2L)
                widget_control,(*state).wid.tdisp,set_value=tmp

                ;; Redrawing the spectrum plot:
                p3d_wavecal_dispmask_gui_plot,state,/spectrum,error=error

                ;; Indicating that the button has been pressed, nothing more
                ;; to do at the moment:
                (*state).draww.pressed=1L
              endif
            end ;; case: 1b
            4b: begin ;; Turning off this mode
              ;; Not accepting more events of this kind
              (*state).draww.mode=0L

              ;; Redrawing the spectrum plot:
              p3d_wavecal_dispmask_gui_plot,state,/spectrum,error=error
              if error ne 0 then return

              ;; Re-sensitizing the control panel:
              widget_control,(*state).wid.bsavm,sensitive=0L ;; save & exit
              widget_control,(*state).wid.bsave,sensitive=0L ;; -"-
              widget_control,(*state).wid.brmas,sensitive=0L ;; reset
              widget_control,(*state).wid.bmact,sensitive=1L
              widget_control,(*state).wid.bconp,sensitive=1L
            end ;; case: 4b
          endcase ;; event.press
        end ;; case: (*state).draww.refposchange
        else: return
      endcase ;; (*state).draww.mode
    end ;; case: 'spectrumimage'


    ;;===============================--------------------
    ;; Match the selected emission lines with those present in the image:

    'matchmasktocentroids': begin

      ;; Updating the status widget:
      if (*state).track then widget_control,(*state).wid.lstat, $
          set_value='[Match line mask and data] Checking which lines in th' + $
          'e line list have a corresponding peak in the reference spectrum'

      ;; Un-sensitizing the control panel:
      widget_control,(*state).wid.bmact,sensitive=0L,/hourglass
      widget_control,(*state).wid.bconp,sensitive=0L

      ;; Extracting a reference spectrum, from which a smoothed version is
      ;; subtracted - emphasizing the emission lines:
      pspec=(*(*state).image)[*,(*state).refrow]
      p3d_misc_smooth_1d,pspec,20L,med,/median,topwid=event.top, $
          logunit=(*state).logunit,verbose=(*state).verbose,error=error, $
          debug=(*state).debug
      if error ne 0 then return
      pspec=(pspec-med)>0.0

      idx=where(*(*state).linepix gt 0L,count)
      if ~count then begin
        msg='No lines found in the matching process!'
        ret=dialog_message(msg,/information,dialog_parent=event.top,/center)
      endif else begin
        inwav= *(*state).calwl
        inpix= *(*state).linepix
        jnwav=(*(*state).calwl  )[idx]
        jnpix=(*(*state).linepix)[idx]

        ;;=====================----------
        ;; Finding out which pre-selected calibration lines are found in the
        ;; spectrum of the input image:

        p3d_wavecal_match_maskwd,pspec,inwav,jnpix,jnwav, $
            (*state).linewidth,identline,identpix,identwave,poldeg=4L, $
            niterate=5L,findwidth=(*state).linewidth,topwid=event.top, $
            logunit=(*state).logunit,verbose=(*state).verbose,error=error, $
            debug=(*state).debug
        if error ne 0 then begin
          ;; Re-sensitizing the control panel:
          widget_control,(*state).wid.bmact,sensitive=1L
          widget_control,(*state).wid.bconp,sensitive=1L
          return
        endif

        count=n_elements(identwave)
        if ~count then begin
          infostring='Centroiding RMS: -no line was positively matched-'
        endif else begin
          centroid_rms=sqrt(total((inpix[identline]-identpix)^2)/count)
          infostring='Centroiding RMS: '+string(centroid_rms,format='(e9.2)')
        endelse

        if ~count then begin
          msg=['{p3d_wavecal_match_maskwd} No entries of the line mask', $
               '    were positively identified in the calibration data!']
          ret=dialog_message(msg,/information,dialog_parent=event.top,/center)
        endif else begin ;; ~count

          ;;=====================----------
          ;; Updating the list of calibration lines to contain the
          ;; corresponding matched ones:

          tmplines=*(*state).lines
          tmplinpx=*(*state).linepix
          tmpcalwl=*(*state).calwl

          if ptr_valid((*state).lines) then ptr_free,(*state).lines
          (*state).lines=ptr_new(tmplines[identline,*])
          (*state).slines=size(*(*state).lines)

          if ptr_valid((*state).linepix) then ptr_free,(*state).linepix
          (*state).linepix=ptr_new(tmplinpx[identline])

          if ptr_valid((*state).calwl) then ptr_free,(*state).calwl
          (*state).calwl=ptr_new(tmpcalwl[identline])
          (*state).ncalwl=n_elements(*(*state).calwl)

          if widget_info((*state).lltid,/valid_id) then begin
            p3d_wavecal_dispmask_gui_view_linelist,state,tmp
            widget_control,(*state).lltid,set_value=tmp
          endif

          for k=0L,(*state).slines[1L]-1L do begin
            (*(*state).linepix)[k] =identpix[k]
            (*(*state).lines)[k,*]+=identpix[k]- $
                                    (*(*state).lines)[k,(*state).refrow]
          endfor ;; k=0L,(*state).slines[1L]-1L

          if ptr_valid((*state).plotlines) then ptr_free,(*state).plotlines
          (*state).plotlines=ptr_new((*(*state).lines)[*,(*state).refrow])

          ;; Redrawing the image and the spectrum plots:
          p3d_wavecal_dispmask_gui_plot,state,/image,/spectrum,error=error
          if error ne 0 then return
          xyouts,0.05,0.85,infostring,/normal,charsize=1.5

          ;; Re-sensitizing the control panel:
          widget_control,(*state).wid.bfitm,sensitive=1L ;; Fit centroids
          widget_control,(*state).wid.bfitl,sensitive=1L ;; -"-
          widget_control,(*state).wid.bmact,sensitive=1L
          widget_control,(*state).wid.bconp,sensitive=1L

        endelse ;; ~count
      endelse ;; ~count
    end ;; case: 'matchmasktocentroids'


    ;;===============================--------------------
    ;; Fitting the selected emission line positions with those of the image:

    'fitcentroids': begin

      ;; Updating the status widget:
      if (*state).track then widget_control,(*state).wid.lstat, $
         set_value='[Fit center positions] Calculating accurate positions' + $
         ' by fitting the matched center positions with a Gaussian profile.'

      ;; Un-sensitizing the control panel:
      widget_control,(*state).wid.bmact,sensitive=0L,/hourglass
      widget_control,(*state).wid.bconp,sensitive=0L

      lowsize=(*state).refrow/(*state).refdist
      uppsize=((*state).si[2L]-1L-(*state).refrow)/(*state).refdist
      refsize=lowsize+uppsize+1L

      refpos=(*state).refrow-lowsize*(*state).refdist+ $
             dindgen(refsize)*(*state).refdist
      refimage=dblarr((*state).si[1L],refsize)

      if (*state).nrows gt 0L then begin
        for k=0L,refsize-1L do $
           refimage[*,k]=total((*(*state).image)[*, $
              (refpos[k]-(*state).nrows)>0L: $
              (refpos[k]+(*state).nrows)<((*state).si[2L]-1L)],2L)>0.0
      endif else begin ;; nrows gt 0L
        refimage=(*(*state).image)[*,refpos]>0.0
      endelse ;; nrows gt 0L

      ;;========================================------------------------------
      ;; Fitting the center positions for all lines and spectra more accurately
      ;; using a Gaussian profile:

      p3d_wavecal_correct_maskpos,refimage, $
          (*(*state).lines)[*,(*state).refrow],(*state).linewidth,imlinepos, $
          identline,offsets=*(*state).lines,refrow=(*state).refrow, $
          fwhm=(*state).fwhm,method=(*state).method,monitor=(*state).cfitm, $
          stawid=(*state).wid.lstat,topwid=(*state).top, $
          logunit=(*state).logunit, $
          verbose=(*state).verbose,error=error,debug=(*state).debug
      if error ne 0 then return

      if identline[0L] eq -1L then begin
        if (*state).track then begin
          msg='[Create a dispersion mask] No emission line was detected in' + $
              ' the data!'
          widget_control,(*state).wid.lstat,set_value=msg
        endif
      endif else begin

        tmplines=*(*state).lines
        tmplinpx=*(*state).linepix
        tmpcalwl=*(*state).calwl

        if ptr_valid((*state).lines) then ptr_free,(*state).lines
        (*state).lines=ptr_new(imlinepos[identline,*])
        (*state).slines=size(*(*state).lines)

        if ptr_valid((*state).linepix) then ptr_free,(*state).linepix
        (*state).linepix=ptr_new(tmplinpx[identline])

        if ptr_valid((*state).plotlines) then ptr_free,(*state).plotlines
        (*state).plotlines=ptr_new((*(*state).lines)[*,(*state).refrow])

        if ptr_valid((*state).calwl) then ptr_free,(*state).calwl
        (*state).calwl=ptr_new(tmpcalwl[identline])
        (*state).ncalwl=n_elements(*(*state).calwl)

        if ptr_valid((*state).cimlinep) then ptr_free,(*state).cimlinep
        (*state).cimlinep=ptr_new(*(*state).lines)

        if widget_info((*state).lltid,/valid_id) then begin
          p3d_wavecal_dispmask_gui_view_linelist,state,tmp
          widget_control,(*state).lltid,set_value=tmp
        endif

        count=n_elements(identline)
        if ~count then begin
          infostring='Center fitting RMS: -no line was positively matched-'
        endif else begin
          tmp=total((tmplines[identline,*]-*(*state).lines)^2)/count
          centroid_rms=sqrt(temporary(tmp))
          infostring='Center fitting RMS (all spectra & lines): '+ $
                     string(centroid_rms,format='(e9.2)')
        endelse ;; ~count

        ;; Redrawing the image and the spectrum plots:
        p3d_wavecal_dispmask_gui_plot,state,/image,/spectrum,error=error
        if error ne 0 then return
        xyouts,0.05,0.85,infostring,/normal,charsize=1.5

        ;; Re-sensitizing the control panel:
        widget_control,(*state).wid.bcmam,sensitive=1L ;; Create disp.mask
        widget_control,(*state).wid.bcmas,sensitive=1L ;; -"-
        widget_control,(*state).wid.bmact,sensitive=1L
        widget_control,(*state).wid.bconp,sensitive=1L

      endelse ;; identline[0L] eq -1L
    end ;; case: 'fitcentroids'


    ;;===============================--------------------
    ;; Create a dispersion mask:

    'maskcreate': begin

      ;; Updating the status widget:
      if (*state).track then widget_control,(*state).wid.lstat, $
          set_value='[Create a dispersion mask] Calculating a dispersion m' + $
          'ask for every spectrum.'

      widget_control,(*state).top,/hourglass

      ;; Saving the current setup:
      if ptr_valid((*state).oldimage) then ptr_free,(*state).oldimage
      (*state).oldimage=ptr_new(*(*state).image)

      if ptr_valid((*state).oldtvimage) then ptr_free,(*state).oldtvimage
      (*state).oldtvimage=ptr_new(*(*state).tvimage)

      (*state).oldlambda=(*state).lambda
      (*state).olddisp  =(*state).disp
      (*state).oldddisp =(*state).ddisp

      if ptr_valid((*state).oldlines) then ptr_free,(*state).oldlines
      (*state).oldlines=ptr_new(*(*state).lines)

      ;; Calculating polynomial fitting parameters relating matched emission
      ;; lines in the spectrum image with emission lines in the calibration
      ;; line list:
      if (*state).usedeadfibers then deadfibers=(*state).deadfibers
      p3d_wavecal_fit_maskwd,*(*state).image,(*state).ncalwl,*(*state).lines, $
          *(*state).calwl,(*state).refdist,(*state).lxdeg,(*state).lydeg, $
          tmpfitpar,dispmid,residualcut=(*state).residualcut,chisq=chisq, $
          maxspecnum=maxspecnum,refrow=(*state).refrow,topwid=(*state).top, $
          deadfibers=deadfibers,sdeadfibers=(*state).sdeadfibers, $
          stawid=(*state).wid.lstat,logunit=(*state).logunit, $
          verbose=(*state).verbose,error=error,debug=(*state).debug
      if error ne 0 then return

      if ptr_valid((*state).fitpar) then ptr_free,(*state).fitpar
      (*state).fitpar=ptr_new(tmpfitpar)
      if ptr_valid((*state).chisq) then ptr_free,(*state).chisq
      (*state).chisq=ptr_new(chisq)

      ;; Creating a two-dimensional wavelength array using the polynomial
      ;; fitting parameters which were returned by p3d_wavecal_fit_maskwd:
      x=dindgen((*state).si[1L])
      template=dblarr((*state).si[1L],(*state).si[2L])
      for k=0L,(*state).si[2L]-1L do for L=0L,(*state).lxdeg do $
         template[*,k]+=(*(*state).fitpar)[k,L]*x^L

      ;; Calculating the dispersion correction of the spectrum image, using the
      ;; wavelength template:
      p3d_wavecal_dispersion_correction,*(*state).image,template,disp0, $
          dispmid,corrected,daxis=1L,topwid=(*state).top, $
          logunit=(*state).logunit,verbose=(*state).verbose,error=error, $
          debug=(*state).debug
      if error ne 0 then return

      ;; Updating the state structure with the new data
      ;; (image might have changed size):
      (*state).maskdone=1L

      if ptr_valid((*state).image) then ptr_free,(*state).image
      (*state).image=ptr_new(corrected,/no_copy)
      (*state).si=size(*(*state).image)

      (*state).npix=(*state).si[1L]
      (*state).refcol=(*state).npix/2L-1L
      (*state).tvpix=(*state).npix*relbin

      if ptr_valid((*state).tvimage) then ptr_free,(*state).tvimage
      (*state).tvimage=ptr_new(congrid(*(*state).image,(*state).tvpix, $
                                       (*state).si[2L],/cubic))
      (*state).stvi=size(*(*state).tvimage)

      for k=0L,(*state).ncalwl-1L do $
         (*(*state).lines)[k,*]=((*(*state).calwl)[k]-disp0)/dispmid

      if ptr_valid((*state).plotlines) then ptr_free,(*state).plotlines
      (*state).plotlines=ptr_new(*(*state).calwl)

      (*state).lambda=disp0+findgen((*state).npix)*dispmid
      (*state).plotlambda=(*state).lambda

      ;; Redrawing all plots:
      (*state).xrange= $
         (*state).plotlambda[([0L,(*state).winxsize]+(*state).xoffset)/relbin]

      p3d_wavecal_dispmask_gui_plot,state,/image,/wavrange,/spectrum, $
          error=error,/erase
      if error ne 0 then return

      (*state).ok=1L
      (*state).created=1b

      ;; Re-sensitizing the control panel:
      widget_control,(*state).top,update=0L
      widget_control,(*state).wid.bmatn,sensitive=0L ;; match curvature
      widget_control,(*state).wid.bmatc,sensitive=0L ;; -"-
      widget_control,(*state).wid.bshim,sensitive=0L ;; shift mask
      widget_control,(*state).wid.bshif,sensitive=0L ;; -"-
      widget_control,(*state).wid.bdlim,sensitive=0L ;; delete line in mask
      widget_control,(*state).wid.bdlin,sensitive=0L ;; -"-
      widget_control,(*state).wid.bmatm,sensitive=0L ;; match centroids
      widget_control,(*state).wid.bmatl,sensitive=0L ;; -"-
      widget_control,(*state).wid.bfitm,sensitive=0L ;; fit centroids
      widget_control,(*state).wid.bfitl,sensitive=0L ;; -"-
      widget_control,(*state).wid.bcmam,sensitive=0L ;; create disp.mask
      widget_control,(*state).wid.bcmas,sensitive=0L ;; -"-
      widget_control,(*state).wid.bpfit,sensitive=0L ;; plot pre-fits
      widget_control,(*state).wid.bffit,sensitive=1L ;; plot final fit
      widget_control,(*state).wid.bsavm,sensitive=1L ;; save & exit
      widget_control,(*state).wid.bsave,sensitive=1L ;; -"-
      widget_control,(*state).wid.bsrem,sensitive=1L ;; ref. pos. change
      widget_control,(*state).wid.bsrep,sensitive=1L ;; -"-
      widget_control,(*state).wid.bcdim,sensitive=0L ;; change disp.
      widget_control,(*state).wid.bcdis,sensitive=0L ;; -"-
      widget_control,(*state).wid.bcdin,sensitive=0L ;; reset disp.
      widget_control,(*state).wid.bcdir,sensitive=0L ;; -"-
      widget_control,(*state).wid.brmas,sensitive=1L ;; reset
      widget_control,(*state).wid.brall,sensitive=1L ;; reset all
      widget_control,(*state).wid.bscom,sensitive=0L ;; subtract bg.
      widget_control,(*state).wid.bscon,sensitive=0L ;; -"-
      if widget_info((*state).wid.bicit,/valid_id) then $
        widget_control,(*state).wid.bicit,sensitive=0L ;; image combine panel

      ;; Text widgets:
      widget_control,(*state).wid.tcomi,sensitive=1L
      widget_control,(*state).wid.tcoma,sensitive=1L
      widget_control,(*state).wid.tdisc,set_value='0'
      widget_control,(*state).wid.tdisc,editable=0L
      widget_control,(*state).wid.tdisp,set_value=strtrim(dispmid,2L)
      widget_control,(*state).wid.tdisp,editable=0L
      tmp=strtrim((*state).lambda[(*state).refcol],2L)
      widget_control,(*state).wid.tdisw,set_value=tmp
      widget_control,(*state).wid.tdisw,editable=0L
      widget_control,(*state).top,/update

      tmp=widget_info((*state).wid.bffms,/uname)
      idx=strpos(tmp,'::',/reverse_search)
      tmp=strmid(tmp,0L,idx+2L)+' #='+strtrim(maxspecnum+1L,2L)+'.'
      widget_control,(*state).wid.bffms,set_uname=tmp
      (*state).slvam=maxspecnum

;      if (*state).grateq_header then $
;         widget_control,(*state).wid.tdiso,editable=0L
    end ;; case: 'maskcreate'


    ;;===============================--------------------
    ;; Control panel tab widget events:

    'controltab': begin ;; handler for tab selection
      case event.tab of
        2L: begin
          widget_control,(*state).wid.bmact,sensitive=0L
          widget_control,(*state).wid.bscom,sensitive=0L
          widget_control,(*state).wid.bsrem,sensitive=0L
          widget_control,(*state).wid.bcdim,sensitive=0L

          widget_control,(*state).wid.dima,draw_button_events=0L
          widget_control,(*state).wid.dima,draw_motion_events=0L
          if (*state).scrollbars then $
             widget_control,(*state).wid.dima,draw_viewport_events=0L

          if ~widget_info((*state).wid.bffit,/sensitive) then begin
            p3d_wavecal_dispmask_gui_fitdisp,state
          endif else begin
            wset,(*state).wid.plo
            erase
            wset,(*state).wid.ima
            erase

            xpos =0.5*(*state).winxsize+(*state).xoffset
            tmp='Please click back here if you reset the tool'
            xyouts,xpos,0.8*(*state).winysize,tmp, $
                   alignment=0.5,charsize=4.0,/device,charthick=3.0
            tmp='Use can use this function -before-'
            xyouts,xpos,0.55*(*state).winysize,tmp, $
                   alignment=0.5,charsize=4.0,/device,charthick=3.0
            tmp='you create a dispersion mask'
            xyouts,xpos,0.3*(*state).winysize,tmp, $
                   alignment=0.5,charsize=4.0,/device,charthick=3.0
          endelse
        end
        3L: begin
          widget_control,(*state).wid.bmact,sensitive=0L
          widget_control,(*state).wid.bscom,sensitive=0L
          widget_control,(*state).wid.bsrem,sensitive=0L
          widget_control,(*state).wid.bcdim,sensitive=0L

          widget_control,(*state).wid.dima,draw_button_events=0L
          widget_control,(*state).wid.dima,draw_motion_events=0L
          if (*state).scrollbars then $
             widget_control,(*state).wid.dima,draw_viewport_events=0L
          widget_control,(*state).wid.dplo,draw_button_events=0L
          widget_control,(*state).wid.dplo,draw_motion_events=0L
          if ptr_valid((*state).fitpar) then begin
            p3d_wavecal_dispmask_gui_plotfits,state
          endif else begin
            wset,(*state).wid.plo
            erase
            wset,(*state).wid.ima
            erase

            xpos=0.5*(*state).winxsize+(*state).xoffset
            tmp='Please ''Create a Dispersion Mask'''
            xyouts,xpos,0.65*(*state).winysize,tmp, $
                alignment=0.5,charsize=4.0,/device,charthick=3.0
            tmp='before clicking back here.'
            xyouts,xpos,0.35*(*state).winysize,tmp, $
                alignment=0.5,charsize=4.0,/device,charthick=3.0
          endelse
        end
        else: begin
          widget_control,(*state).wid.bmact,/sensitive
          widget_control,(*state).wid.bscom,/sensitive
          widget_control,(*state).wid.bsrem,/sensitive
          widget_control,(*state).wid.bcdim,/sensitive

          widget_control,(*state).wid.dima,draw_button_events=1L
          widget_control,(*state).wid.dima,draw_motion_events=1L
          if (*state).scrollbars then $
             widget_control,(*state).wid.dima,draw_viewport_events=1L
          widget_control,(*state).wid.dplo,draw_button_events=1L
          widget_control,(*state).wid.dplo,draw_motion_events=1L
          wset,(*state).wid.plo
          erase
          wset,(*state).wid.ima
          erase
          p3d_wavecal_dispmask_gui_plot,state,/image,/spectrum,/wavrange, $
              error=error
        end
      endcase ;; event.tab
    end ;; case: 'controltab'

    ;;===============================--------------------
    ;; Fit the dispersion solution:

    ;; get the degree to use in the poly-fit
    'fitpolynomialorder': begin
      widget_control,event.id,get_value=val

      defvalue=(*state).lxdeg
      defstr='The entered value must be an integer.'
      on_ioerror,format_error
      tmp=long(val[0L])
      on_ioerror,NULL

      if tmp ge 0L then begin
        (*state).lxdeg=tmp
        ;; Signal return-press by moving cursor to the left:
        widget_control,event.id,set_value=strtrim(tmp,2L)
      endif else begin
        widget_control,event.id,set_value=strtrim((*state).lxdeg,2L)
      endelse ;; tmp ge 0L
    end ;; case: 'fitpolynomialorder'

    ;; Fit the dispersion:
    'fitpolynomialplot': begin
      p3d_wavecal_dispmask_gui_fitdisp,state
    end ;; case: 'fitpolynomialplot'

    ;; Plot the residuals:
    'fitshowresiduals': begin
      p3d_wavecal_dispmask_gui_fitdisp,state,residuals=residuals
      p3d_wavecal_dispmask_gui_plot,state,/spectrum,residuals=residuals, $
          error=error
      yvec=residuals.wave-residuals.fitwave
      rms=sqrt(total(yvec^2)/n_elements(yvec))
      xpos=((*state).xoffset+0.05*(*state).winxsize)/(*state).tvpix
      infostring='RMS:'+string(rms,format='(e10.3)')
      xyouts,xpos,0.85,/normal,infostring,charsize=1.5
    end ;; case: 'fitshowresiduals'


    ;;===============================--------------------
    ;; Plot the final polynomial fits:

    'fittotalspectrum': begin
      widget_control,event.top,/hourglass
      widget_control,event.id,get_value=val
      case index of
        0: tmp=val[0L]
        1: begin
          ;; Text widget:
          defvalue=(*state).slval+1L
          defstr='The entered value must be an integer.'
          on_ioerror,format_error
          tmp=long(val[0L])
          on_ioerror,NULL
        end
      endcase

      if tmp ge 2L and tmp le (*state).si[1L]-1L and $
        tmp-1L ne (*state).slval then begin

        (*state).slval=tmp-1L

        ;; Updating the related widgets:
        tmp=strtrim((*state).slval+1L,2L)
        widget_control,(*state).wid.bfftx,set_value=tmp
        widget_control,(*state).wid.bffsl,set_value=(*state).slval+1L
        sensitive=(*state).slval+1L gt 2L
        widget_control,(*state).wid.bffdc,sensitive=sensitive
        sensitive=(*state).slval+1L lt (*state).si[1L]-1L
        widget_control,(*state).wid.bffic,sensitive=sensitive

        p3d_wavecal_dispmask_gui_plotfits,state

      endif else begin
        ;; Re-setting the value of the widget that created the event:
        if ~index then widget_control,(*state).wid.bffsl, $
          set_value=(*state).slval+1L
        if index eq 1L then widget_control,(*state).wid.bfftx, $
          set_value=strtrim((*state).slval+1L,2L)
      endelse

    end ;; case: 'fittotalspectrum'

    'fittotaldecrincr': begin

      (*state).slval+=~index?-1L:1L

      ;; Updating the related widgets:
      tmp=strtrim((*state).slval+1L,2L)
      widget_control,(*state).wid.bfftx,set_value=tmp
      widget_control,(*state).wid.bffsl,set_value=(*state).slval+1L
      sensitive=(*state).slval+1L gt 2L
      widget_control,(*state).wid.bffdc,sensitive=sensitive
      sensitive=(*state).slval+1L lt (*state).si[1L]-1L
      widget_control,(*state).wid.bffic,sensitive=sensitive

      p3d_wavecal_dispmask_gui_plotfits,state

    end ;; case: 'fittotaldecrincr'

    'fittotalmaxspec': begin
      (*state).slval=(*state).slvam

      ;; Updating the related widgets:
      tmp=strtrim((*state).slval+1L,2L)
      widget_control,(*state).wid.bfftx,set_value=tmp
      widget_control,(*state).wid.bffsl,set_value=(*state).slval+1L
      sensitive=(*state).slval+1L gt 2L
      widget_control,(*state).wid.bffdc,sensitive=sensitive
      sensitive=(*state).slval+1L lt (*state).si[1L]-1L
      widget_control,(*state).wid.bffic,sensitive=sensitive

      p3d_wavecal_dispmask_gui_plotfits,state

    end ;; case: 'fittotalmaxspec'

    ;;===============================--------------------
    ;; Image split bin and image order:

    'imagesplitbin': begin
      widget_control,event.top,/hourglass
      widget_control,event.id,get_value=val
      case index of
        0: tmp=val[0L]
        1: begin
          ;; Text widget:
          defvalue=(*state).isplitbin+1L
          defstr='The entered value must be an integer.'
          on_ioerror,format_error
          tmp=long(val[0L])
          on_ioerror,NULL
        end
      endcase

      if tmp ge 2L and tmp le (*state).si[1L]-1L and $
        tmp-1L ne (*state).isplitbin then begin

        (*state).isplitbin=tmp-1L

        image=p3d_wavecal_dispmask_gui_imcombine((*state).imagename, $
            (*state).isplitbin,(*state).imageorder,(*state).daxis)
        *(*state).image=image
        *(*state).tvimage=congrid(*(*state).image,(*state).tvpix, $
                                   (*state).si[2L],/cubic)

        ;; Updating the related widgets:
        tmp=strtrim((*state).isplitbin+1L,2L)
        widget_control,(*state).wid.bictx,set_value=tmp
        widget_control,(*state).wid.bicsl,set_value=(*state).isplitbin+1L
        sensitive=(*state).isplitbin+1L gt 2L
        widget_control,(*state).wid.bicdc,sensitive=sensitive
        sensitive=(*state).isplitbin+1L lt (*state).si[1L]-1L
        widget_control,(*state).wid.bicic,sensitive=sensitive

        p3d_wavecal_dispmask_gui_plot,state,/image,/spectrum,/wavrange, $
            error=error
      endif else begin
        ;; Re-setting the value of the widget that created the event:
        if ~index then widget_control,(*state).wid.bicsl, $
          set_value=(*state).isplitbin+1L
        if index eq 1L then widget_control,(*state).wid.bictx, $
          set_value=strtrim((*state).isplitbin+1L,2L)
      endelse

    end ;; case: 'imagesplitbin'

    'imagesplitbindecrincr': begin
      (*state).isplitbin+=~index?-1L:1L

      image=p3d_wavecal_dispmask_gui_imcombine((*state).imagename, $
          (*state).isplitbin,(*state).imageorder,(*state).daxis)
      *(*state).image=image
      *(*state).tvimage=congrid(*(*state).image,(*state).tvpix, $
                   (*state).si[2L],/cubic)

      ;; Updating the related widgets:
      tmp=strtrim((*state).isplitbin+1L,2L)
      widget_control,(*state).wid.bictx,set_value=tmp
      widget_control,(*state).wid.bicsl,set_value=(*state).isplitbin+1L
      sensitive=(*state).isplitbin+1L gt 2L
      widget_control,(*state).wid.bicdc,sensitive=sensitive
      sensitive=(*state).isplitbin+1L lt (*state).si[1L]-1L
      widget_control,(*state).wid.bicic,sensitive=sensitive

      p3d_wavecal_dispmask_gui_plot,state,/image,/spectrum,/wavrange, $
          error=error
    end ;; case: 'imagesplitbindecrincr'

    'imageorder': begin
      (*state).imageorder=~(*state).imageorder

      image=p3d_wavecal_dispmask_gui_imcombine((*state).imagename, $
          (*state).isplitbin,(*state).imageorder,(*state).daxis)
      *(*state).image=image
      *(*state).tvimage=congrid(*(*state).image,(*state).tvpix, $
                   (*state).si[2L],/cubic)

      ;; Updating the related widgets:
      tmp=' '+strtrim((*state).imageorder,2L)+' '
      widget_control,(*state).wid.bicms,set_value=tmp

      p3d_wavecal_dispmask_gui_plot,state,/image,/spectrum,/wavrange, $
          error=error
    end ;; case: 'fittotalmaxspec'

    ;;===============================--------------------
    ;; Reset the dispersion value:

    'dispersionvaluereset': begin

      ;; Setting the dispersion widgets back to their default values:
      (*state).refwave=(*state).crefwave
      (*state).refdisp=(*state).crefdisp
      (*state).ddisp  =(*state).rddisp

      p3d_wavecal_dispmask_gui_calc_linearlambda,state

      disp0=(*state).disp
      (*state).geo*=disp0[(*state).refcol]/(*state).disp[(*state).refcol]

      p3d_wavecal_dispmask_gui_calc_linepositions,state

      ptr_free,(*state).plotlines
      (*state).plotlines=ptr_new((*(*state).lines)[*,(*state).refrow])

      ;; Redrawing all plots:
      p3d_wavecal_dispmask_gui_plot,state,/image,/wavrange,/spectrum, $
                                    error=error
      if error ne 0 then return

      ;; Re-sensitizing the dispersion-related widgets:
      widget_control,(*state).wid.bsavm,sensitive=0L ;; save & exit
      widget_control,(*state).wid.bsave,sensitive=0L ;; -"-
      widget_control,(*state).wid.bcdim,sensitive=1L ;; change disp.
      widget_control,(*state).wid.bcdis,sensitive=1L ;; -"-
      widget_control,(*state).wid.bcdin,sensitive=0L ;; reset disp.
      widget_control,(*state).wid.bcdir,sensitive=0L ;; -"-

      ;; Text widgets:
      tmp=strtrim((*state).ddisp,2L)
      widget_control,(*state).wid.tdisc,set_value=tmp
      widget_control,(*state).wid.tdisc,editable=1L
      tmp=strtrim((*state).refdisp,2L)
      widget_control,(*state).wid.tdisp,set_value=tmp
      widget_control,(*state).wid.tdisp,editable=1L
      tmp=strtrim((*state).refwave,2L)
      widget_control,(*state).wid.tdisw,set_value=tmp
      widget_control,(*state).wid.tdisw,editable=1L

      msg='The dispersion-related values have been reset to the routine e' + $
          'ntry values.'
      if (*state).track then widget_control,(*state).wid.lstat,set_value=msg

    end ;; case: 'dispersionvaluereset'


    ;;===============================--------------------
    ;; Subtract the continuum:

    'continuumsubtract': begin

      ;; Updating the status widget:
      if (*state).track then widget_control,(*state).wid.lstat, $
          set_value='[Subtract the background] Subtracting a dispersion-di' + $
          'rection smoothed version of the spectrum, in order to enhance e' + $
          'mission lines'

      ;; Calculating a smoothed version of the image, which is then subtracted
      ;; from the image:

      p3d_misc_median_smooth,*(*state).image,med,20L,0L,topwid=event.top, $
          logunit=(*state).logunit,verbose=(*state).verbose,error=error, $
          debug=(*state).debug
      if error ne 0 then return
      *(*state).image-=med

      *(*state).tvimage=congrid(*(*state).image,(*state).tvpix, $
                                (*state).si[2L],/cubic)

      ;; Setting the color range:
      if (*state).cmanual then begin
        medsub=median(med)
        (*state).colmin-=medsub
        (*state).colmax-=medsub
      endif else begin
        widget_control,(*state).wid.bcmmd,get_value=val
        index=widget_info((*state).wid.bcmmd,/droplist_select)
        val=val[index]

        if ~index then begin
          ;; 100% range:
          minval=min(*(*state).image,max=maxval)
        endif else begin
          ;; Calculating the clipping range:
          p3d_misc_dataclip,*(*state).image,range,percentage=val, $
              topwid=event.top,logunit=(*state).logunit, $
              verbose=(*state).verbose,error=error,debug=(*state).debug
          if error ne 0 then return

          minval=range[0L] & maxval=range[1L]
        endelse ;; ~index

        (*state).colmin=minval
        (*state).colmax=maxval
      endelse

      ;; Text widgets:
      tmp=strtrim((*state).colmin,2L)
      widget_control,(*state).wid.tcomi,set_value=tmp
      tmp=strtrim((*state).colmax,2L)
      widget_control,(*state).wid.tcoma,set_value=tmp

      ;; Redrawing the image and the spectrum plots:
      ctab=widget_info((*state).wid.bconp,/tab_current)
      if ctab ne 1L and ctab ne 2L then begin
        p3d_wavecal_dispmask_gui_plot,state,/image,/spectrum,error=error
        if error ne 0 then return
      endif

      ;; Un-sensitizing the subtract background button:
      widget_control,(*state).wid.bscom,sensitive=0L ;; subtract bg.
      widget_control,(*state).wid.bscon,sensitive=0L ;; -"-
      if widget_info((*state).wid.bicit,/valid_id) then $
        widget_control,(*state).wid.bicit,sensitive=0L ;; image combine panel

    end ;; case: 'continuumsubtract'


    ;;===============================--------------------
    ;; Reset the mask creation:

    'reset': begin
      widget_control,(*state).top,/hourglass

      if ptr_valid((*state).image) then ptr_free,(*state).image
      (*state).image=ptr_new(*(*state).oldimage)
      (*state).si=size(*(*state).image)

      if ptr_valid((*state).tvimage) then ptr_free,(*state).tvimage
      (*state).tvimage=ptr_new(*(*state).oldtvimage)
      (*state).stvi=size(*(*state).tvimage)

      if ptr_valid((*state).lines) then ptr_free,(*state).lines
      (*state).lines=ptr_new(*(*state).oldlines)
      (*state).slines=size(*(*state).lines)

      if ptr_valid((*state).plotlines) then ptr_free,(*state).plotlines
      (*state).plotlines=ptr_new((*(*state).lines)[*,(*state).refrow])

      (*state).maskdone   = 0L
      (*state).npix       = (*state).rnpix
      (*state).tvpix      = (*state).rtvpix
      (*state).lambda     = (*state).oldlambda
      (*state).disp       = (*state).olddisp
      (*state).ddisp      = (*state).oldddisp
      (*state).refdisp    = (*state).disp[(*state).refcol]
      (*state).refwave    = (*state).lambda[(*state).refcol]
      (*state).plotlambda = (*state).rplambda

      ;; Redrawing all plots:
      p3d_wavecal_dispmask_gui_plot,state,/image,/wavrange,/spectrum, $
          error=error
      if error ne 0 then return

      ;; Re-sensitizing the control panel:
      widget_control,(*state).wid.bmact,sensitive=1L,update=0L
      widget_control,(*state).wid.bconp,sensitive=1L
      widget_control,(*state).wid.bmatn,sensitive=1L ;; match curvature
      widget_control,(*state).wid.bmatc,sensitive=1L ;; -"-
      widget_control,(*state).wid.bshim,sensitive=1L ;; shift mask
      widget_control,(*state).wid.bshif,sensitive=1L ;; -"-
      widget_control,(*state).wid.bdlim,sensitive=1L ;; delete line in mask
      widget_control,(*state).wid.bdlin,sensitive=1L ;; -"-
      widget_control,(*state).wid.bmatm,sensitive=1L ;; match centroids
      widget_control,(*state).wid.bmatl,sensitive=1L ;; -"-
      widget_control,(*state).wid.bfitm,sensitive=0L ;; fit centroids
      widget_control,(*state).wid.bfitl,sensitive=0L ;; -"-
      widget_control,(*state).wid.bcmam,sensitive=0L ;; create disp.mask
      widget_control,(*state).wid.bcmas,sensitive=0L ;; -"-
      widget_control,(*state).wid.bpfit,sensitive=1L ;; plot pre-fits
      widget_control,(*state).wid.bffit,sensitive=0L ;; plot final fit
      widget_control,(*state).wid.bsavm,sensitive=0L ;; save & exit
      widget_control,(*state).wid.bsave,sensitive=0L ;; -"-
      widget_control,(*state).wid.bscom,sensitive=1L ;; subtract bg.
      widget_control,(*state).wid.bscon,sensitive=1L ;; -"-
      if widget_info((*state).wid.bicit,/valid_id) then $
        widget_control,(*state).wid.bicit,sensitive=1L ;; image combine panel
      widget_control,(*state).wid.bcdim,sensitive=1L ;; change disp.
      widget_control,(*state).wid.bcdis,sensitive=1L ;; -"-
      widget_control,(*state).wid.bcdin,sensitive=0L ;; rest disp.
      widget_control,(*state).wid.bcdir,sensitive=0L ;; -"-
      widget_control,(*state).wid.brmas,sensitive=0L ;; reset

      ;; Text widgets:
      tmp=strtrim((*state).ddisp,2L)
      widget_control,(*state).wid.tdisc,set_value=tmp
      widget_control,(*state).wid.tdisc,editable=1L
      tmp=strtrim((*state).refdisp,2L)
      widget_control,(*state).wid.tdisp,set_value=tmp
      widget_control,(*state).wid.tdisp,editable=1L
      tmp=strtrim((*state).refwave,2L)
      widget_control,(*state).wid.tdisw,set_value=tmp
      widget_control,(*state).wid.tdisw,editable=1L

;     if (*state).grateq_header then $
;        widget_control,(*state).wid.tdiso,editable=1L
      widget_control,(*state).top,/update

    end ;; case: 'reset'

    ;;===============================--------------------
    ;; Reset all:

    'resetall': begin
      widget_control,(*state).top,/hourglass

      if ptr_valid((*state).image) then ptr_free,(*state).image
      (*state).image=ptr_new(*(*state).rimage)
      (*state).si=size(*(*state).image)

      if ptr_valid((*state).tvimage) then ptr_free,(*state).tvimage
      (*state).tvimage=ptr_new(*(*state).rtvimage)
      (*state).stvi=size(*(*state).tvimage)

      if ptr_valid((*state).calwl) then ptr_free,(*state).calwl
      (*state).calwl=ptr_new((*state).rcalwl)
      (*state).ncalwl=n_elements(*(*state).calwl)

      if widget_info((*state).lltid,/valid_id) then begin
        p3d_wavecal_dispmask_gui_view_linelist,state,tmp
        widget_control,(*state).lltid,set_value=tmp
      endif

      ;; Should not 'lines' be reset as well?

      (*state).maskdone=0L
      (*state).tvpix     = (*state).rtvpix
      (*state).refcol    = (*state).rrefcol
      (*state).refrow    = (*state).rrefrow
      (*state).ddisp     = (*state).rddisp
      (*state).refdisp   = (*state).rrefdisp
      (*state).refwave   = (*state).rrefwave
      (*state).plotlambda= (*state).rplambda
      (*state).geo       = dblarr((*state).si[2L])
      (*state).ceo       = dblarr((*state).si[2L])
      (*state).crf       = 0d0
      (*state).isplitbin = (*state).risplitbin
      (*state).imageorder= (*state).rimageorder
      *(*state).saved=0b

      p3d_wavecal_dispmask_gui_calc_linearlambda,state
      p3d_wavecal_dispmask_gui_calc_linepositions,state

      if ptr_valid((*state).plotlines) then ptr_free,(*state).plotlines
      (*state).plotlines=ptr_new((*(*state).lines)[*,(*state).refrow])

      ;; Deleting the dispersion mask fitting parameters:
      if ptr_valid((*state).fitpar) then ptr_free,(*state).fitpar
      if ptr_valid((*state).cimlinep) then ptr_free,(*state).cimlinep
      if ptr_valid((*state).chisq) then ptr_free,(*state).chisq

      ;; Setting the color range:
      widget_control,(*state).wid.bcmmd,get_value=val
      index=widget_info((*state).wid.bcmmd,/droplist_select)
      val=val[index]

      (*state).cmanual=0L
      if ~index then begin
        ;; 100% range:
        minval=min(*(*state).image,max=maxval)
      endif else begin
        ;; Calculating the clipping range:
        p3d_misc_dataclip,*(*state).image,range,percentage=val, $
            topwid=event.top,logunit=(*state).logunit, $
            verbose=(*state).verbose,error=error,debug=(*state).debug
        if error ne 0 then return

        minval=range[0L] & maxval=range[1L]
      endelse
      (*state).colmin=minval
      (*state).colmax=maxval

      (*state).xrange=([0d0,(*state).winxsize]+(*state).xoffset)* $
                      (*state).tvbin

      ;; Resetting the tab that is looked at currently:
      widget_control,(*state).wid.bconp,set_tab_current=0L

      ;; Redrawing all plots:
      p3d_wavecal_dispmask_gui_plot,state,/image,/wavrange,/spectrum, $
          error=error
      if error ne 0 then return

      ;; Updating widgets:
      widget_control,(*state).wid.bmact,sensitive=1L,update=0
      widget_control,(*state).wid.bconp,sensitive=1L
      widget_control,(*state).wid.bmatn,sensitive=1L ;; match curvature
      widget_control,(*state).wid.bmatc,sensitive=1L ;; -"-
      widget_control,(*state).wid.bshim,sensitive=1L ;; shift mask
      widget_control,(*state).wid.bshif,sensitive=1L ;; -"-
      widget_control,(*state).wid.bdlim,sensitive=1L ;; delete line in mask
      widget_control,(*state).wid.bdlin,sensitive=1L ;; -"-
      widget_control,(*state).wid.bmatm,sensitive=1L ;; match centroids
      widget_control,(*state).wid.bmatl,sensitive=1L ;; -"-
      widget_control,(*state).wid.bfitm,sensitive=0L ;; fit centroids
      widget_control,(*state).wid.bfitl,sensitive=0L ;; -"-
      widget_control,(*state).wid.bcmam,sensitive=0L ;; create disp.mask
      widget_control,(*state).wid.bcmas,sensitive=0L ;; -"-
      widget_control,(*state).wid.bpfit,sensitive=1L ;; plot pre-fits
      widget_control,(*state).wid.bffit,sensitive=0L ;; plot final fit
      widget_control,(*state).wid.bsavm,sensitive=0L ;; save & exit
      widget_control,(*state).wid.bsave,sensitive=0L ;; -"-
      widget_control,(*state).wid.bcdim,sensitive=1L ;; change disp.
      widget_control,(*state).wid.bcdis,sensitive=1L ;; -"-
      widget_control,(*state).wid.bscom,sensitive=1L ;; subtract bg.
      widget_control,(*state).wid.bscon,sensitive=1L ;; -"-
      if widget_info((*state).wid.bicit,/valid_id) then $
        widget_control,(*state).wid.bicit,sensitive=1L ;; image combine panel
      widget_control,(*state).wid.bcdin,sensitive=0L ;; reset disp.
      widget_control,(*state).wid.bcdir,sensitive=0L ;; -"-
      widget_control,(*state).wid.brmas,sensitive=0L ;; reset

      ;; Text widgets:
      tmp=strtrim((*state).colmin,2L)
      widget_control,(*state).wid.tcomi,set_value=tmp
      tmp=strtrim((*state).colmax,2L)
      widget_control,(*state).wid.tcoma,set_value=tmp
      tmp=strtrim((*state).refrow+1L,2L)
      widget_control,(*state).wid.trefr,set_value=tmp
      tmp=strtrim((*state).refcol+1L,2L)
      widget_control,(*state).wid.trefc,set_value=tmp

      tmp=strtrim((*state).ddisp,2L)
      widget_control,(*state).wid.tdisc,set_value=tmp
      widget_control,(*state).wid.tdisc,editable=1L
      tmp=strtrim((*state).refdisp,2L)
      widget_control,(*state).wid.tdisp,set_value=tmp
      widget_control,(*state).wid.tdisp,editable=1L
      tmp=strtrim((*state).refwave,2L)
      widget_control,(*state).wid.tdisw,set_value=tmp
      widget_control,(*state).wid.tdisw,editable=1L
;     if (*state).grateq_header then $
;        widget_control,(*state).wid.tdiso,editable=1L

      ;; Image combination widgets:
      if widget_info((*state).wid.bicit,/valid_id) then begin
        tmp=strtrim((*state).isplitbin+1L,2L)
        widget_control,(*state).wid.bictx,set_value=tmp
        widget_control,(*state).wid.bicsl,set_value=(*state).isplitbin+1L
        tmp=' '+strtrim((*state).imageorder,2L)+' '
        widget_control,(*state).wid.bicms,set_value=tmp
        sensitive=(*state).isplitbin+1L gt 1L
        widget_control,(*state).wid.bicdc,sensitive=sensitive
        sensitive=(*state).isplitbin+1L lt (*state).stvi[1L]
        widget_control,(*state).wid.bicic,sensitive=sensitive
      endif ;; widget_info((*state).wid.bicit,/valid_id)

      widget_control,(*state).top,/update

    end ;; case: 'resetall'


    ;;===============================--------------------
    ;; Save the dispersion mask, and exit the widget tool:

    'savemaskandexit': begin
      widget_control,(*state).top,/hourglass

      ;; Updating the status widget:
      if (*state).track then widget_control,(*state).wid.lstat, $
          set_value='[Save and exit] Saving the dispersion mask to "'+ $
          file_basename((*state).tfile[0L])+'".'

      p3d_wavecal_dispmask_add_spec,*(*state).fitpar,(*state).calibspec, $
          lparams,topwid=(*state).top,logunit=(*state).logunit, $
          verbose=(*state).verbose,error=error,debug=(*state).debug
      if error ne 0 then return

      hdr=headfits((*state).imagename[0L]) & header=hdr

      fxhmake,hdr,*(*state).fitpar

      sp=size(lparams)
      fxaddpar,header,'NAXIS1',sp[1L]
      fxaddpar,header,'NAXIS2',sp[2L]
      fxaddpar,header,'BITPIX',fxpar(hdr,'BITPIX')
      fxaddpar,header,'BZERO',0

      ;; Adding information on the extracted data file:
      p3d_misc_fitsparsplit,'IMEXTR',strtrim((*state).imagename[0L],2L), $
          header,' Raw data filename',topwid=topwid,logunit=logunit, $
          verbose=verbose,error=error,debug=debug

      if n_elements((*state).imagename) eq 2L then begin
        p3d_misc_fitsparsplit,'IMEXT2',strtrim((*state).imagename[1L],2L), $
            header,' Raw data filename 2.',topwid=topwid,logunit=logunit, $
            verbose=verbose,error=error,debug=debug
        fxaddpar,header,'IMORDER',(*state).imageorder
        fxaddpar,header,'IMSPBIN',(*state).isplitbin
        fxaddpar,header,'IMDAXIS',(*state).daxis
      endif ;; n_elements((*state).imagename) eq 2L

      ;; Writing the final file:
      writefits,(*state).tfile[0L],lparams,header,compress=(*state).compress
      *(*state).saved=1b

      ;;===============================--------------------
      ;; Logging:

      msg='Saved the dispersion mask to: "'+ $
          p3d_misc_pathify((*state).tfile[0L],/dpath)+'".'
      error=p3d_misc_logger(msg,(*state).logunit,rname=rname, $
          verbose=(*state).verbose ge 1)

      ;; Exiting this tool:
      widget_control,event.top,/destroy
      return
    end ;; case: 'savemaskandexit'


    ;;===============================--------------------
    ;; Set the ref. wavelength (0), ref. dispersion (1), and ddisp (2):

    'refwavedisp': begin
      widget_control,event.id,get_value=val

      case index of
        0: defvalue=(*state).refwave
        1: defvalue=(*state).refdisp
        2: defvalue=(*state).ddisp
      endcase

      defstr='The entered value must be of decimal type.'
      on_ioerror,format_error
      tmp=double(val[0L])
      on_ioerror,NULL

      okval=index eq 2L?1L:tmp gt 0d0
      if tmp ne defvalue and okval then begin
        case index of
          0: begin & (*state).refwave=tmp & (*state).crefwave=tmp & end
          1: begin & (*state).refdisp=tmp & (*state).crefdisp=tmp & end
          2: (*state).ddisp=tmp
        endcase

        p3d_wavecal_dispmask_gui_calc_linearlambda,state
        p3d_wavecal_dispmask_gui_calc_linepositions,state

        ptr_free,(*state).plotlines
        (*state).plotlines=ptr_new((*(*state).lines)[*,(*state).refrow])

        ;; Redrawing all plots:
        p3d_wavecal_dispmask_gui_plot,state,/image,/wavrange,/spectrum, $
            error=error
        if error ne 0 then return

        widget_control,(*state).wid.bcdin,sensitive=1L ;; reset disp.
        widget_control,(*state).wid.bcdir,sensitive=1L ;; -"-
      endif ;; tmp ne defvalue and tmp gt 0d0

    end ;; case: 'refwavedisp'

    ;;===============================--------------------
    ;; Reference spectrum (image row, image column):

    'imagerefrowcol': begin
      widget_control,event.id,get_value=val

      defvalue=~index?(*state).refrow:(*state).refcol
      defstr='The entered value must be an integer.'
      on_ioerror,format_error

      ;; Checking the value, it mustn't be <0 or >(# of spectra||bins):
      tmp__=~index?2L:1L
      tmp=long(val[0L])-1L
      on_ioerror,NULL

      if tmp ne defvalue and $
         tmp ge 0L and tmp le (*state).si[tmp__]-1L then begin

        if ~index then begin
          if (*state).usedeadfibers then deadfibers=(*state).deadfibers
          tmp=p3d_misc_findspec(tmp,(*state).stvi[2L], $
                  deadfibers,topwid=(*state).top,logunit=(*state).logunit, $
                  verbose=(*state).verbose,error=error,debug=(*state).debug)
          if error ne 0 then return
        endif ;; ~index

        if ~index then (*state).refrow=tmp else (*state).refcol=tmp
        widget_control,event.id,set_value=strtrim(tmp+1L,2L)

        ;; Updating the widget slider in the final fit plot:
        if ~index then begin
          (*state).slval=(*state).refrow
          widget_control,(*state).wid.bffsl,set_value=tmp+1L
          widget_control,(*state).wid.bfftx,set_value=strtrim(tmp+1L,2L)
          sensitive=tmp+1L gt 1L
          widget_control,(*state).wid.bffdc,sensitive=sensitive
          sensitive=tmp+1L lt (*state).stvi[2L]
          widget_control,(*state).wid.bffic,sensitive=sensitive
        endif

      endif else begin
        widget_control,event.id,set_value=strtrim(defvalue+1L,2L)
      endelse

      ;; Redrawing the image and the spectrum:
      p3d_wavecal_dispmask_gui_plot,state,/image,/spectrum,error=error
      if error ne 0 then return
    end ;; case: 'imagerefrowcol'


    ;;================================================
    ;; Colortable selection:

    'colortable': begin
      xloadct,bottom=(*state).cbottom,group=event.top, $
          ncolors=!d.table_size-(*state).cbottom,updatecbdata=state, $
          /use_current,updatecallback='p3d_wavecal_dispmask_gui_xloadct'

       ;; Only allowing one instance of xloadct to be run:
       widget_control,(*state).wid.bcolo,sensitive=0L
    end ;; case: 'colortable'

    'colortablesel': begin

      p3d_misc_colortable,long(index),bottom=(*state).cbottom,colors=colors, $
          topwid=event.top,error=error,verbose=(*state).verbose, $
          debug=(*state).debug
      if error ne 0 then return

      (*state).colortable=long(index)
      (*state).clr=colors

      ctab=widget_info((*state).wid.bconp,/tab_current)
      if ctab ne 1L and ctab ne 2L then begin
        p3d_wavecal_dispmask_gui_plot,state,/image,/wavrange,/spectrum, $
            error=error
      endif
      
    end ;; case: 'colortablesel'

    ;;===============================--------------------
    ;; Toggle-button to view or hide plots of Gaussian fits:

    'viewfits': begin
      (*state).cfitm=~((*state).cfitm-2L)?0L:(*state).cfitm+1L
      case (*state).cfitm of
        0L:   tmp='Hide all centering fits '
        1L:   tmp='View refe. spectrum fits'
        else: tmp='View fits of all spectra'
      endcase
      widget_control,(*state).wid.bvfit,set_value=tmp
    end ;; case: 'viewfits'

    ;;===============================--------------------
    ;; Invert the image value ranges:

    'colorinvert': begin
      *(*state).image=-*(*state).image

      *(*state).tvimage=congrid(*(*state).image,(*state).tvpix, $
                                (*state).si[2L],/cubic)

      tmpcol=(*state).colmin
      (*state).colmin=-(*state).colmax
      (*state).colmax=-tmpcol

      tmp=strtrim((*state).colmin,2L)
      widget_control,(*state).wid.tcomi,set_value=tmp
      tmp=strtrim((*state).colmax,2L)
      widget_control,(*state).wid.tcoma,set_value=tmp

      ;; Redrawing the image and the spectrum plots:
      ctab=widget_info((*state).wid.bconp,/tab_current)
      if ctab ne 1L and ctab ne 2L then begin
        p3d_wavecal_dispmask_gui_plot,state,/image,/spectrum,error=error
        if error ne 0 then return
      endif
    end ;; case: 'colorinvert'

    ;;================================================
    ;; Spectrum image; color range droplist:

    'imagecolordroplist': begin

      ;; Determining if the event came from the droplist or the menu buttons:
      if tmpstrname eq 'widget_button' then begin
        val=strtrim(index,2L)+'%'
        case index of
          100: choice=1L
          99.5: choice=2L
          99: choice=3L
          98: choice=4L
          97: choice=5L
          95: choice=6L
          90: choice=7L
          80: choice=8L
          70: choice=9L
        endcase

        ;; Updating the droplist widget:
        widget_control,(*state).wid.bcmmd,set_droplist_select=choice
      endif else begin
        ;; Extracting the percentage (droplist):
        widget_control,event.id,get_value=val
        val=val[event.index]
        choice=event.index
      endelse

      case choice of
        0: begin
          ;; Values are not changed:
          (*state).cmanual=1L
        end
        1: begin
          minval=min(*(*state).image,max=maxval)
          (*state).cmanual=0L
        end
        else: begin
          ;; Calculating the clipping range (histogram):
          p3d_misc_dataclip,*(*state).image,range,percentage=val, $
              topwid=event.top,verbose=(*state).verbose,error=error, $
              debug=(*state).debug
          if error ne 0 then return

          minval=range[0L] & maxval=range[1L]
          (*state).cmanual=0L
        end
      endcase

      if choice gt 0L then begin
        (*state).colmin=minval
        (*state).colmax=maxval
      endif

      ;; Redrawing the image and the spectrum:
      ctab=widget_info((*state).wid.bconp,/tab_current)
      if ctab ne 1L and ctab ne 2L then begin
        p3d_wavecal_dispmask_gui_plot,state,/image,/spectrum,error=error
        if error ne 0 then return
      endif

      ;; Updating the color value widgets:
      tmp=strtrim((*state).colmin,2L)
      widget_control,(*state).wid.tcomi,set_value=tmp
      tmp=strtrim((*state).colmax,2L)
      widget_control,(*state).wid.tcoma,set_value=tmp

    end ;; case: 'imagecolordroplist'

    'imagecolorminmax': begin
      widget_control,event.id,get_value=val
      
      defvalue=index?(*state).colmax:(*state).colmin
      defstr='The entered value must be of decimal type.'
      on_ioerror,format_error

      tmp=float(val)
      if tmp ne (*state).colmin and tmp ne (*state).colmax then begin
        case index of
          0: (*state).colmin=tmp
          1: (*state).colmax=tmp
        endcase

        ;; Redrawing the image:
        p3d_wavecal_dispmask_gui_plot,state,/image,/spectrum,error=error
        if error ne 0 then return

        (*state).cmanual=1L ;; Making manual color value settings

        ;; Updating widgets:
        widget_control,(*state).wid.bcmmd,set_droplist_select=0L
      endif
    end ;; case: 'imagecolorminmax'

    ;;===============================--------------------
    ;; Reading the contents of one or several linelists:

    'readlinelist': begin
      path=file_dirname((*state).imagename[0L],/mark_directory)
      title='p3d: please select one or more linelist files'
      filename=dialog_pickfile(/must_exist,/multiple_files,filter='*.dat', $
          dialog_parent=event.top,path=path,title=title)
      if filename[0L] ne '' then begin
        nfilename=n_elements(filename)

        on_ioerror,file_error
        for i=0L,nfilename-1L do begin
          defstr='Error reading the line list "'+filename[i]+'".'
          silent=(*state).verbose lt 3
          readcol,filename[i],tmp,format='a',comment=';',silent=silent
          calwl=~i?tmp:[calwl,tmp]
        endfor ;; i=0L,nfilename-1L
        calwl=double(calwl)
        on_ioerror,NULL ;; turning off the IO error handler

        ;; Only keeping the lines within the current wavelength range:
        pos=where(calwl ge min((*state).lambda) and $
                  calwl le max((*state).lambda),count)

        if ~count and (*state).track then begin
          msg='No lines were found in the selected range; reverting to the' + $
              ' previous set of lines.'
          widget_control,(*state).wid.lstat,set_value=msg
        endif

        ptr_free,(*state).calwl
        (*state).calwl =ptr_new(calwl[pos])
        (*state).ncalwl=n_elements(*(*state).calwl)

        if widget_info((*state).lltid,/valid_id) then begin
          p3d_wavecal_dispmask_gui_view_linelist,state,tmp
          widget_control,(*state).lltid,set_value=tmp
        endif

        ;; Updating the calibration line arrays in state:
        p3d_wavecal_dispmask_gui_calc_linepositions,state

        ;; Normalizing the geometry array:
        (*state).geo-=(*state).geo[(*state).refrow]

        ptr_free,(*state).plotlines
        (*state).plotlines=ptr_new((*(*state).lines)[*,(*state).refrow])

        ;; Redrawing the image and the spectrum plots:
        p3d_wavecal_dispmask_gui_plot,state,/image,/spectrum,error=error
        if error ne 0 then return

        ;; Saving the filenames:
        ptr_free,(*state).linelist
        (*state).linelist=ptr_new(filename)

      endif ;; filename[0L] ne ''
    end ;; case: 'readlinelist'

    ;;===============================--------------------
    ;; Save the linelist:

    'savelinelist': begin
      path=file_dirname((*state).imagename[0L],/mark_directory)
      title='p3d: please specify a file to write to'
      filename=dialog_pickfile(/overwrite_prompt,dialog_parent=event.top, $
          title=title,path=path)
      if filename ne '' then begin
        defstr='Error while saving the linelist to "'+filename+'".'
        on_ioerror,file_error
        openw,unit,filename,/get_lun

        ;; Include a header:
        fmtstr='(f9.3)'
        array=string(format=fmtstr,transpose(*(*state).calwl))
        for i=0L,n_elements(array)-1L do printf,unit,strtrim(array[i],2L)
        free_lun,unit
        on_ioerror,NULL
      endif ;; filename ne ''
    end ;; case: 'savelinelist'

    ;;===============================--------------------
    ;; View the linelist:

    'viewlinelist': begin
      p3d_wavecal_dispmask_gui_view_linelist,state,tmp
      xdisplayfile,'',done_button='Exit',/grow,text=tmp,group=event.top, $
        title='Viewing the currently used linelist',wtext=wid
      (*state).lltid=wid
    end ;; case: 'viewlinelist'


    ;;===============================--------------------
    ;; Spectrum order setting [PMAS]:

    'spectrumorder': begin
      widget_control,event.id,get_value=val
      t_order=long(val[0L])

      defvalue=(*state).n_order
      defstr='The entered value must be an integer.'
      on_ioerror,format_error

      ;; Changing the decomposed mode - if required:
      device,get_decomposed=decomposed
      if decomposed then device,decomposed=0L

;THIS IS NOT A PORTABLE IMPLEMENTATION (2048)
      p3d_wavecal_grating_eq,(*state).pmas.lpmm,0L,(*state).pmas.alpha, $
          /grotpos,2048L,t_order,pix,tlambda,/flip,topwid=event.top, $
          logunit=(*state).logunit,verbose=(*state).verbose,error=error, $
          debug=(*state).debug
      if error ne 0 then return

      oldflip=(*state).doflip
      if tlambda[2047L] gt tlambda[0L] then $
         (*state).doflip=0L $
      else $
         (*state).doflip=1L

      if (*state).doflip then tlambda=reverse(tlambda,1L)
      if (*state).doflip ne oldflip then $
         *(*state).image=reverse(*(*state).image,1L)

      *(*state).tvimage=congrid(*(*state).image,(*state).tvpix, $
                      (*state).si[2L],/cubic)

      (*state).pmas.pix=pix
      (*state).pmas.tlambda=tlambda

      n=n_elements((*state).linelist)
      on_ioerror,file_error
      for k=0L,n-1L do begin
        defstr='Error reading the line list "'+(*(*state).linelist)[k]+'".'
        readcol,(*(*state).linelist)[k],tmp,format='A',comment='#', $
            silent=(*state).verbose lt 3
        calwl=~k?tmp:[calwl,tmp]
      endfor
      calwl=double(calwl)

      on_ioerror,NULL ;; turning off the IO error handler

      (*state).lambda=congrid(tlambda,(*state).npix, $
                              (*state).si[1L],/interp)
      (*state).plotlambda=findgen((*state).npix)

       dlambda= (*state).lambda[1L:(*state).npix-1L]- $
                (*state).lambda[0L:(*state).npix-2L]
      ddlambda=         dlambda[1L:(*state).npix-2L]- $
                        dlambda[0L:(*state).npix-3L]

      (*state).refwave= (*state). lambda[refcol]
      (*state).refdisp=          dlambda[refcol]
      (*state).ddisp=mean(ddlambda,/double)

      (*state).crefdisp=(*state).refdisp
      (*state).crefwave=(*state).refwave

      (*state).disp=dblarr((*state).npix)
      (*state).disp[round((*state).refcol)]=(*state).refdisp

      for k=round((*state).refcol)+1L,(*state).npix-1L do $
         (*state).disp[k]=(*state).disp[k-1L]+(*state).ddisp
      for k=round((*state).refcol)-1L,0L,-1L  do $
         (*state).disp[k]=(*state).disp[k+1L]-(*state).ddisp

      (*state).lambda=dblarr((*state).npix)
      (*state).lambda[round((*state).refcol)]= $
         (*state).refwave-((*state).refcol-round((*state).refcol))* $
         (*state).disp[round((*state).refcol)]

      for k=round((*state).refcol)+1L,(*state).npix-1L do $
         (*state).lambda[k]=(*state).lambda[k-1L]+(*state).disp[k]
      for k=round((*state).refcol)-1L,0L,-1L  do $
         (*state).lambda[k]=(*state).lambda[k+1L]-(*state).disp[k]

      pos=where(calwl ge min((*state).lambda) and $
                calwl le max((*state).lambda),count)
      if count ne 0L then begin
        (*state).n_order=t_order

        ;; Only keeping the lines within the current wavelength range:
        ptr_free,(*state).calwl
        (*state).calwl=ptr_new(calwl[pos])
        (*state).ncalwl=n_elements(*(*state).calwl)

        if widget_info((*state).lltid,/valid_id) then begin
          p3d_wavecal_dispmask_gui_view_linelist,state,tmp
          widget_control,(*state).lltid,set_value=tmp
        endif

        ;; Updating the calibration line arrays in state:
        ptr_free,(*state).linepix
        (*state).linepix=ptr_new(dblarr((*state).ncalwl))

        ptr_free,(*state).lines
        (*state).lines=ptr_new(dblarr((*state).ncalwl,(*state).si[2L]))
        (*state).slines=size(*(*state).lines)

        if (*state).lambda[0L] lt (*state).lambda[(*state).npix-1L] then begin
          for k=0L,(*state).ncalwl-1L do begin
            pos=where((*state).lambda ge (*(*state).calwl)[k])
            (*(*state).linepix)[k]=pos[0L]
            (*(*state).lines)[k,*]=(*(*state).linepix)[k]
          endfor
        endif else begin
          for k=0L,n-1L do begin
            pos=where((*state).lambda le (*(*state).calwl)[k])
            (*(*state).linepix)[k]=pos[0L]
            (*(*state).lines)[k,*]=(*(*state).linepix)[k]
          endfor
        endelse

        ptr_free,(*state).plotlines
        (*state).plotlines=ptr_new(*(*state).linepix)

        *(*state).rimage  =  *(*state).image
;       resetlines        =  (*state).lines
;       resetdisp         =  (*state).disp
        (*state).rddisp   =  (*state).ddisp
;       resetlambda       =  (*state).lambda
;       resetlinepix      =  (*state).linepix
        *(*state).rtvimage= *(*state).tvimage
        (*state).rnpix    =  (*state).npix
        (*state).rtvpix   =  (*state).tvpix
        (*state).rplambda =  (*state).plotlambda

        ;; Redrawing all plots:
        p3d_wavecal_dispmask_gui_plot,state,/image,/wavrange,/spectrum, $
                                    error=error
        if error ne 0 then return

        ;; This fits badly in the above routine:
        wset,(*state).wid.ima
        for k=0L,(*state).ncalwl-1L do $
           plots,[(*(*state).linepix)[k]*relbin, $
                  (*(*state).linepix)[k]*relbin], $
                 [0L,(*state).stvi[2L]],/device

        tmp=strtrim((*state).refwave,2L)
        widget_control,(*state).wid.tdisw,set_value=tmp
        tmp=strtrim((*state).refdisp,2L)
        widget_control,(*state).wid.tdisp,set_value=tmp
        tmp=strtrim((*state).ddisp,2L)
        widget_control,(*state).wid.tdisc,set_value=tmp

      endif else begin
        print,rname+'No line found in the linelist for the wavelength ra' + $
              'nge ' + strtrim((*state).lambda[0L],2L) + $
              ' - '  + strtrim((*state).lambda[(*state).npix-1L],2L) + '�.'
        widget_control,(*state).wid.tdiso, $
                       set_value=strtrim((*state).n_order,2L)
      endelse

      ;; Changing back the decomposed state, if required:
      if decomposed then device,decomposed=decomposed

    end ;; case: 'spectrumorder'


    ;;===============================--------------------
    ;; Exit the widget tool:

    'exit': begin
      if (*state).created then begin
        msg=['A dispersion mask has been created, but is not saved.', $
             'Do you nevertheless want to quit this tool?']
        ret=dialog_message(msg,/center,dialog_parent=event.top,/default_no, $
                /question)
        if ret ne 'Yes' then return
      endif
      widget_control,event.top,/destroy
      return
    end ;; case: 'exit'

  endcase ;; uval

  if decomposed then device,decomposed=decomposed

  return

file_error:
  if n_elements(unit) ne 0L then begin
    tmp=fstat(unit)
    if tmp.open then free_lun,unit ;; closing files which may be open
  endif
  if (*state).track then widget_control,(*state).wid.lstat,set_value=defstr
  on_ioerror,NULL
  if decomposed then device,decomposed=decomposed
  return

format_error:
  widget_control,event.id,set_value=strtrim(defvalue,2L)
  if (*state).track then widget_control,(*state).wid.lstat,set_value=defstr
  on_ioerror,NULL
  if decomposed then device,decomposed=decomposed
  return
END ;;; procedure: p3d_wavecal_dispmask_gui_event


PRO p3d_wavecal_dispmask_gui_resize,event
  compile_opt hidden,IDL2

  ;; Get the state information:
  widget_control,event.top,get_uvalue=state

  widget_control,event.id,tlb_get_size=base_size
  xchange=base_size[0L]-(*state).base_size[0L]
  ychange=base_size[1L]-(*state).base_size[1L]

  ;; Checking the new size against the minimum and maximum possible values:
  ;; If the height becomes too large to fit the screen, then the
  ;; largest possible y-size is used, where as the x-size has to be smaller:

  tmpima=widget_info((*state).wid.dima,/geometry)
  if tmpima.xsize+xchange lt (*state).minxsize then $
     xchange=(*state).minxsize-tmpima.xsize
  
  tmpplo=widget_info((*state).wid.dplo,/geometry)
  poldy=tmpplo.ysize
  pnewy=(tmpplo.xsize+xchange)/4L & chy=pnewy-tmpplo.ysize

  ioldx=tmpima.xsize
  ioldy=tmpima.ysize
  inewy=tmpima.ysize+ychange-chy<(*state).si[2L]

  ychange=long(inewy-ioldy)
  if tmpima.ysize+ychange lt (*state).minysize then begin
    ychange=(*state).minysize-tmpima.ysize
    inewy=tmpima.ysize+ychange-chy<(*state).si[2L]
  endif

  tmp=(*state).base_size[1L]+(pnewy-poldy)+(inewy-ioldy)<(*state).maxysize
  ychange_real=tmp-(*state).base_size[1L]
  xchange=4L*(ychange_real-ychange)

  wxsize=tmpima.xsize+xchange
  wysize=tmpima.ysize+ychange

  if ~(*state).scrollbars then wysize=wysize>(*state).si[2L]

  ;; Re-sizing the draw widgets:
  widget_control,(*state).top,update=0L
;  widget_control,(*state).dimawid,draw_xsize=wxsize,draw_ysize=wysize
  widget_control,(*state).wid.dwav,draw_xsize=wxsize
  widget_control,(*state).wid.dplo,draw_xsize=wxsize,draw_ysize=wxsize/4L
  widget_control,(*state).top,update=1L

  ;; Resizing the status line widget:
  labwidth=wxsize
  widget_control,(*state).wid.lstat,xsize=labwidth

  ;; Storing the base size in state:
  tmp=widget_info(event.top,/geometry)
  (*state).base_size=[tmp.xsize,tmp.ysize]

  ;; Re-sizing the TV-images in (*state):
  (*state).tvpix=wxsize
  (*state).tvbin=double((*state).npix)/(*state).tvpix
  (*state).stvi=size(tvimage)

  tvimage=congrid(*(*state).image,(*state).tvpix,(*state).si[2L],/cubic)
  if ptr_valid((*state).tvimage) then ptr_free,(*state).tvimage
  (*state).tvimage=ptr_new(tvimage,/no_copy)

  (*state).rtvpix=(*state).tvpix
  if ptr_valid((*state).rtvimage) then begin
    rtvimage=congrid(*(*state).rtvimage,(*state).rtvpix,(*state).si[2L], $
                     /cubic)
    ptr_free,(*state).rtvimage
    (*state).rtvimage=ptr_new(rtvimage,/no_copy)
  endif

  if ptr_valid((*state).oldtvimage) then begin
    oldtvimage=congrid(*(*state).oldtvimage,(*state).tvpix,(*state).si[2L], $
                       /cubic)
    ptr_free,(*state).oldtvimage
    (*state).oldtvimage=ptr_new(oldtvimage,/no_copy)
  endif

  (*state).ypmax=wxsize/4L-1L

  ;; Redrawing all plots:
  p3d_wavecal_dispmask_gui_plot,state,/image,/wavrange,/spectrum, $
                             error=error
  if error ne 0 then return

  return
END ;;; procedure: p3d_wavecal_dispmask_gui_resize


PRO p3d_wavecal_dispmask_gui_cleanup,id
  compile_opt hidden,IDL2

  ;; Get the state information:
  widget_control,id,get_uvalue=state

  if (*state).verbose ge 3 then print,'p3d_wavecal_dispmask_gui: cleanup.'

  if ptr_valid((*state).lines)      then ptr_free,(*state).lines
  if ptr_valid((*state).linepix)    then ptr_free,(*state).linepix
  if ptr_valid((*state).calwl)      then ptr_free,(*state).calwl
  if ptr_valid((*state).plotlines)  then ptr_free,(*state).plotlines
  if ptr_valid((*state).linelist)   then ptr_free,(*state).linelist
  if ptr_valid((*state).image)      then ptr_free,(*state).image
  if ptr_valid((*state).rimage)     then ptr_free,(*state).rimage
  if ptr_valid((*state).tvimage)    then ptr_free,(*state).tvimage
  if ptr_valid((*state).rtvimage)   then ptr_free,(*state).rtvimage
  if ptr_valid((*state).oldimage)   then ptr_free,(*state).oldimage
  if ptr_valid((*state).oldtvimage) then ptr_free,(*state).oldtvimage
  if ptr_valid((*state).oldlines)   then ptr_free,(*state).oldlines
  if ptr_valid((*state).fitpar)     then ptr_free,(*state).fitpar
  if ptr_valid((*state).chisq)      then ptr_free,(*state).chisq
  if ptr_valid((*state).cimlinep)   then ptr_free,(*state).cimlinep
  if ptr_valid((*state).draww.storage) then ptr_free,(*state).draww.storage

  if ptr_valid(state) then ptr_free,state

  return
END ;;; end of: p3d_wavecal_dispmask_gui_cleanup


PRO p3d_wavecal_dispmask_gui,imagefile,linelist,gratingfile,ofilename, $
        linewidth=linewidth,refdist=refdist,nrows=nrows,kwrdlist=kwrdlist, $
        pmas=pmas,cdelt=cdelt,xdeg=xdeg,ydeg=ydeg,saved=saved, $
        colortable=colortable,cindex=cindex,cindv=cindv,bottom=bottom, $
        residualcut=residualcut,fwhm=fwhm,method=method,postable=postable, $
        deadfibers=deadfibers,sdeadfibers=sdeadfibers, $
        removecalfibers=removecalfibers,userparfile=userparfile, $
        imageorder=imageorder,imagesplitbin=isplitbin,dflip=dflip, $
        compress=compress,dbin=dbin,track=track,daxis=daxis,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  error=0 & rname='p3d_wavecal_dispmask_gui: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)
  if ~n_elements(logunit) then logunit=0L

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_wavecal_dispmask_gui'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  if ~n_elements(daxis) then daxis=1L
  s=size(daxis)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif

  si=size(imagefile)
  nimagefile=si[si[0L]+2L]
  if nimagefile ne 1L and nimagefile ne 2L or si[si[0L]+1L] ne 7L then begin
    errmsg='IMAGEFILE must be a string pointing to one, or two, existing file.'
    goto,error_handler
  endif
  for i=0L,si[si[0L]+2L]-1L do begin
    if ~file_test(imagefile[i],/regular,/read) then begin
      errmsg='Cannot find (or read) the IMAGEFILE "'+imagefile[i]+'".'
      goto,error_handler
    endif
  endfor

  s=size(linelist)
  if ~s[s[0L]+2L] or s[s[0L]+1L] ne 7L then begin
    errmsg='LINELIST must be a string (array), with names of one or severa' + $
           'l existing file(s).'
    goto,error_handler
  endif
  for i=0L,s[s[0L]+2L]-1L do begin
    if ~file_test(linelist[i],/regular,/read) then begin
      errmsg='Cannot find the LINELIST file ['+strtrim(i,2L)+'] "'+ $
             linelist[i]+'".'
      goto,error_handler
    endif
  endfor

  s=size(gratingfile)
  if s[s[0L]+2L] ne 0L then begin
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='GRATINGFILE must be a scalar string; with the name of an exi' + $
             'sting file.'
      goto,error_handler
    endif
    if ~file_test(gratingfile,/regular,/read) then begin
      errmsg='Cannot find the GRATINGFILE file "'+gratingfile+'".'
      goto,error_handler
    endif
  endif ;; s[s[0L]+2L] ne 0L

  s=size(postable)
  if s[s[0L]+2L] ne 0L then begin
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='POSTABLE must, if set, be a string with the name of an exis' + $
             'ting file.'
      goto,error_handler
    endif
    if ~file_test(postable,/regular,/read) then begin
      errmsg='The position table file POSTABLE="'+postable+'" does not exist.'
      goto,error_handler
    endif
  endif else begin
    postable=''
  endelse

  usedeadfibers=0L
  spc=size(deadfibers)
  if spc[spc[0L]+2L] ge 1L then begin
    if spc[spc[0L]+1L] ge 4L and spc[spc[0L]+1L] le 11L then begin
      errmsg='DEADFIBERS must, if set, be an array of integer type.'
      goto,error_handler
    endif

    spc=size(sdeadfibers)
    if spc[spc[0L]+2L] ne n_elements(deadfibers) or $
       spc[spc[0L]+1L] ne 7L then begin
      errmsg='SDEADFIBERS must be an array of string type with as' + $
             'many elements as DEADFIBERS.'
       goto,error_handler
    endif

    usedeadfibers=1L
  endif ;; spc[spc[0L]+2L] ge 1L
  if ~usedeadfibers then begin & deadfibers=1L & sdeadfibers='' & endif

  s=size(kwrdlist)
  if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
    errmsg='KWRDLIST must be set to a scalar string with the name of an e' + $
           'xisting file.'
    goto,error_handler
  endif
  if ~file_test(kwrdlist,/regular,/read) then begin
    errmsg='The fits-header keywords file KWRDLIST="'+kwrdlist+'" does no' + $
           't exist.'
    goto,error_handler
  endif

  useuserpar=0
  if n_elements(userparfile) ne 0L then begin
    s=size(userparfile)
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='USERPARFILE, if set, must be a scalar string.'
      goto,error_handler
    endif
    if userparfile ne '' then begin
      if ~file_test(userparfile,/regular,/read) then begin
        errmsg='The file USERPARFILE "'+userparfile+'" does not exist.'
        goto,error_handler
      endif
      useuserpar=1
    endif
  endif ;; n_elements(userparfile) ne 0L

  sf=size(fwhm)
  if sf[sf[0L]+2L] ne 1L or $
    (sf[sf[0L]+1L] ge 6L and sf[sf[0L]+1L] le 11L) then begin
    errmsg='FWHM must be set to a scalar decimal value; FWHM>0.0.'
    goto,error_handler
  endif
  if fwhm lt 0d0 then begin
    errmsg='FWHM must be set to a scalar decimal value; FWHM>0.0.'
    goto,error_handler
  endif

  if ~n_elements(method) then method='Gaussian'
  sf=size(method)
  if sf[0L] ne 0L or sf[sf[0L]+2L] ne 1L or sf[sf[0L]+1L] ne 7L then begin
    errmsg='METHOD must be set to a scalar string.'
    goto,error_handler
  endif

  ;; Using information in the user parameter file, if it exists:
  xgotstr=0L & ggotstr=0L & igotstr=0L & jgotstr=0L
  gsetupstr=' [default]'
  if useuserpar then begin
    readcol,userparfile,uparname,uparval1,uparval2,format='a,a,a', $
        comment=';',silent=verbose lt 3,delimiter=' '
    if n_elements(uparname) ne 0L then begin
      idx=strpos(strlowcase(uparname),'grating_setup')
      tmp=where(idx ge 0L,count)
      if count ge 1L then begin
        str=strtrim(uparval1[tmp[0L]],2L)
        errmsg='Could not convert wave0="'+str+'" to a decimal value.'
        on_ioerror,error_handler
        wave0=double(str)
        on_ioerror,NULL

        str=strtrim(uparval2[tmp[0L]],2L)
        errmsg='Could not convert wave1="'+str+'" to a decimal value.'
        on_ioerror,error_handler
        wave1=double(str)
        on_ioerror,NULL

        ggotstr=1L
        gsetupstr=', wave.range='+strtrim(string(wave0,format='(f9.3)'),2L)+ $
                  '-'+strtrim(string(wave1,format='(f9.3)'),2L)+ $
                  ' [userparfile]'
      endif ;; count ge 1L

      idx=strpos(strlowcase(uparname),'imagesplitbin')
      tmp=where(idx ge 0L,count)
      if count ge 1L then begin
        str=strtrim(uparval1[tmp[0L]],2L)

        errmsg='Could not convert imagesplitbin="'+str+'" to an integer value.'
        on_ioerror,error_handler
        isplitbin=long(str)
        on_ioerror,NULL

        igotstr=1L
        isplsstr='[userparfile]'
      endif ;; count ge 1L

      idx=strpos(strlowcase(uparname),'imageorder')
      tmp=where(idx ge 0L,count)
      if count ge 1L then begin
        str=strtrim(uparval1[tmp[0L]],2L)

        errmsg='Could not convert imageorder="'+str+'" to an integer value.'
        on_ioerror,error_handler
        imageorder=long(str)
        on_ioerror,NULL

        jgotstr=1L
        jsplsstr='[userparfile]'
      endif ;; count ge 1L
    endif ;; n_elements(uparname) ne 0L
  endif ;; useuserpar

  if ~xgotstr then begin
    xdegstr=~n_elements(xdeg)?'[default]':'[input]'
    lxdeg=~n_elements(xdeg)?3L:xdeg
    s=size(lxdeg)
    if s[s[0L]+2L] ne 1L or $
      (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
      errmsg='XDEG must, if set, be a scalar integer; XDEG>=0.'
      goto,error_handler
    endif
    if lxdeg lt 0L then begin
      errmsg='XDEG must, if set, be a scalar integer; XDEG>=0.'
      goto,error_handler
    endif
  endif

  lydeg=~n_elements(ydeg)?-1L:ydeg
  s=size(lydeg)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='YDEG must, if set, be a scalar integer; YDEG=-1,YDEG>=0.'
    goto,error_handler
  endif
  if lydeg ne -1L and lydeg lt 0L then begin
    errmsg='YDEG must, if set, be a scalar integer; YDEG=-1,YDEG>=0.'
    goto,error_handler
  endif

  if ~n_elements(linewidth) then linewidth=4L
  s=size(linewidth)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='LINEWIDTH must, if set, be a scalar integer; LINEWIDTH>=0.'
    goto,error_handler
  endif
  if linewidth lt 0L then begin
    errmsg='LINEWIDTH must, if set, be a scalar integer; LINEWIDTH>=0.'
    goto,error_handler
  endif

  if ~n_elements(refdist) then refdist=0L
  s=size(refdist)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='REFDIST must, if set, be a scalar integer; REFDIST>=1.'
    goto,error_handler
  endif
  if refdist lt 1L then begin
    errmsg='REFDIST must, if set, be a scalar integer; REFDIST>=1.'
    goto,error_handler
  endif

  if ~n_elements(nrows) then nrows=0L
  s=size(nrows)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='NROWS must, if set, be a scalar integer; NROWS>=0.'
    goto,error_handler
  endif
  if nrows lt 0L then begin
    errmsg='NROWS must, if set, be a scalar integer; NROWS>=0.'
    goto,error_handler
  endif

  if ~n_elements(removecalfibers) then removecalfibers=1L
  sd=size(removecalfibers)
  if sd[sd[0L]+2L] ne 1L or $
    (sd[sd[0L]+1L] ge 4L and sd[sd[0L]+1L] le 11L) then begin
    errmsg='REMOVECALFIBERS is a keyword that, if set, must be a scalar i' + $
           'nteger; 0||1.'
    goto,error_handler
  endif
  if removecalfibers ne 0L and removecalfibers ne 1L then begin
    errmsg='REMOVECALFIBERS is a keyword that, if set, must be a scalar i' + $
           'nteger; 0||1.'
    goto,error_handler
  endif

  residualcut=~n_elements(residualcut)?-1L:residualcut
  s=size(residualcut)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='RESIDUALCUT must, if set, be a decimal scalar; RESIDUALCUT>=0.'
    goto,error_handler
  endif
  if residualcut ne -1L and residualcut lt 0L then begin
    errmsg='RESIDUALCUT must, if set, be a scalar integer; RESIDUALCUT>=0.'
    goto,error_handler
  endif

  if ~n_elements(dflip) then dflip=0L
  s=size(dflip)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='DFLIP must be a scalar integer; DFLIP=0||1.'
    goto,error_handler
  endif
  if dflip ne 0L and dflip ne 1L then begin
    errmsg='DFLIP must be a scalar integer; DFLIP=0||1.'
    goto,error_handler
  endif
  n_order=~dflip?1L:-1L

  if ~keyword_set(track) then track=0L
  s=size(track)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='TRACK is a keyword that, if set, must be a scalar integer; 0||1.'
    goto,error_handler
  endif
  if track ne 0L and track ne 1L then begin
    errmsg='TRACK is a keyword that, if set, must be a scalar integer; 0||1.'
    goto,error_handler
  endif

  if ~n_elements(ofilename) then ofilename='/tmp/tmp.fits'
  tfile=ofilename

  compress=keyword_set(compress)
  dflip=keyword_set(dflip)

  if ~n_elements(dbin) then dbin=1L
  s=size(dbin)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='DBIN must be a scalar integer; DBIN==1||2||3.'
    goto,error_handler
  endif
  if dbin lt 1L or dbin gt 3L then begin
    errmsg='DBIN must be a scalar integer; DBIN==1||2||3.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Setting up colortable-related variables:

  if ~n_elements(cindv) or ~n_elements(bottom) or $
     ~n_elements(cindex) then begin
    ;; Colors are only re-defined if the necessary arrays are not defined:
    p3d_misc_colortable,colortable,bottom=bottom,index=cindex,indv=cindv, $
        colors=colors,/define,verbose=verbose,error=error,debug=debug
    if error ne 0 then return
  endif else begin
    ;; Reading the colors:
    tvlct,red,green,blue,/get
    colors={red:red,green:green,blue:blue}
  endelse

  ;;========================================------------------------------
  ;; Loading the input data:

  if postable ne '' then begin
    p3d_misc_read_postable,postable=postable,rownum=calibspec,id=id, $
        xpos=xpos,ypos=ypos,/calibration,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return

    image=readfits(imagefile[0L],hdr,silent=verbose lt 3,/no_unsigned)
    if nimagefile eq 2L then begin
      image2=readfits(imagefile[1L],hdr2,silent=verbose lt 3,/no_unsigned)
      si2=size(image2)
    endif

    ;; Removing calibration spectra from the spectrum image:
    if removecalfibers then begin
      p3d_wavecal_delete_spec,image,calibspec,origimage,daxis=daxis, $
          topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then return

      if nimagefile eq 2L then begin
        p3d_wavecal_delete_spec,image2,calibspec,origimage2,daxis=daxis, $
            topwid=topwid,logunit=logunit,verbose=verbose,error=error, $
            debug=debug
        if error ne 0 then return
      endif
      if daxis eq 2L then origimage2=transpose(origimage2)
      si2=size(origimage2)
    endif ;; removecalfibers
    if daxis eq 2L then origimage=transpose(origimage)
    si=size(origimage)
  endif else begin ;; postable ne ''
    origimage=readfits(imagefile,hdr,silent=verbose lt 3,/no_unsigned)
    if daxis eq 2L then origimage=transpose(origimage)
    si=size(origimage)
    if nimagefile eq 2L then begin
      origimage2=readfits(imagefile2,hdr2,silent=verbose lt 3,/no_unsigned)
      if daxis eq 2L then origimage2=transpose(origimage2)
      si2=size(origimage2)
    endif
  endelse ;; postable ne ''

  if nimagefile eq 2L then begin
    if si2[1L] ne si[1L] or si2[2L] ne si[2L] then begin
      errmsg='The two image files have different dimensions (['+ $
             strtrim(si[1L],2L)+','+strtrim(si[2L],2L)+'] vs. ['+ $
             strtrim(si2[1L],2L)+','+strtrim(si2[2L],2L)+ $
             ']), this is not allowed!'
      goto,error_handler
    endif
  endif ;; nimagefile eq 2L

  ;; Retrieving the number of pixels in the dispersion direction:
  npix=si[1L]

  ;; Setting the reference spectrum to be the one in the middle of the image:
  refrow=si[2L]/2L-1L
  refrow=p3d_misc_findspec(refrow,si[2L],deadfibers, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug)
  if error ne 0 then return

  ;; For PPAK data cross-talk from an adjacent (now removed) calibration fiber
  ;; may distort the spectrum in image row 183, hence in this case the
  ;; reference spectrum is replaced with image row 190 /JAG:
  if ~(refrow-183L) then refrow=190L

  ;; Likewise setting the reference spectrum bin to be the center one:
  refcol=npix/2L-1L

  ;;========================================------------------------------
  ;; Retrieving CCD binning parameters:

  p3d_misc_binextract,kwrdlist,hdr,xbin=xbin,ybin=ybin, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return

  ;; Reading information about the grating/detector:
  tmp=p3d_misc_get_hdr_kwrd(kwrdlist,'GRAT_ID',topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug)
  if error ne 0 then return
  if tmp ne '-1' then begin
    grating_id=p3d_misc_fitsstrex(hdr,tmp,topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug)
    if error ne 0 then return
  endif ;; tmp ne '-1'

  tmp=p3d_misc_get_hdr_kwrd(kwrdlist,'GRATLPMM',topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug)
  if error ne 0 then return
  if tmp ne '-1' then lpmm=fxpar(hdr,tmp)

  tmp=p3d_misc_get_hdr_kwrd(kwrdlist,'CAMANGLE',topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug)
  if error ne 0 then return
  if tmp ne '-1' then camangle=fxpar(hdr,tmp)

  tmp=p3d_misc_get_hdr_kwrd(kwrdlist,'GROTPOS',topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug)
  if error ne 0 then return
  grotpos=fxpar(hdr,tmp) ;& grotpos=double(grotpos)

  tmp=p3d_misc_get_hdr_kwrd(kwrdlist,'CENWAVE',topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug)
  if error ne 0 then return
  cwave=tmp eq '0'?'0':fxpar(hdr,tmp)

  tmp=p3d_misc_get_hdr_kwrd(kwrdlist,'CENDISP',topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug)
  if error ne 0 then return
  cdisp=tmp eq '0'?'0':strtrim(fxpar(hdr,tmp),2L)

  cdisgroup=p3d_misc_get_hdr_kwrd(kwrdlist,'CENDISGROUP',topwid=topwid, $
      logunit=logunit,verbose=verbose,error=error,debug=debug)
  if error ne 0 then return

  grateq_header=grotpos eq 0d0?0L:1L

  ;;========================================------------------------------
  ;; The following variables have to be defined for each instrument:
  ;;   IMAGE   - the spectrum image, oriented such that the shortest
  ;;             wavelength is located to index 0.
  ;;   LAMBDA  - an array of wavelengths, for use with the image, with the
  ;;             shortest value at index 0.
  ;;   DISP    - the width of every spectrum bin.
  ;;   DDISP   - the mean width of all spectrum bins.
  ;;   REFWAVE - the reference [center] wavelength.
  ;;   REFDISP - the reference [center] spectrum bin width.

  ok=0L & t_order=n_order
  if grateq_header and ~ggotstr then begin

    ;;========================================------------------------------
    ;;========================================------------------------------
    ;; Setting up parameters with instruments which provide information to
    ;; use the grating equation:
    ;;========================================------------------------------
    ;;========================================------------------------------

    if ~n_elements(grating_id) then begin
      errmsg=['[PMAS-style wavelength-range mode] The grating id', $
              'parameter GRAT_ID must be set in the instrument', $
              'keywords file.']
      goto,error_handler
    endif

    ;; Finding out which PMAS grating is used:
    readcol,gratingfile,gratname,gratid,format='a,i',comment=';', $
            silent=verbose lt 3
    idx=where(gratid eq grating_id,count)
    if ~count then begin
      errmsg='[grating equation] The grating ID ['+grating_id+ $
        '] could not be found.'
      goto,error_handler
    endif
    grat_name=strtrim(gratname[idx[0L]],2L)

    if n_elements(lpmm) ne 0L and n_elements(camangle) ne 0L then begin
      ;; SPIRAL IFU:

      lpmm=double(lpmm)
      camangle=double(camangle)
      tmp=p3d_misc_get_hdr_kwrd(kwrdlist,'FOCALLENGTH',topwid=topwid, $
              logunit=logunit,verbose=verbose,error=error,debug=debug)
      if error ne 0 then return
      fl=double(tmp)

      tmp=p3d_misc_get_hdr_kwrd(kwrdlist,'ORDER',topwid=topwid, $
              logunit=logunit,verbose=verbose,error=error,debug=debug)
      if error ne 0 then return
      order=fxpar(hdr,tmp) & order=long(order)

      p3d_wavecal_grating_eq,lpmm,0,grotpos,eta=2L*grotpos-camangle,flen=fl, $
          2048L,order,pix,tlambda,grotoffset=grotoffset,topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then return

    endif else begin ;; n_elements(lpmm) ne 0L and n_elements(camangle) ne 0L
      lpmm=long(strmid(grat_name,1L,strlen(grat_name)-1L))
      alpha=grotpos
      ingrot=alpha

      dpix=(daxis?xbin:ybin)*npix
      p3d_wavecal_grating_eq,lpmm,0,ingrot,grotpos=1b,grotoffset=grotoffset, $
          dpix,t_order,pix,tlambda,/flip,topwid=topwid,logunit=logunit, $
          verbose=verbose,error=error,debug=debug
      if error ne 0 then return
    endelse ;; n_elements(lpmm) ne 0L and n_elements(camangle) ne 0L

    title=', GRATING: '+strtrim(grat_name,2L)+ $
          ', GROTPOS: '+string(format='(f7.2)',grotpos-grotoffset)

    ;; The wavelength array is oriented to have the shortest wavelength in
    ;; index 0:
    doflip=tlambda[n_elements(tlambda)-1L] gt tlambda[0L]?0L:1L
    if doflip then tlambda=rotate(tlambda,2L)

    image=origimage
    if nimagefile eq 2L then image2=temporary(origimage2)

    lambda=congrid(tlambda,npix,/interp)

    disp=lambda[1L:npix-1L]-lambda[0L:npix-2L] 
    ddisp=mean(disp[1L:npix-2L]-disp[0L:npix-3L],/double)

    refwave=lambda[refcol]
    refdisp=disp[refcol]
    n_order=t_order

  endif else if ~ggotstr and $
     cwave ne '0' and cdisp ne '0' and cdisgroup ne '0' then begin
    ;;========================================------------------------------
    ;;========================================------------------------------
    ;; The file header already contains information about the center wavelength
    ;; and the dispersion:
    ;;========================================------------------------------
    ;;========================================------------------------------

     errmsg=['It must be possible to convert the value of the', $
             'CENWAVE parameter to a decimal value. It is not.', $
             '  CENWAVE="'+strtrim(cwave,2L)+'"']
    on_ioerror,format_error
    cwave=double(cwave)
    on_ioerror,NULL

    errmsg=['It must be possible to convert the value of the' + $
            'CENDISGROUP parameter to an integer. It is not.', $
            '  CENDISGROUP="'+strtrim(cdisgroup,2L)+'"']
    on_ioerror,format_error
    cdisgroup=long(cdisgroup)-1L
    on_ioerror,NULL

    cdiss=strsplit(cdisp) & ncdiss=n_elements(cdiss)
    if cdisgroup lt 0L or cdisgroup gt ncdiss then begin
      errmsg=['The header keyword value of CENDISGROUP must satisfy', $
              '0<=CENDISGROUP<='+strtrim(ncdiss,2L)+'. It does not.', $
              '  CENDISGROUP="'+strtrim(cdisgroup+1L,2L)+'"']
      goto,error_handler
    endif
    tmp=strsplit(cdisp,/extract)
    cdiss=tmp[cdisgroup]

    case cdisgroup of
      0: tmp='1st'
      1: tmp='2nd'
      2: tmp='3rd'
      else: tmp=strtrim(cdisgroup+1L,2L)+'th'
    endcase
    errmsg=['It must be possible to convert the value of the CENDISP', $
            'parameter ('+tmp+' group) to a decimal value. It is not.', $
            '  CENDISP['+strtrim(cdisgroup,2L)+']="'+strtrim(cdisp,2L)+'"']
    on_ioerror,format_error
    cdisp=double(cdiss)
    on_ioerror,NULL

    ;; Calculating the necessary quantities:
    ddisp=0d0
    refdisp=cdisp
    refwave=cwave
    image=origimage
    if nimagefile eq 2L then image2=temporary(origimage2)

    wave0=cwave-npix*cdisp/2L
    wave1=cwave+npix*cdisp/2L
    title=' init: '+string(wave0,format='(i6)') + $
                '-'+string(wave1,format='(i6)')

    doflip=0L
  endif else begin ;; grateq_header
    ;;========================================------------------------------
    ;;========================================------------------------------
    ;; Setting up parameters for all instruments, but those providing enough
    ;; information to use the grating equation:
    ;;========================================------------------------------
    ;;========================================------------------------------

    ;; Reading the spectrum setup from the grating file:
    if ~ggotstr then begin

      if ~n_elements(gratingfile) then begin
        errmsg=['[pre-defined wavelength range mode] A grating filename mu' + $
                'st be specified in the instrument parameter file to conti' + $
                'nue.']
        goto,error_handler
      endif ;; ~n_elements(gratingfile)

      ;; Finding out which PMAS grating is used:
      readcol,gratingfile,gratname,wave0val,wave1val,format='a,f,f', $
          comment=';',silent=verbose lt 3

      idx=where(gratname eq grating_id,count)
      if ~count then begin
        errmsg='Cannot find the "'+grating_id+'" entry in the gratingfile' + $
               ' "'+gratingfile+'".'
        goto,error_handler
      endif
      wave0=wave0val[idx[0L]]
      wave1=wave1val[idx[0L]]

      ;; Checking the values of wave0 and wave1:
      if wave0 lt 0.0 then begin
        errmsg='The lower wavelength range wave0='+ $
               string(wave0,format='(f9.3)')+' <0! ['+ $
               p3d_misc_pathify(gratingfile)+']'
        goto,error_handler
      endif
      if wave1 lt 0.0 or wave1 le wave0 then begin
        errmsg='The upper wavelength range wave1='+ $
               string(wave1,format='(f9.3)')+' <0! ['+ $
               p3d_misc_pathify(gratingfile)+']'
        goto,error_handler
      endif
      if wave1 le wave0 then begin
        errmsg='The upper wavelength range wave1='+ $
               string(wave1,format='(f9.3)')+' < wave0='+ $
               string(wave0,format='(f9.3)')+' ! ['+ $
               p3d_misc_pathify(gratingfile)+']'
        goto,error_handler
      endif

    endif ;; ~ggotstr

    ;; Calculating the necessary quantities:
    ddisp=0d0
    refdisp=(wave1-wave0)/npix ;; Wavelength bin width at the center
    refwave=(wave1+wave0)/2L
    image=origimage
    if nimagefile eq 2L then image2=temporary(origimage2)
    title=' init: '+string(wave0,format='(i6)') + $
                '-'+string(wave1,format='(i6)')

    doflip=0L

  endelse ;; grateq_header

  ;; Recalculating DISP and LAMBDA:
  state=ptr_new({npix:npix,disp:dblarr(npix),ddisp:ddisp, $
                refcol:refcol,refdisp:refdisp,refwave:refwave, $
                lambda:dblarr(npix)}) ;; temporary state variable
  p3d_wavecal_dispmask_gui_calc_linearlambda,state
  disp  =(*state).disp
  lambda=(*state).lambda
  ptr_free,state

  plotlambda=dindgen(npix) ;; Used on the x-axis when plotting

  ;;========================================------------------------------
  ;; Combining the images, if there are two...

  if nimagefile eq 2L then begin
    if ~jgotstr then begin
      jsplsstr=~n_elements(imageorder)?'[default]':'[input]'
      if ~n_elements(imageorder) then imageorder=0L
      s=size(imageorder)
      if s[s[0L]+2L] ne 1L or $
        (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
        errmsg='IMAGEORDER must, if set, be a scalar integer; 0||1.'
        goto,error_handler
      endif
    endif ;; ~jgotstr

    if imageorder ne 0L and imageorder ne 1L then begin
      errmsg='IMAGEORDER must, if set, be a scalar integer; 0||1.'
      goto,error_handler
    endif

    if ~igotstr then begin
      isplsstr=~n_elements(isplitbin)?'[default]':'[input]'
      if ~n_elements(isplitbin) then isplitbin=si[1L]/2L
      s=size(isplitbin)
      if s[s[0L]+2L] ne 1L or $
        (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
        errmsg='IMAGESPLITBIN must, if set, be a scalar integer; 0<=IMAGES' + $
               'PLITBIN<=wavelength_bins-1.'
        goto,error_handler
      endif
    endif ;; ~igotstr

    if isplitbin lt 1L or isplitbin gt si[1L] then begin
      errmsg='IMAGESPLITBIN must, if set, be a scalar integer; 1<=IMAGES' + $
             'PLITBIN<=wavelength_bins.'
      goto,error_handler
    endif
    isplitbin-- ;; keeping ISPLITBIN zero-based in the tool

    image=p3d_wavecal_dispmask_gui_imcombine(imagefile,isplitbin,imageorder, $
        daxis)

  endif else begin ;; nimagefile eq 2L
    imageorder=0L
    isplitbin=0L
  endelse ;;  nimagefile eq 2L

  ;;========================================------------------------------
  ;; Collecting a list of calibration lines to use in the mask creation.
  ;; The linelist is held in the variable CALWL:

  llinelist=linelist

  ;; Reading all files:
  on_ioerror,file_error
  for k=0L,n_elements(linelist)-1L do begin
    defstr='Error reading the line list "'+linelist[k]+'".'
    readcol,linelist[k],tmp,format='a',comment=';',silent=verbose lt 3
    calwl=~k?tmp:[calwl,tmp]
  endfor
  calwl=double(calwl)
  on_ioerror,NULL

  ;; Only keeping the lines within the current wavelength range:
  range=max(lambda)-min(lambda)
  pos=where(calwl ge min(lambda)-0.05*range and $
            calwl le max(lambda)+0.05*range,count)
  if ~count then begin
    errmsg='The selected line lists do not contain any lines in the select' + $
        'ed wavelength range.'
    goto,error_handler
  endif
  calwl=calwl[pos]

  ncalwl=size(calwl,/n_elements)
  linepix=dblarr(ncalwl)-1d0
  lines=dblarr(ncalwl,si[2L])


  ;; Finding the pixel at the wavelength which is closest to each line
  ;; in the calibration line list. The result is stored in LINEPIX,
  ;; and also in LINES (using the same pixel for every spectrum):
  if lambda[0L] lt lambda[npix-1L] then begin
    for k=0L,ncalwl-1L do begin
      pos=where(lambda ge calwl[k],count)
      if count ne 0L then linepix[k]=pos[0L]
      lines[k,*]=linepix[k]
    endfor
  endif else begin
    for k=0L,ncalwl-1L do begin
      pos=where(lambda le calwl[k],count)
      if count ne 0L then linepix[k]=pos[0L]
      lines[k,*]=linepix[k]
    endfor
  endelse ;; lambda[0L] lt lambda[npix-1L]
  plotlines=linepix

  resetimage    = image
  if nimagefile eq 2L then resetimage2=image2
  resetnpix     = npix
  resetddisp    = ddisp
  resetrefcol   = refcol
  resetrefrow   = refrow
  resetrefwave  = refwave
  resetrefdisp  = refdisp
  resetplambda  = plotlambda
  resetcalwl    = calwl
  resetisplitbin= isplitbin
  resetimageorder=imageorder


  geo=dblarr(si[2L])

  ;;========================================------------------------------
  ;; Logging:

  msg='Creating a dispersion mask - using a graphical tool.'
  msg=[msg, $
       ' polynomial order='+string(lxdeg,format='(i9)')+' :: used on the' + $
       ' dispersion axis '+xdegstr, $
       ' grating setup'+gsetupstr, $
       '           refcol='+string(refcol+1L,format='(i9)')+ $
       ' :: Reference wavelength bin', $
       '           refrow='+string(refrow+1L,format='(i9)')+ $
       ' :: Reference spectrum', $
       '             dbin='+string(dbin,format='(i9)')+ $
       ' :: Disp.axis rebinning par. (1 [no rebinning], 2 [/2], 3 [/2 unti' + $
       'l screen is fitted])', $
       '             npix='+string(npix,format='(i9)')+ $
       ' :: Number of pixels on the dispersion axis.', $
       '            cdelt='+string(ddisp,format='(f9.3)')+ $
       ' :: Initial dispersion value.', $
       '            crval='+string(lambda[0L],format='(f9.3)')+ $
       ' :: Initial zero-pixel wavelength.']
  if nimagefile eq 2L then begin
    msg=[msg,'    imagesplitbin='+string(isplitbin+1L,format='(i9)')+' :: '+ $
         isplsstr, $
         '       imageorder='+string(imageorder,format='(i9)')+' :: '+ $
         jsplsstr]
  endif ;; nimagefile eq 2L
  msg=[msg,'  Dispersion axis: '+strtrim(daxis,2L)+'.']
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  ;;========================================------------------------------
  ;; Calculating a default widget geometry:

  screensize=get_screen_size()

  ;; The default x-size should be smaller than the number of pixels in the x-
  ;; direction, and is divided by factors of 2 in order to fit the dispersion
  ;; direction pixels:
  ;; Note! The minimum width is set before realization of the widget hierarchy.

  case dbin of
    1L: begin
      maxxsize=screensize[0L]-100L
      wxsize=si[1L]
    end ;; 1L
    2L: begin
      minxsize=si[1L]/2L
      wxsize=(screensize[0L]-100L)<minxsize
      maxxsize=wxsize
    end ;; 2L
    else: begin
      minxsize=si[1L]
      while minxsize gt screensize[0L]-10L do minxsize/=2L

      wxsize=(screensize[0L]-100L)<minxsize
      maxxsize=wxsize
    end ;; else
  endcase ;; dbin

  ;; The default y-size is trickier. The value is changed if it does
  ;; not fit on the screen below, before realization:
  if screensize[1L] le 600L then begin
    maxysize=150L
    ploysize=180L
  endif else if screensize[1L] le 800L then begin
    maxysize=200L
    ploysize=250L
  endif else begin
    maxysize=400L
    ploysize=350L
  endelse
  minpysize=150L
  wysize=si[2L]
  maxysize=maxysize<wysize

  ;;========================================------------------------------
  ;; Setting up the widget tool:

  loadct,get_names=names
  names=['Sauron',names]
  nnames=n_elements(names)
  values=['100%','99.5%','99%','98%','97%','95%','90%','80%','70%']
  wid={plo:0L,wav:0L,ima:0L,lstat:0L,dima:0L,dwav:0L,dplo:0L, $
       tcomi:0L,tcoma:0L,trefr:0L,trefc:0L, $
       tdisw:0L,tdisp:0L,tdisc:0L,tdiso:0L,tpolo:0L, $
       bbase:0L,bcolm:0L,bcolq:0L,bcolw:lonarr(nnames), $
       bcolo:0L,bsico:lonarr(n_elements(values)), $
       binvm:0L,bsrem:0L,bsrep:0L,bsavm:0L,bsave:0L, $
       bmatm:0L,bfitm:0L,bmatl:0L,bfitl:0L,bcmam:0L,bcmas:0L, $
       bcdim:0L,bcdis:0L,bcdin:0L,bcdir:0L,bcdmn:0L, $
       bmatn:0L,bmatc:0L,bshim:0L,bshif:0L,bcmmd:0L, $
       bdlim:0L,bdlin:0L,bconp:0L,brmas:0L,brall:0L, $
       bscom:0L,bscon:0L,brell:0L,bsavl:0L,bview:0L, $
       bfitp:0L,bplot:0L,bexit:0L,bmact:0L,bvfit:0L, $
       bpfit:0L,bffit:0L,bffsl:0L,bfftx:0L,bffdc:0L,bffic:0L,bffms:0L, $
       bicit:0L,bicsl:0L,bictx:0L,bicdc:0L,bicic:0L,bicms:0L}

  title_='p3d: create a dispersion mask for '
  title_+=file_basename(imagefile[0L])
  if nimagefile eq 2L then title_+='+'+file_basename(imagefile[1])
  title_+=title
  wid.bbase=widget_base(title=title_,/column,/base_align_center, $
      group_leader=topwid,space=2L,xpad=0L,ypad=0L,mbar=mbbarwid, $
      resource_name='p3d') ;;,/tlb_size_events)

  ;;===========
  ;; File menu:

  mfilewid=widget_button(mbbarwid,value='File',/menu, $
      event_pro='p3d_wavecal_dispmask_gui_event',uname='With the options o' + $
      'f this menu you can save data of this tool.',tracking_events=track)

  ;; Line list buttons:
  wid.brell=widget_button(mfilewid,value='Read line list(s)', $
      uvalue='readlinelist',tracking_events=track,uname='Click in order to' + $
      ' select one (or several) alternative line lists to use.')
  wid.bsavl=widget_button(mfilewid,value='Save the current line list', $
      uvalue='savelinelist',tracking_events=track,uname='Click in order to' + $
      ' save the currently used line list to a text file.')

  wid.bview=widget_button(mfilewid,value='View the current line list', $
      uvalue='viewlinelist',tracking_events=track,uname='Click in order to' + $
      ' view the currently used line list.')


  ;; Reset buttons:
  wid.brmas=widget_button(mfilewid,value='Reset',uvalue='reset', $
      /separator,uname='Click in order to reset changes', $
      accelerator='Ctrl+R',tracking_events=track)
  wid.brall=widget_button(mfilewid,value='Undo all changes', $
      uvalue='resetall',uname='Click in order to restore the tool to its s' + $
      'tate when it was first started.',accelerator='Ctrl+U', $
      tracking_events=track)
  wid.bexit=widget_button(mfilewid,value='Exit',uvalue='exit',/separator, $
      event_pro='p3d_wavecal_dispmask_gui_event',uname='Click in order to ' + $
      'exit this tool without saving any changes.',accelerator='Ctrl+Q', $
      tracking_events=track)


  ;;===========
  ;; View menu:

  mviewwid=widget_button(mbbarwid,value='View',/menu, $
      event_pro='p3d_wavecal_dispmask_gui_event',uname='With the options o' + $
      'f this menu you can change display settings.',tracking_events=track)

  ;; Color histogram:
  for i=0L,n_elements(values)-1L do begin
    tmp=strmid(values[i],0L,strpos(values[i],'%'))
    str='Color map: use '+values[i]+' of the pixel values'
    wid.bsico[i]=widget_button(mviewwid,value=str, $
        uvalue='imagecolordroplist_'+tmp,uname='Click to use '+values[i]+ $
        ' of the pixel value range when calculating lower and upper limits' + $
        ' of the color map.',tracking_events=track, $
        accelerator='F'+strtrim(i+1L,2L))
  endfor

  wid.binvm=widget_button(mviewwid,value='Invert Colors', $
      uvalue='colorinvert',tracking_events=track,uname='Click in order to ' + $
      'invert the colors in the image, and also flip the min-max range in ' + $
      'the spectrum plot.')

  ;; Color tables:
  wid.bcolm=widget_button(mviewwid,/menu,/separator, $
      value='Select colortable',uvalue='colortablemenu', $
      uname='Click to choose a colortable',tracking_events=track)

  wid.bcolo=widget_button(wid.bcolm,value='xloadct',uvalue='colortable',$
      uname='Click to select a colortable interactively using XLOADCT.', $
      tracking_events=track)

  nbuttons=nnames/12 & mbuttons=nnames mod 12
  imax=nbuttons+(mbuttons gt 0L?1L:0L)
  iimax=lonarr(imax)+11L
  iimax[imax-1L]=mbuttons-1L

  k=0L
  for i=0L,imax-1L do begin
    wid.bcolq=widget_button(wid.bcolm,/menu,separator=~i, $
        value='colortables '+strtrim(i+1L,2L)+'/'+strtrim(imax,2L), $
        uvalue='colortablemenu',uname='Click to choose a colortable in thi' + $
        's group',tracking_events=track)

    for j=0L,iimax[i] do begin
      wid.bcolw[k]=widget_button(wid.bcolq,value=names[k], $
          uvalue='colortablesel_'+strtrim(k-1L,2L),uname='Click to select '+ $
          'the '+names[k]+' colortable.',tracking_events=track)
      k++
    endfor ;; j=0L,iimax[i]
  endfor ;; i=0L,imax-1L

  ;;=============
  ;; Action menu:

  wid.bmact=widget_button(mbbarwid,value='Action',/menu, $
      event_pro='p3d_wavecal_dispmask_gui_event',uname='With the options o' + $
      'f this menu you can perform actions on the mask.',tracking_events=track)

  wid.bmatn=widget_button(wid.bmact,value='Match curvature', $
      uvalue='matchcurvature',tracking_events=track,uname='Click on a brig' + $
      'ht line to match the mask with the curvature of the spectrum image.', $
      accelerator='Ctrl+1')
  wid.bshim=widget_button(wid.bmact,value='Shift mask', $
      uvalue='waverangeshift',tracking_events=track,uname='Drag the mask o' + $
      'ver the spectrum image to align emission line centroids.', $
      accelerator='Ctrl+2')
  wid.bdlim=widget_button(wid.bmact,value='Delete lines', $
      uvalue='maskremoveline',tracking_events=track,uname='Click on a line' + $
      ' to remove it from the mask.',accelerator='Ctrl+3')
  wid.bmatm=widget_button(wid.bmact,value='Match centroids', $
      uvalue='matchmasktocentroids',tracking_events=track,uname='Match the' + $
      ' mask to line centroids in the spectrum image (can be done repeated' + $
      'ly).',accelerator='Ctrl+4')
  wid.bfitm=widget_button(wid.bmact,value='Fit centroids',sensitive=0L, $
      uvalue='fitcentroids',tracking_events=track,uname='Fit the entries o' + $
      'f the mask to line centroids in the spectrum image more accurately.', $
      accelerator='Ctrl+5')
  wid.bcmam=widget_button(wid.bmact,value='Create dispersion mask', $
      sensitive=0L,uvalue='maskcreate',uname='Click to fit a polynomial to' + $
      ' lines in the mask [wavelength(pixel)].', $
      tracking_events=track,accelerator='Ctrl+6')
  wid.bsavm=widget_button(wid.bmact,value='Save mask and exit',/separator, $
      uvalue='savemaskandexit',accelerator='Ctrl+W',sensitive=0L, $
      uname='Click to save the calculated dispersion mask and exit this GU' + $
      'I.',tracking_events=track)

  ;;=============
  ;; Options menu:

  bmoptwid=widget_button(mbbarwid,value='Options',/menu, $
      event_pro='p3d_wavecal_dispmask_gui_event',uname='With this menu you' + $
      'can select the reference spectrum/bin and subtract the background.', $
      tracking_events=track)

  wid.bscom=widget_button(bmoptwid,value='Subtract Background', $
      uvalue='continuumsubtract',uname='Click to subtract the background (' + $
      'continuum) in the spectrum image.',tracking_events=track)
  wid.bsrem=widget_button(bmoptwid,value='Set Reference Position', $
      uvalue='imagerefchange',tracking_events=track,uname='Click in the sp' + $
      'ectrum image to select another reference position.')
  wid.bcdim=widget_button(bmoptwid,value='Change Dispersion', $
      uvalue='dispersioninteractivechange',tracking_events=track, $
      uname='Click in the spectrum image to interactively stretch the cons' + $
      'tant dispersion interactively.')
  wid.bcdin=widget_button(bmoptwid,value='Reset Disp',sensitive=0L, $
      tracking_events=track,uvalue='dispersionvaluereset', $
      uname='Click to reset the dispersion value to the entry value.')


  ;;===========
  ;; The rest:

  basesub=widget_base(wid.bbase,/column,/base_align_center, $
      space=2L,xpad=0L,ypad=0L,event_pro='p3d_wavecal_dispmask_gui_event')

  imscroll=0L
  uname1='[Spectrum Image] This field shows the extracted spectra. Superpo' + $
         'sed vedrtical lines belong to the line mask. The ref. spectrum i' + $
         's indicated with a red disk.'
  uname2='[Wavelength Range] This field shows the wavelength range of the ' + $
         'reference spectrum in the spectrum image.'
  uname3='[Spectrum plot] This field shows the ref. spectrum. White crosse' + $
         's indicate the position of lines in the mask.'
  draww={mode:0L,refposchange:1L,maskcurvature:2L,dispchange:3L, $
         waverangeshift:4L,maskremoveline:5L,dispmanual:6L, $
         active:0b,pressed:0L,tempi:lonarr(10L),tempf:fltarr(10L), $
         storage:ptr_new()}

  if si[2L] gt maxysize or wxsize gt maxxsize then begin
    btmpwid=widget_base(basesub,xpad=0L,ypad=0L,/align_left,xoffset=2L) ;
    wid.dima=widget_draw(btmpwid,retain=2L,xsize=wxsize,ysize=wysize,/scroll, $
        x_scroll_size=maxxsize,y_scroll_size=maxysize, $
        uname=uname1,uvalue='spectrumimage',/button_events,/motion_events, $
        tracking_events=track,/viewport_events)
    tmp=widget_info(wid.dima,/geometry)
    diff=tmp.scr_xsize-tmp.xsize

    btmpwid=widget_base(basesub,xpad=0L,ypad=0L,/align_left,xoffset=2L)
    wid.dwav=widget_draw(btmpwid,xsize=maxxsize,ysize=15L,uvalue='immotion', $
        retain=2L,uname=uname2,/frame,tracking_events=track,xoffset=2L)
    btmpwid=widget_base(basesub,xpad=0L,ypad=0L,/align_left,xoffset=2L)
    wid.dplo=widget_draw(btmpwid,xsize=maxxsize,ysize=ploysize, $
        uvalue='spectrumplot',/motion_events,/button_events,retain=2L, $
        /frame,uname=uname3,tracking_events=track,xoffset=2L)
    scrollbars=1L
  endif else begin
    wid.dima=widget_draw(basesub,xsize=wxsize,ysize=wysize,retain=2L, $
        /frame,uname=uname1,uvalue='spectrumimage',/button_events, $
        /motion_events,tracking_events=track)
    wid.dwav=widget_draw(basesub,xsize=wxsize,ysize=15L,uvalue='immotion', $
        retain=2L,/frame,uname=uname2,tracking_events=track)
    wid.dplo=widget_draw(basesub,xsize=wxsize,ysize=ploysize, $
        uvalue='spectrumplot',/motion_events,/button_events,retain=2L,/frame, $
        uname=uname3,tracking_events=track)
    scrollbars=0L
  endelse

  wid.lstat=widget_label(basesub,/sunken_frame,resource_name='field', $
      uname='This field shows what is going on and provides hints on diffe' + $
      'rent fields in this GUI.',tracking_events=track,value=' ')

  ;;===========
  ;; Define a tab widgets base:

  bconpwid=widget_base(wid.bbase,/row,/base_align_center, $
      space=10L,xpad=0L,ypad=0L,event_pro='p3d_wavecal_dispmask_gui_event')

  wid.bconp=widget_tab(bconpwid,uvalue='controltab',location=0L, $
      /align_center,xoffset=0L,yoffset=0L)

  ;; Action buttons:
  buttbase=widget_base(wid.bconp,/row,/base_align_center,space=5L, $
      title='Line Mask Action',xpad=2L,ypad=2L,tracking_events=track, $
      uname='Use the actions on this row in consecutive order from the left.')

  wid.bmatc=widget_button(buttbase,value='Match curvature', $
      uvalue='matchcurvature',uname='Click on a bright line to match the m' + $
      'ask with the curvature of the spectrum image.', $
      tracking_events=track,resource_name='field')
  wid.bshif=widget_button(buttbase,value='Shift mask', $
      tracking_events=track,uvalue='waverangeshift',uname='Drag the mask o' + $
      'ver the spectrum image to align emission line centroids.', $
      resource_name='field')
  wid.bdlin=widget_button(buttbase,value='Delete lines', $
      uvalue='maskremoveline',uname='Click on a line to remove it from the' + $
      ' mask.',tracking_events=track,resource_name='field')
  wid.bmatl=widget_button(buttbase,value='Match centroids', $
      uvalue='matchmasktocentroids',uname='Match the mask to line centroid' + $
      's in the spectrum image (can be done repeatedly).', $
      tracking_events=track,resource_name='field')
  wid.bfitl=widget_button(buttbase,value='Fit centroids', $
      uvalue='fitcentroids',uname='Fit the entries of the mask to line cen' + $
      'troids in the spectrum image more accurately.',tracking_events=track, $
      sensitive=0L,resource_name='field')
  wid.bcmas=widget_button(buttbase,value='Create dispersion mask', $
      uvalue='maskcreate',uname='Click to fit a polynomial to lines in the' + $
      ' mask [wavelength(pixel)].',tracking_events=track,sensitive=0L, $
      resource_name='field')
  wid.bsave=widget_button(buttbase,value="Save mask and exit",sensitive=0L, $
      uvalue='savemaskandexit',resource_name='field',uname='Click to save ' + $
      'the calculated dispersion mask and exit this GUI.', $
      tracking_events=track)

  ;;===============================------------------
  ;; Dispersion-related properties:

  ;; Options:  Reference Position and Dispersion Settings:
  refdibas=widget_base(wid.bconp,/row,/frame,/base_align_center,space=5L, $
      title='Dispersion',xpad=2L,ypad=0L,tracking_events=track, $
      uname='Use the widgets in this row to change the dispersion values.')

  batmpwid=widget_base(refdibas,/row,space=5L,xpad=0L,ypad=0L, $
      /base_align_center)
  wid.bcdis=widget_button(batmpwid,value='Change Disp.', $
      uvalue='dispersioninteractivechange',tracking_events=track, $
      uname='Click in the spectrum image to interactively stretch the cons' + $
      'tant dispersion interactively.')
  wid.bcdir=widget_button(batmpwid,value='Reset Disp',sensitive=0L, $
      tracking_events=track,uvalue='dispersionvaluereset', $
      uname='Click to reset the dispersion value to the entry value.')
  wid.bcdmn=widget_button(batmpwid,value='Manual Disp.', $
      uvalue='dispersionmanualset',tracking_events=track, $
      uname='Click to define an initial dispersion solution manually in th' + $
      'e spectrum plot.')

  refdidis=widget_base(refdibas,/row,space=0L,xpad=0L,ypad=0L, $
      /base_align_center,tracking_events=track,uname='Use the widgets in t' + $
      'his row to change dispersion-related parameters.')

  str=[' Disp.: ','Delta: ','Center-bin wavelength:']
  refdila2=widget_label(refdidis,value=str[0L])
  wid.tdisp=widget_text(refdidis,value=strtrim(refdisp,2L),/editable, $
      uvalue='refwavedisp_1',xsize=12L,uname='Enter a value that will be u' + $
      'sed as the dispersion (width of spectrum bin) at the reference posi' + $
      'tion.',tracking_events=track,resource_name='field',/kbrd_focus_events)

  refdicha=widget_base(refdibas,/row,space=0L,xpad=0L,ypad=0L, $
      /base_align_center)
  refdila3=widget_label(refdicha,value=str[1L])
  wid.tdisc=widget_text(refdicha,value=strtrim(ddisp,2L),/editable, $
      uvalue='refwavedisp_2',xsize=12L,uname='Enter a value that will be u' + $
      'sed as the differential change of the dispersion at the reference p' + $
      'osition.',tracking_events=track,resource_name='field', $
      /kbrd_focus_events)

  refdiwav=widget_base(refdibas,/row,space=0L,xpad=0L,ypad=0L, $
      /base_align_center)
  refdila1=widget_label(refdiwav,value=str[2L])
  wid.tdisw=widget_text(refdiwav,value=strtrim(refwave,2L),/editable, $
      uvalue='refwavedisp_0',xsize=12L,uname='Enter a value that will be us'+ $
      'ed as the wavelength at the reference position.', $
      tracking_events=track,resource_name='field',/kbrd_focus_events)

  ;;===============================------------------
  ;; Fit dispersion solution 

  wid.bpfit=widget_base(wid.bconp,/row,/base_align_center,space=5L,$
      title='Check the current mask fit',xpad=2L,ypad=0L, $
      tracking_events=track,uname='Use the buttons in this row to check th' + $
      'e fit of the -current- mask (best applied after matching centroids).')

  ltempwid=widget_label(wid.bpfit,value='These fits relate to -current- ma' + $
      'sk positions (possibly unrelated to the data)')
  batmpwid=widget_base(wid.bpfit,/row,space=0L,xpad=0L,ypad=0L,/grid_layout)
  wid.bfitp=widget_button(wid.bpfit,value='  Fit  ', $
      uvalue='fitpolynomialplot',uname='Click to fit the -current- mask po' + $
      'sitions [wavelength(pixel)] with a polynomial.',tracking_events=track)
  wid.bplot=widget_button(wid.bpfit,value=' Residuals ', $
      tracking_events=track,uvalue='fitshowresiduals',uname='Click to show' + $
      ' residuals of the polynomial fit to the -current- mask positions.')


  ;; Checking the final fit of the dispersion mask:
  wid.bffit=widget_base(wid.bconp,/row,/base_align_center,space=5L, $
      title='Check the final fit',xpad=2L,ypad=0L,sensitive=0L, $
      tracking_events=track,uname='Use the widgets in this row to examine ' + $
      'the quality of the created dispersion mask.')

  blfile=filepath(subdir=['resource','bitmaps'],'shift_left.bmp')
  brfile=filepath(subdir=['resource','bitmaps'],'shift_right.bmp')
  bufile=filepath(subdir=['resource','bitmaps'],'shift_up.bmp')


  wid.bfftx=widget_text(wid.bffit,value=strtrim(refrow+1L,2L),/editable, $
      xsize=5L,uvalue='fittotalspectrum_1',tracking_events=track, $
      uname='Enter a number of a spectrum to show a fit for.', $
      resource_name='field',/kbrd_focus_events)

  batmpwid=widget_base(wid.bffit,/row,space=0L,xpad=0L,ypad=0L, $
      /base_align_center)
  wid.bffdc=widget_button(batmpwid,value=blfile,uvalue='fittotaldecrincr_0', $
      uname='Click to decrement the spectrum number.', $
          tracking_events=track,/bitmap)
  wid.bffsl=widget_slider(batmpwid,/suppress_value,/drag,xsize=500L, $
      uvalue='fittotalspectrum_0',minimum=0L,maximum=si[2L]-1L, $
      tracking_events=track,uname='Drag the slider to show polynomial fits' + $
      ' and residuals for other spectra.')
  wid.bffic=widget_button(batmpwid,value=brfile,uvalue='fittotaldecrincr_1', $
      uname='Click to increment the spectrum number.', $
      tracking_events=track,/bitmap)
  wid.bffms=widget_button(wid.bffit,value=bufile,uvalue='fittotalmaxspec', $
      uname='Click to view the spectrum with the maximum residual (avoidin' + $
      'g dead and unused fibers) :: ',tracking_events=track,/bitmap)

  ;; View - range of values:
  colorbas=widget_base(wid.bconp,/row,/base_align_center,space=5L, $
      title=' View ',xpad=2L,ypad=0L,tracking_events=track, $
      uname='Use the widgets in this row to control the appearance of the ' + $
      'spectrum image.')

  wid.bscon=widget_button(colorbas,value='Subtract the background', $
      uvalue='continuumsubtract',uname='Click to subtract the background (' + $
      'continuum) in the spectrum image.',tracking_events=track)

  wid.tcomi=widget_text(colorbas,value=' ',/editable,xsize=10L, $
      uvalue='imagecolorminmax_0',uname='This is the minimum pixel value t' + $
      'hat is used in the scaling of the color map of the spectrum image a' + $
      'nd reference spectrum.',tracking_events=track,resource_name='field', $
      /kbrd_focus_events)
  wid.tcoma=widget_text(colorbas,value=' ',/editable,xsize=10L, $
      uvalue='imagecolorminmax_1',uname='This is the maximum pixel value t' + $
      'hat is used in the scaling of the color map of the spectrum image a' + $
      'nd the reference spectrum.',tracking_events=track, $
      resource_name='field',/kbrd_focus_events)

  values=['user','100%','99.5%','99%','98%','97%','95%','90%','80%','70%']
  wid.bcmmd=widget_droplist(colorbas,value=values, $
      uvalue='imagecolordroplist',uname='Select the range of values which ' + $
      'are used when determining the minimum and maximum pixel values to p' + $
      'lot.',tracking_events=track)

  wid.bvfit=widget_button(colorbas,value='Hide all centering fits ', $
      uvalue='viewfits',tracking_events=track,uname='Click to switch betwe' + $
      'en hiding all, viewing the ref. spectrum, and viewing all spectra w' + $
      'hen fitting center positions.', $
      sensitive=strlowcase(method) eq 'gaussian')

  ;;===============================------------------
  ;; Reference point in the image:

  refpobas=widget_base(wid.bconp,/row,/frame,/base_align_center, $
      space=5L,xpad=2L,ypad=0L,title='Reference spectrum/bin', $
      tracking_events=track,uname='Use the widgets in this row to select a' + $
      'nother reference spectrum and reference wavelength bin.')

  bifile=filepath(subdir=['resource','bitmaps'],'scale_active.bmp')
  wid.bsrep=widget_button(refpobas,value=bifile,uvalue='imagerefchange', $
      tracking_events=track,uname='Click in the spectrum image to select a' + $
      'nother reference position.',/bitmap)

  str=['Reference spectrum:','Reference bin:']
  refporow=widget_base(refpobas,/row,space=0L,xpad=0L,ypad=0L, $
      /base_align_center)
  refpola1=widget_label(refporow,value=str[0L])
  wid.trefr=widget_text(refporow,value=strtrim(refrow+1L,2L), $
      uvalue='imagerefrowcol_0',xsize=5L,/editable,tracking_events=track, $
      uname='Enter a value that will be used as the reference spectrum in t'+ $
      'he calculations, and shown in the spectrum plot.', $
      resource_name='field',/kbrd_focus_events)

  refpocol=widget_base(refpobas,/row,space=0L,xpad=0L,ypad=0L, $
      /base_align_center)
  refpola2=widget_label(refpocol,value=str[1L])
  wid.trefc=widget_text(refpocol,value=strtrim(refcol+1L,2L), $
      uvalue='imagerefrowcol_1',xsize=5L,/editable,tracking_events=track, $
      uname='Enter a value that will be used as the reference wavelength bi'+ $
      'n in the calculations.',resource_name='field',/kbrd_focus_events)

  ;;===============================------------------
  ;; Fitting polynomial order:

  batmpwid=widget_base(bconpwid,/row,space=0L,xpad=0L,ypad=0L, $
      /base_align_center)
  fitpola3=widget_label(batmpwid,value='Pol. Order:')
  wid.tpolo=widget_text(batmpwid,value=strtrim(lxdeg,2L),/editable, $
      xsize=3L,uvalue='fitpolynomialorder',tracking_events=track, $
      uname='Enter the polynomial order that is used when creating the dis' + $
      'persion mask.',resource_name='field',/kbrd_focus_events)

  ;;===============================------------------
  ;; Image combination setup

  if nimagefile eq 2L then begin
    wid.bicit=widget_base(wid.bconp,/row,/base_align_center,space=5L, $
        title='Imcombine',xpad=2L,ypad=0L,sensitive=1L)

    wid.bictx=widget_text(wid.bicit,value=strtrim(isplitbin+1L,2L),/editable, $
        xsize=5L,uvalue='imagesplitbin_1',tracking_events=track, $
        uname='Enter a spectrum bin number to use as a separating point' + $
        ' between the two images.',resource_name='field',/kbrd_focus_events)

    batmpwid=widget_base(wid.bicit,/row,space=0L,xpad=0L,ypad=0L, $
        /base_align_center)
    wid.bicdc=widget_button(batmpwid,value=blfile, $
        uvalue='imagesplitbindecrincr_0',tracking_events=track,/bitmap, $
        uname='Click to decrement the spectrum wavelength bin used.')
    wid.bicsl=widget_slider(batmpwid,/suppress_value,/drag,xsize=500L, $
        uvalue='imagesplitbin_0',minimum=1L,maximum=si[1L]-1L, $
        tracking_events=track,uname='Drag the slider to select another bin' + $
        ' where the two images are combined.')
    wid.bicic=widget_button(batmpwid,value=brfile, $
        uvalue='imagesplitbindecrincr_1',uname='Click to increment the spe' + $
        'ctrum wavelength bin used.',tracking_events=track,/bitmap)
    tmp=' '+strtrim(imageorder,2L)+' '
    wid.bicms=widget_button(wid.bicit,value=tmp,uvalue='imageorder', $
        uname='Click to change which of the two images is used starting at' + $
        ' wavelength bin 0. ',tracking_events=track)
  endif ;; nimagefile eq 2L

; if grateq_header then begin
;  refpoord=widget_base(refposub,/row,space=0L,xpad=0L,ypad=0L)
;  refpola3=widget_label(refpoord,value='[PMAS] Order:',/align_right,xsize=90L)
;  wid.tdiso=widget_text(refpoord,value=strtrim(n_order,2L), $
;      uvalue='spectrumorder',/editable,xsize=3L,tracking_events=track, $
;      uname='[PMAS] Enter a value that will be used as the spectrum orde' + $
;      'r when calculating the wavelength range.')
; endif

  ;;========================================------------------------------
  ;; Realizing the tool, and getting window indices for the draw widgets:

  widget_control,wid.bbase,/realize
;  widget_control,wid.bbase,update=0L
  widget_control,wid.dima,get_value=tmp & wid.ima=tmp
  widget_control,wid.dwav,get_value=tmp & wid.wav=tmp
  widget_control,wid.dplo,get_value=tmp & wid.plo=tmp
  widget_control,wid.bfitm,sensitive=0L ;; Fit centroids
  widget_control,wid.bfitl,sensitive=0L ;; -"-
  widget_control,wid.bcmas,sensitive=0L ;; Create disp.mask
  widget_control,wid.bcmam,sensitive=0L ;; -"-
  widget_control,wid.bffsl,set_value=refrow+1L
  widget_control,wid.bffsl,set_slider_max=si[2L]
  widget_control,wid.bffsl,set_slider_min=1L
  if nimagefile eq 2L then begin
    widget_control,wid.bicsl,set_value=isplitbin+1L
    widget_control,wid.bicsl,set_slider_max=si[1L]
    widget_control,wid.bicsl,set_slider_min=1L
  endif
  widget_control,wid.bcmmd,set_droplist_select=6L ;; Setting default to 95% 

  ;;========================================------------------------------
  ;; Re-sizing the draw widgets (so that they use the minimum size provided):

  wbass=widget_info(basesub,/geometry)
  wdima=widget_info(wid.dima,/geometry)
  scrlw=wdima.scr_xsize-wdima.xsize ;; calculating the scroll bar width

  available_x0=wbass.xsize-2L*wbass.xpad-2L*wbass.margin-2L
  minxsize=available_x0

  maxxsize=available_x0-scrlw

  ;;========================================------------------------------
  ;; Checking if there is more space available to show more lines in the
  ;; spectrum image:

  wbase=widget_info(wid.bbase,/geometry)
  available_y0=screensize[1L]-(wbase.ysize+60L)
  minysize=50L ;; very small...
  ysize=(wdima.ysize+available_y0)<wysize
  widget_control,wid.dima,ysize=ysize

  ;; Re-sizing the status line widget:
  wbass=widget_info(basesub,/geometry)
;  tmpdrw=widget_info(dplowid,/geometry)
  labwidth=wbass.xsize-2L*wbass.xpad
  widget_control,wid.lstat,xsize=labwidth

  ;;========================================------------------------------
  ;; Setting tvimage, the image shown in the draw widget:

  tvpix=wxsize
  tvbin=double(npix)/tvpix & relbin=1d0/tvbin
  tvimage=congrid(image,tvpix,si[2L],/cubic)
  resettvimage=tvimage
  resettvpix  =tvpix

  ypmin=20L
  ypmax=ploysize-5L

  xoffset=(tvpix-wdima.xsize)/2L
  widget_control,wid.dima,get_draw_view=dview
  widget_control,wid.dima,set_draw_view=[xoffset,dview[1L]]
  xrange=([0d0,maxxsize]+xoffset)*tvbin

  ;;========================================------------------------------
  ;; Storing the widget size in a state structure:

  tmp=widget_info(wid.bbase,/geometry)
  base_size=[tmp.xsize,tmp.ysize]

;  widget_control,wid.bbase,update=1L

  saved_=ptr_new(0b)

  ;; Saving information about the widget in a structure:
  state={image:ptr_new(image),origimage:origimage,si:size(image), $
         imageorder:imageorder,isplitbin:isplitbin, $
         rimageorder:resetimageorder,risplitbin:resetisplitbin, $
         residualcut:residualcut,npix:npix,rnpix:resetnpix, $
         xbin:xbin,ybin:ybin,fwhm:fwhm,method:method,cfitm:0L, $
         tvimage:ptr_new(tvimage),tvpix:tvpix,tvbin:tvbin,stvi:size(tvimage), $
         lambda:lambda,disp:disp,ddisp:ddisp,geo:geo,ceo:geo,crf:0d0, $
         lines:ptr_new(lines),linepix:ptr_new(linepix),slines:size(lines), $
         calwl:ptr_new(calwl),ncalwl:ncalwl, $
         plotlines:ptr_new(plotlines),plotlambda:plotlambda, $
         refcol:refcol,refrow:refrow,refwave:refwave,refdisp:refdisp, $
         crefwave:refwave,crefdisp:refdisp,cshift:0d0, $
         linewidth:linewidth,refdist:refdist,nrows:nrows, $
         rimage:ptr_new(resetimage), $
         rddisp:resetddisp,rcalwl:resetcalwl, $
         rtvimage:ptr_new(resettvimage),rtvpix:resettvpix, $
         rrefcol:resetrefcol,rrefrow:resetrefrow,rrefwave:resetrefwave, $
         rrefdisp:resetrefdisp,rplambda:resetplambda, $
         oldimage:ptr_new(image),oldlines:ptr_new(lines), $
         oldtvimage:ptr_new(tvimage),oldlambda:lambda, $
         olddisp:disp,oldddisp:ddisp, $
         xrange:xrange,xoffset:xoffset, $
         winxsize:maxxsize,winysize:maxysize, $
         ypmin:ypmin,ypmax:ypmax, $
         lxdeg:lxdeg,lydeg:lydeg,logunit:logunit, $
         monopressed:0,showline:-1L,track:track, $
         fitpar:ptr_new(),chisq:ptr_new(),cimlinep:ptr_new(), $
         wid:wid,top:wid.bbase,lltid:0L, $
         base_size:base_size,minxsize:minxsize,maxysize:maxysize, $
         minysize:minysize,scrollbars:scrollbars, $
         doflip:doflip,kwrdlist:kwrdlist,n_order:n_order, $
         imagename:imagefile, $
         deadfibers:deadfibers,sdeadfibers:sdeadfibers, $
         usedeadfibers:usedeadfibers,compress:compress, $
         tfile:tfile,linelist:ptr_new(llinelist),created:0b, $
         calibspec:calibspec,daxis:daxis,saved:saved_, $
         xlogfile:linelist,draww:draww,slval:refrow,slvam:0L, $
         clr:colors,rclr:colors,colmin:0d0,colmax:0d0,cmanual:0L, $
         colortable:colortable,cbottom:bottom,cindex:cindex,cindv:cindv, $
         maskdone:0L,ok:0b,debug:debug,verbose:verbose}

  ;; Storing state information:
  pstate=ptr_new(state,/no_copy)
  widget_control,wid.bbase,set_uvalue=pstate

  ;;========================================------------------------------
  ;; Calculating the values clipping range :: cmanual=0:

  val=values[6L]
  p3d_misc_dataclip,image,range,percentage=val,topwid=wid.bbase, $
      logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then begin
    p3d_wavecal_dispmask_gui_cleanup,wid.bbase
    return
  endif 

  (*pstate).colmin=range[0L]
  (*pstate).colmax=range[1L]

  ;; Updating the range widgets:
  widget_control,(*pstate).wid.tcomi,set_value=strtrim(range[0L],2L)
  widget_control,(*pstate).wid.tcoma,set_value=strtrim(range[1L],2L)

  ;; Drawing the first plots:
  p3d_wavecal_dispmask_gui_plot,pstate,/image,/wavrange,/spectrum,error=error
  if error ne 0 then begin
    p3d_wavecal_dispmask_gui_cleanup,wid.bbase
    return
  endif 

  ;;========================================------------------------------
  ;; Managing the widget tool:

  xmanager,'p3d_wavecal_dispmask_gui',wid.bbase, $
      cleanup='p3d_wavecal_dispmask_gui_cleanup', $
      event_handler='p3d_wavecal_dispmask_gui_resize'

  ;; Returning information on if anything was saved:
  saved=*saved_
  ptr_free,saved_

  return

format_error:
    on_ioerror,NULL
    goto,error_handler

file_error:
  errmsg=defstr
  if n_elements(unit) ne 0L then begin
    tmp=fstat(unit)
    if tmp.open then free_lun,unit ;; closing files which may be open
  endif
  on_ioerror,NULL

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  on_ioerror,NULL
  return
END ;;; procedure: p3d_wavecal_dispmask_gui
