/*-----------------------------------------------------------------*/
/*! 
  \file interfaces_C_F90.h 
  \brief C interface to fortran 77/90/95

  \author  G. Boue
           EXOEarths, Centro de Astrofisica, Universidade do Porto.

   Copyright (C) 2012, CAUP
   email of the author : gwenael.boue@astro.up.pt

   This work has been supported by the European Research Council/European
   Community under the FP7 through Starting Grant agreement number 239953, as
   well as from Fundacao para a Ciencia e a Tecnologia (FCT) through program
   Ciencia 2007 funded by FCT/MCTES (Portugal) and POPH/FSE (EC), and in the
   form of grants reference PTDC/CTE-AST/098528/2008, SFRH/BPD/71230/2010, and
   SFRH/BPD/81084/2011.

*/
/*-----------------------------------------------------------------*/

/*----------------------------------------------------------------*/
/* License of the file :

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
/*----------------------------------------------------------------*/
#include "aromeconfig.h"
#if HAVE_STDIO_H
#include <stdio.h>
#endif
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_ERRNO_H
#include <errno.h>
#endif
#include "arome.h"

/*! \def FC_FUNC_
   define the name of fortran suffixes with name with underscore
*/
#ifndef FC_FUNC_
#define FC_FUNC_(x,X) x##_
#endif /*FC_FUNC_*/


static inline t_orbit *f90arome_getaddressorbit(long long *orbit)
/********************************************************/
/*! return orbit as an address 
   @param orbit (inout) parameter to cast
*/
/********************************************************/
{
   return (t_orbit *) (*orbit);
}

int FC_FUNC_(f90arome_new_orbit, F90AROME_NEW_ORBIT) 
    (long long *orbit)
/***************************************************************/
/* allocate an orbit structure                                 */
/***************************************************************/
{
  int res = 0;
  *orbit = (long long) arome_new_orbit();
  res = (*orbit != 0);
  return res;
}

int FC_FUNC_(f90arome_set_orbit_eo, F90AROME_SET_ORBIT_EO)
    (double *per, double *sma, double *ecc, double *om, double *inc, double *lambda, long long *orbit)
/***************************************************************/
/* set the orbital parameters of the planet
   using eccentricity and argument of pericenter
   returns the number of errors */
/***************************************************************/
{
  return arome_set_orbit_eo(*per, *sma, *ecc, *om, *inc, *lambda, f90arome_getaddressorbit(orbit));
}

int FC_FUNC_(f90arome_set_orbit_kh, F90AROME_SET_ORBIT_KH)
    (double *per, double *sma, double *k, double *h, double *inc, double *lambda, long long *orbit)
/***************************************************************/
/* set the orbital parameters of the planet
   using k=e*cos(om) and h=e*sin(om)
   returns the number of errors */
/***************************************************************/
{
  return arome_set_orbit_kh(*per, *sma, *k, *h, *inc, *lambda, f90arome_getaddressorbit(orbit));
}

double FC_FUNC_(f90arome_get_orbit_transit_time, F90AROME_GET_ORBIT_TRANSIT_TIME)
       (long long *orbit)
/***************************************************************/
/* Computes the time of transit */
/***************************************************************/
{
   return arome_get_orbit_transit_time(f90arome_getaddressorbit(orbit));
}

int FC_FUNC_(f90arome_get_orbit_xyz, F90AROME_GET_ORBIT_XYZ)
   (long long *orbit, double *t, double *x, double *y, double *z)
/***************************************************************/
/* Computes the planet position at a given time t
   returns the number of errors */
/***************************************************************/
{
   return arome_get_orbit_xyz(f90arome_getaddressorbit(orbit), *t, x, y, z);
}

int FC_FUNC_(f90arome_mget_orbit_xyz, F90AROME_MGET_ORBIT_XYZ)
   (long long *orbit, double *t, int *n, double *x, double *y, double *z)
/***************************************************************/
/* Computes the planet position at given times t
   returns the number of errors */
/***************************************************************/
{
   return arome_mget_orbit_xyz(f90arome_getaddressorbit(orbit), t, *n, x, y, z);
}

void FC_FUNC_(f90arome_free_orbit, F90AROME_FREE_ORBIT)
    (long long *orbit)
/***************************************************************/
/* release the memory occupied by the structure */
/***************************************************************/
{
   arome_free_orbit(f90arome_getaddressorbit(orbit));
}


static inline t_arome *f90arome_getaddressarome(long long *arome)
/********************************************************/
/*! return arome as an address 
   @param arome (inout) parameter to cast
*/
/********************************************************/
{
   return (t_arome *) (*arome);
}

int FC_FUNC_(f90arome_alloc_quad, F90AROME_ALLOC_QUAD) 
    (long long *arome, double *u1, double *u2)
/***************************************************************/
/* allocate an arome structure with a quadratic limb-darkening */
/***************************************************************/
{
  int res = 0;
  *arome = (long long) arome_alloc_quad(*u1, *u2);
  res = (*arome != 0);
  return res;
}

int FC_FUNC_(f90arome_alloc_nl, F90AROME_ALLOC_NL) 
    (long long *arome, double *c1, double *c2, double *c3, double *c4)
/***************************************************************/
/* allocate an arome structure with a nonlinear limb-darkening */
/***************************************************************/
{
  int res = 0;
  *arome = (long long) arome_alloc_nl(*c1, *c2, *c3, *c4);
  res = (*arome != 0);
  return res;
}


int FC_FUNC_(f90arome_reset_quad, F90AROME_RESET_QUAD)
    (double *u1, double *u2, long long *arome)
/***************************************************************/
/* re-initialize the quadratic coefficients
   return number of errors */
/***************************************************************/
{
   return arome_reset_quad(*u1, *u2, f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_reset_nl, F90AROME_RESET_NL)
    (double *c1, double *c2, double *c3, double *c4, long long *arome)
/***************************************************************/
/* re-initialize the nonlinear coefficients
   return number of errors */
/***************************************************************/
{
   return arome_reset_nl(*c1, *c2, *c3, *c4, f90arome_getaddressarome(arome));
}


void FC_FUNC_(f90arome_free, F90AROME_FREE)
    (long long *arome)
/***************************************************************/
/* release the memory occupied by the structure */
/***************************************************************/
{
   arome_free(f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_set_lineprofile, F90AROME_SET_LINEPROFILE)
    (double *beta0, double *Vsini, double *sigma0, double *zeta, int *Kmax, long long *arome)
/***************************************************************/
/* set the parameters of the line profile
   returns the number of errors */
/***************************************************************/
{
  return arome_set_lineprofile(*beta0, *Vsini, *sigma0, *zeta, *Kmax, f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_set_planet, F90AROME_SET_PLANET)
    (double *Rp, long long *arome)
/***************************************************************/
/* set the parameters of the planet
   returns the number of errors */
/***************************************************************/
{
  return arome_set_planet(*Rp, f90arome_getaddressarome(arome));
}
   

int FC_FUNC_(f90arome_init_ccf, F90AROME_INIT_CCF)
    (long long *arome)
/***************************************************************/
/* initialize constants for the CCF technique
   returns the number of errors */
/***************************************************************/
{
   return arome_init_CCF(f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_init_iodine, F90AROME_INIT_IODINE)
    (long long *arome)
/***************************************************************/
/* initialize constants for the iodine cell technique
   returns the number of errors */
/***************************************************************/
{
   return arome_init_iodine(f90arome_getaddressarome(arome));
}

int FC_FUNC_(f90arome_calc_fvpbetap, F90AROME_CALC_FVPBETAP)
    (double *x, double *y, double *z, long long *arome)
/***************************************************************/
/* Computes the flux, the subplanet velocity and dispersion
   returns the number of errors */
/***************************************************************/
{
   return arome_calc_fvpbetap(*x, *y, *z, f90arome_getaddressarome(arome));
}


double FC_FUNC_(f90arome_get_rm_ccf, F90AROME_GET_RM_CCF)
       (long long *arome)
/***************************************************************/
/* Computes the RM effect measured by the CCF technique */
/***************************************************************/
{
   return arome_get_RM_CCF(f90arome_getaddressarome(arome));
}


double FC_FUNC_(f90arome_get_rm_iodine, F90AROME_GET_RM_IODINE)
       (long long *arome)
/***************************************************************/
/* Computes the RM effect measured by the iodine cell technique */
/***************************************************************/
{
   return arome_get_RM_iodine(f90arome_getaddressarome(arome));
}


double FC_FUNC_(f90arome_get_rm_mean, F90AROME_GET_RM_MEAN)
       (long long *arome)
/***************************************************************/
/* Computes the RM effect measured by the weighted mean */
/***************************************************************/
{
   return arome_get_RM_mean(f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_get_rm_ccf_e, F90AROME_GET_RM_CCF_E)
   (long long *arome, double *res)
/***************************************************************/
/* Computes the RM effect measured by the CCF technique
   returns the number of errors */
/***************************************************************/
{
   return arome_get_RM_CCF_e(f90arome_getaddressarome(arome), res);
}


int FC_FUNC_(f90arome_get_rm_iodine_e, F90AROME_GET_RM_IODINE)
    (long long *arome, double *res)
/***************************************************************/
/* Computes the RM effect measured by the iodine cell technique
   returns the number of errors */
/***************************************************************/
{
   return arome_get_RM_iodine_e(f90arome_getaddressarome(arome), res);
}


int FC_FUNC_(f90arome_get_rm_mean_e, F90AROME_GET_RM_MEAN_E)
   (long long *arome, double *res)
/***************************************************************/
/* Computes the RM effect measured by the weighted mean
   returns the number of errors */
/***************************************************************/
{
   return arome_get_RM_mean_e(f90arome_getaddressarome(arome), res);
}


double FC_FUNC_(f90arome_get_flux, F90AROME_GET_FLUX)
       (long long *arome)
/***************************************************************/
/* Returns the flux previously computed */
/***************************************************************/
{
   return arome_get_flux(f90arome_getaddressarome(arome));
}


double FC_FUNC_(f90arome_get_vp, F90AROME_GET_VP)
       (long long *arome)
/***************************************************************/
/* Returns the subplanet velocity previously computed */
/***************************************************************/
{
   return arome_get_vp(f90arome_getaddressarome(arome));
}


double FC_FUNC_(f90arome_get_betapr, F90AROME_GET_BETAPR)
       (long long *arome)
/***************************************************************/
/* Returns the subplanet radial dispersion previously computed */
/***************************************************************/
{
  return arome_get_betapR(f90arome_getaddressarome(arome));
}


double FC_FUNC_(f90arome_get_betapt, F90AROME_GET_BETAPT)
       (long long *arome)
/***************************************************************/
/* Returns the subplanet tangential dispersion previously computed */
/***************************************************************/
{
   return arome_get_betapT(f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_set_flux, F90AROME_SET_FLUX)
    (double *f, long long *arome)
/***************************************************************/
/* Set manually the flux */
/***************************************************************/
{
   return arome_set_flux(*f, f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_set_vp, F90AROME_SET_VP)
    (double *vp, long long *arome)
/***************************************************************/
/* Set manually the subplanet velocity */
/***************************************************************/
{
   return arome_set_vp(*vp, f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_set_betapr, F90AROME_SET_BETAPR)
    (double *betapR, long long *arome)
/***************************************************************/
/* Set manually the subplanet radial dispersion */
/***************************************************************/
{
   return arome_set_betapR(*betapR, f90arome_getaddressarome(arome));
}

 
int FC_FUNC_(f90arome_set_betapt, F90AROME_SET_BETAPT)
    (double *betapT, long long *arome)
/***************************************************************/
/* Set manually the subplanet tangential dispersion */
/***************************************************************/
{
   return arome_set_betapT(*betapT, f90arome_getaddressarome(arome));
}


void FC_FUNC_(f90arome_set_exit_on_error, F90AROME_SET_EXIT_ON_ERROR)
     (void)
/***************************************************************/
/* Force the program to exit after an error */
/***************************************************************/
{
  arome_set_exit_on_error();
}


void FC_FUNC_(f90arome_set_continue_on_error, F90AROME_SET_CONTINUE_ON_ERROR)
     (void)
/***************************************************************/
/* Force the program to continue after an error */
/***************************************************************/
{
  arome_set_continue_on_error();
}

void FC_FUNC_(f90arome_set_func_on_error, F90AROME_SET_FUNC_ON_ERROR)
     (void (*userfunc)(const char *))
/***************************************************************/
/* run userfunc on error */
/***************************************************************/
{
  arome_set_func_on_error(userfunc);
}



int FC_FUNC_(f90arome_set_gcf_itmax, F90AROME_SET_GCF_ITMAX)
    (int *ITMAX, long long *arome)
/***************************************************************/
/* Set the internal parameter of the Numerical Recipes gcf function */
/***************************************************************/
{
   return arome_set_gcf_ITMAX(*ITMAX, f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_set_gcf_eps, F90AROME_SET_GCF_EPS)
    (double *EPS, long long *arome)
/***************************************************************/
/* Set the internal parameter of the Numerical Recipes gcf function */
/***************************************************************/
{
   return arome_set_gcf_EPS(*EPS, f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_set_gcf_fpmin, F90AROME_SET_GCF_FPMIN)
    (double *FPMIN, long long *arome)
/***************************************************************/
/* Set the internal parameter of the Numerical Recipes gcf function */
/***************************************************************/
{
   return arome_set_gcf_FPMIN(*FPMIN, f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_set_gser_itmax, F90AROME_SET_GSER_ITMAX)
    (int *ITMAX, long long *arome)
/***************************************************************/
/* Set the internal parameter of the Numerical Recipes gser function */
/***************************************************************/
{
   return arome_set_gser_ITMAX(*ITMAX, f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_set_gser_eps, F90AROME_SET_GSER_EPS)
   (double *EPS, long long *arome)
/***************************************************************/
/* Set the internal parameter of the Numerical Recipes gser function */
/***************************************************************/
{
   return arome_set_gser_EPS(*EPS, f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_set_qtrap_eps, F90AROME_SET_QTRAP)
    (double *EPS, long long *arome)
/***************************************************************/
/* Set the internal parameter of the Numerical Recipes qtrap function */
/***************************************************************/
{
   return arome_set_qtrap_EPS(*EPS, f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_set_qtrap_jmax, F90AROME_SET_QTRAP_JMAX)
    (int *JMAX, long long *arome)
/***************************************************************/
/* Set the internal parameter of the Numerical Recipes qtrap function */
/***************************************************************/
{
   return arome_set_qtrap_JMAX(*JMAX, f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_malloc, F90AROME_MALLOC)
    (int *n, long long *arome)
/***************************************************************/
/* Allocate memory for vectorial call to arome routines */
/***************************************************************/
{
   return arome_malloc(*n, f90arome_getaddressarome(arome));
}


void FC_FUNC_(f90arome_mfree, F90AROME_MFREE)
     (long long *arome)
/***************************************************************/
/* Free memory for vectorial call to arome routines */
/***************************************************************/
{
   arome_mfree(f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_mcalc_fvpbetap, F90AROME_MCALC_FVPBETAP)
    (double *x, double *y, double *z, int *n, long long *arome)
/***************************************************************/
/* Multiple call to arome_fvpbetap */
/***************************************************************/
{
   return arome_mcalc_fvpbetap(x, y, z, *n, f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_mget_flux, F90AROME_MGET_FLUX)
    (long long *arome, int *n, double *flux)
/***************************************************************/
/* Multiple call to arome_get_flux */
/***************************************************************/
{
   return arome_mget_flux(f90arome_getaddressarome(arome), *n, flux);
}


int FC_FUNC_(f90arome_mget_vp, F90AROME_MGET_VP)
    (long long *arome, int *n, double *vp)
/***************************************************************/
/* Multiple call to arome_get_vp */
/***************************************************************/
{
   return arome_mget_vp(f90arome_getaddressarome(arome), *n, vp);
}


int FC_FUNC_(f90arome_mget_betapr, F90AROME_MGET_BETAPR)
    (long long *arome, int *n, double *betapR)
/***************************************************************/
/* Multiple call to arome_get_betapR */
/***************************************************************/
{
   return arome_mget_betapR(f90arome_getaddressarome(arome), *n, betapR);
}


int FC_FUNC_(f90arome_mget_betapt, F90AROME_MGET_BETAPT)
    (long long *arome, int *n, double *betapT)
/***************************************************************/
/* Multiple call to arome_get_betapT */
/***************************************************************/
{
   return arome_mget_betapT(f90arome_getaddressarome(arome), *n, betapT);
}


int FC_FUNC_(f90arome_mset_flux, F90AROME_SET_FLUX)
    (double *f, int *n, long long *arome)
/***************************************************************/
/* Multiple call to arome_set_flux */
/***************************************************************/
{
   return arome_mset_flux(f, *n, f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_mset_vp, F90AROME_MSET_VP)
    (double *vp, int *n, long long *arome)
/***************************************************************/
/* Multiple call to arome_set_vp */
/***************************************************************/
{
   return arome_mset_vp(vp, *n, f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_mset_betapr, F90AROME_MSET_BETAPR)
    (double *betapR, int *n, long long *arome)
/***************************************************************/
/* Multiple call to arome_set_betapR */
/***************************************************************/
{
   return arome_mset_betapR(betapR, *n, f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_mset_betapt, F90AROME_MSET_BETAPT)
    (double *betapT, int *n, long long *arome)
/***************************************************************/
/* Multiple call to arome_set_betapT */
/***************************************************************/
{
   return arome_mset_betapT(betapT, *n, f90arome_getaddressarome(arome));
}


int FC_FUNC_(f90arome_mget_rm_ccf, F90AROME_MGET_CCF)
    (long long *arome, int *n, double *res)
/***************************************************************/
/* Multiple call to arome_get_RM_CCF_e */
/***************************************************************/
{
   return arome_mget_RM_CCF(f90arome_getaddressarome(arome), *n, res);
}


int FC_FUNC_(f90arome_mget_rm_iodine, F90AROME_MGET_IODINE)
    (long long *arome, int *n, double *res)
/***************************************************************/
/* Multiple call to arome_get_RM_iodine_e */
/***************************************************************/
{
   return arome_mget_RM_iodine(f90arome_getaddressarome(arome), *n, res);
}


int FC_FUNC_(f90arome_mget_rm_mean, F90AROME_MGET_MEAN)
    (long long *arome, int *n, double *res)
/***************************************************************/
/* Multiple call to arome_get_RM_mean_e */
/***************************************************************/
{
   return arome_mget_RM_mean(f90arome_getaddressarome(arome), *n, res);
}



