/*------------------------------------------------------------------------------
NAME
	centroid.c - compute sodium centroid altitude
 
DESCRIPTION
	'centroid' reads lidar data FITS files for the requested night and outputs 
	time, total count, centroid altitude and its standard error for each shot.
  
USAGE
	centroid [options] [directory/]date > output_file

PARAMETERS
	'file' is the date to be processed, in the form 'yyyymmdd'. If the data
	files for this night are not in the current directory, 'files' must 
	be preceeded by a path to the directory that contains the data files
	(eg. /lidar/data/20090924 ). It is not necessary to add the extension
	_00.fits, _01.fits, etc. The program will find all FITS data files
	for the indicated date in the specified directory.
 
OPTIONS
	-h, -help			print help for this command
	-l	length			print at most 'length' hours, from start_time [24]
	-s	start_time		start at 'start_time' (hours UTC) [0]
	-t					don't print a table header
 
REVISION
	110502	P. Hickson
	140515  [CDS] SIMPLE or XTENSION

COPYRIGHT
	Copyright (c) 2010 by P. Hickson. No part of this program may be copied or 
	released to a third party without written permission.
------------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>

#include "lidar.h"

// Function prototypes
void 
	parse_args(int,char**),
	print_help();

// External variables
extern char
	*optarg;
extern int	
	errno,
	optind,
	optopt,
	opterr,
	optreset;

// Global variables
char	
	*prog_name;
u_char		
	*abins;
u_short
	*data;
int
	block		,
	pheader		= 1,
	nfiles		;
double
	length		= 24.0,
	start_time	= 0.0;

//------------------------------------------------------------------------------
#define OPTIONS	"hl:s:t"

void	parse_args(int,char**);

void	usage() {
	
	fprintf(stderr,"Usage: %s [options] file > output_file\n",prog_name);
	fprintf(stderr,"where options include:\n");
	fprintf(stderr,"  -h, -help      print help for this command\n");
	fprintf(stderr,"  -l length      print at most 'length' hours [24]\n");
	fprintf(stderr,"  -s start_time  start at 'start_time' hours UTC [0] \n");
	fprintf(stderr,"  -t             don't print the table header\n");
	exit(1);
}
void	parse_args(int argc, char **argv) {
	
	int		c;
	
	prog_name = argv[0];
	while ((c = getopt(argc,argv,OPTIONS)) != -1) {
		switch (c) {
			case 'h':
				print_help();
				break;
			case 'l':
				length = atof(optarg);
				break;
			case 's':
				start_time = atof(optarg);
				break;
			case 't':
				pheader = 0;
				break;
			default:
				usage();
		}
	}
	nfiles = argc-optind;
	if (nfiles != 1) usage();
}
//------------------------------------------------------------------------------
void print_help() {
	
	printf("\nNAME\n");
	printf("\tcentroid.c - compute sodium centroid altitude\n");
	putchar('\n');
	printf("DESCRIPTION\n");
	printf("\t\'centroid\' reads lidar data FITS files for the requested night and outputs\n");
	printf("\ttime, total count, centroid altitude and its standard error for each shot.\n");
	putchar('\n');
	printf("USAGE\n");
	printf("\tcentroid [options] [directory/]date > output_file\n");
	putchar('\n');
	printf("PARAMETERS\n");
	printf("\t'file' is the date to be processed, in the form 'yyyymmdd'. If the data\n");
	printf("\tfiles for this night are not in the current directory, 'files' must\n");
	printf("\tbe preceeded by a path to the directory that contains the data files\n");
	printf("\t(eg. /lidar/data/20090924 ). It is not necessary to add the extension\n");
	printf("\t_00.fits, _01.fits, etc. The program will find all FITS data files\n");
	printf("\tfor the indicated date in the specified directory.\n");
	putchar('\n');				
	printf("OPTIONS\n");
	printf("\t-h, -help         print help for this command\n");
	printf("\t-l    length      print at most 'length' hours, from start_time [24]\n");
	printf("\t-s    start_time  start at 'start_time' (hours UTC) [0]\n");
	printf("\t-t                don't print a table header\n");
	putchar('\n');
	printf("EXAMPLES\n");
	printf("\tcentroid /data/lidar/fits/20091008 > 20091008_cen.txt\n");
	printf("\t - computes a centroid time series using data from all fits files for the\n");
	printf("\t   UT date 2009-10-08 (ie. files having names of the form \'20091008_XX.fits\'\n");
	printf("\t   located in the directory \'/data/lidar/fits/\') using the default\n");
	printf("\t   parameters (shown in square brackets above under OPTIONS) and writes the\n");
	printf("\t   result to the text file \'20091008_cen.txt\'.\n");
	putchar('\n');				
	printf("\tcentroid -s 12.0 -s 2 20091008 > 20091008_cen.txt\n");
	printf("\t - computes a 2 hour centroid time series, starting at 12:00 UTC, for\n");
	printf("\t   UT 2009-10-08, assuming that the fits files are located in the current\n");
	printf("\t   directory. The result is written to the text file \'20091008_cen.txt\'.\n");
	putchar('\n');
	printf("REVISION\n");
	printf("\t110502\n");
	putchar('\n');					
	exit(0);
}
//------------------------------------------------------------------------------
int main(int argc,char **argv) {

	u_short		*dp;
	char		header[FITS_HDR_SIZ],s[STRSIZE],dir[STRSIZE],date[9],
				file_name[STRSIZE];
	int			bin,blocks,channel,dsize,err,fd,i,j,k,n,seq;
	double		alt,end_time,hour,malt,msalt,malt_err;
	Fits		fits;
	
	// Read the command-line options.
	parse_args(argc,argv);
	
	// Allocate storage
	if ((data = (u_short*)malloc(2*MAXCOUNT*NSHOTS)) == NULL) {
		fprintf(stderr,"%s: error allocating memory\n",prog_name);
		exit(-1);
	}
	
	// Determine the directory, and the date to be processed.
	for (i=strlen(argv[optind]);i>=0;i--) {
		if (argv[optind][i] == '/') {
			break;
		}
	}
	if (i >= 0) {
		strncpy(dir,argv[optind],i+1);
		dir[i+1] = '\0';
	}
	else {
		dir[0] = '\0';
	}
	strncpy(date,argv[optind]+i+1,8);
	date[8] = '\0';
	
	// Loop through file names
	for (seq=0;seq<MAX_SEQ_NO;seq++) {
		
		// Create the file name
		sprintf(file_name,"%s%s_%02d.fits",dir,date,seq);
		
		// Open the data file.
		if ((fd = open(file_name,O_RDONLY)) == -1) {
			if (seq == 0) {
				fprintf(stderr,"%s: cannot open %s\n",prog_name,file_name);
				exit(-1);
			}
			else {
				exit(0);
			}
		}
		if (pheader) fprintf(stderr,"%s:\n",file_name);
		
		block = 0;
		// Start processing the data.
		for (i=0;;i++) {
			
			// Read and decode the FITS header.
			if (read(fd,header,FITS_HDR_SIZ) != FITS_HDR_SIZ) {
				break;
			}
			if (read_fits_header(header,&fits) != 0) {
				fprintf(stderr,"%s: format error in FITS block %d\n",prog_name,
						block);
				exit(-1);
			}
			block++;
			
			// Print a header
			if (pheader && i == 0 && seq == 0) {
				printf("#------------------------------\n");
				printf("# UBC LIDAR  %4d:%02d:%02d\n",fits.year,fits.mon,
					fits.day);
				printf("# UT_HR COUNTS CENTROID   ERROR\n");
				printf("#------------------------------\n");
			}
			
			// Find the number of data blocks.
			blocks = (2*fits.naxis1)/FITS_BLOCK_SIZ;
			if (2*fits.naxis1 > FITS_BLOCK_SIZ*blocks) blocks++;
			dsize = FITS_BLOCK_SIZ*blocks;
			
			// Read the data.
			if (read(fd,data,dsize) != dsize) {
				fprintf(stderr,"%s: error reading %s\n",prog_name,file_name);
				exit(-1);
			}
			block += blocks;
			
			dp = data;
			end_time = start_time+length;
			for (j=0;j<NSHOTS;j++) {
				
				// Find the time
				hour = fits.hour+(fits.min+fits.sec/60.0)/60.0+j*DT/3600.0;
				
				// Print only the time interval requested
				if (hour < start_time || hour > end_time) continue;
				
				malt = msalt = n = 0;
				
				// Process the photons. The mean altitude is given by
				//	<a> =  sum(a_k)/n
				// and its variance is given by
				//	Var(a) = (<a^2> - <a>^2)/n
				// where n is the number of photons.
				//
				for (k=0;k<fits.count[j];k++) {
					
					// Determine the bin and channel numbers for the photon.
					bin = 0x3fff & *dp;
					channel = 0x0003 & (*dp++ >> 14);
					
					// Consider only photons in the second bin sequence as the
					// first is pure Rayleigh.
					//
					if (bin > fits.nbin0) {
						
						// Convert the bin number to an altitude.
						alt = fits.alt+((bin-fits.nbin0)*fits.bsize1
							+fits.bstart1)*M_PER_NS;
						malt  += alt;
						msalt += alt*alt;
						n++;
					}
				}
				malt /= n;
				msalt /= n;
				malt_err = sqrt((msalt-malt*malt)/n);
				
				printf("%08.5f %5d %8.1f %7.1f\n",hour,fits.count[j],malt,
					malt_err);
			}
		}
		
		// Close the data file.
		close(fd);
	}
	
	// Free allocated storage.
	free(data);
}
#include "fits_read_header.c"
