pro fittell,wave,data,error,range,gdasdata,obsaltitude,params,paramserror,waveresult,result,cleaned,cleanederror,exclude=exclude,plot=plot,silent=silent,parinfo=parinfo,titleplot=titleplot
;+
; NAME:
;	FITTELL
;
; PURPOSE:
;	This procedure uses LBLRTM to fit a transmission model to the observed
;	spectrum and then uses this model to remove the telluric lines from
;	the spectrum.
;
; CALLING SEQUENCE:
;	FITTELL,Wave,Data,Error,Range,Gdasdata,Obsaltitude,Params,Paramserror,Waveresult,Result,Cleaned,Cleanederror
;
; INPUTS:
;	Wave:		Wavelength array of observed spectrum
;	Data:		Flux array of observed spectrum
;	Error:		Error of flux measurements
;	Range:		Wavelength range of fit as two-element array in the form [start,end]
;	Gdasdata:	Scalar string containing absolute path to GDAS file
;	Obsaltitude:	Altitude angle observation was taken at in deg
;	
; KEYWORD PARAMETERS:
;	EXCLUDE:	Regions to exclude from wavelength array in the range specified
;			as array with up to 10 entries of the form [start1,end1,...,start5,end5].
;	PLOT:		Set this keyword to get a plot of the result.
;	SILENT:		Set this keyword to suppress informational messages
;	PARINFO:	Parameter info structure to use in MPFIT. Default is a standard structure
;			created by CREATE_STANDARD_PARINFO.
;	TITLEPLOT:	Title of plot.
;
; OUTPUTS:
;	Params:		Fitted parameters ([smoothing_FWHM,velocity_shift,
;			slope_of_straight,intercept_of_straight,water_abundance,
;			methane_abundance,carbondioxide_abundance,oxygen_abundance])
;	Paramserror:	Formal 1-sigma errors of parameters
;	Waveresult:	Wavelength array of the fitted part
;	Result:		Model fitted with all parameters (to be compared to observation)
;	Cleaned:	Telluric line removed spectrum.
;	Cleanederror:	Error of flux measurements divided by transmission model.
;
; EXAMPLE:
;	FITTELL,Wave,Data,Error,[9000.,9300.],'/here/lies/the/Gdasdata',87.2,Params,Paramserror,Waveresult,Result,Cleaned,Cleanederror
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-

; check the input parameters & keywords
if n_params() lt 11 then begin
    message,/info,'Syntax: fittell,wave,data,error,range,gdasdata,obsaltitude,params,paramserror,waveresult,result,cleaned,cleanederror,exclude=exclude, plot=plot,silent=silent,parinfo=parinfo,titleplot=titleplot'
    retall
endif

if not keyword_set(parinfo) then parinfo=create_standard_parinfo() else parinfo=parinfo

if not keyword_set(silent) then begin   ; make sure quiet has a value
      quiet=0
      silent=0
      anfang=systime(/julian) ; to measure execution time
endif else quiet=1 ; suppress output messages of mpfitfun

; define index of region to fit
if keyword_set(exclude) then induse=exclusion(wave,range,exclude) else induse=where(wave ge range[0] and wave le range[1])
wavuse=wave[induse]
datuse=data[induse]
erruse=error[induse]

; actual fit using MPFIT
if keyword_set(silent) then fctargs={gdasdata:gdasdata,obsaltitude:obsaltitude,silent:silent} else fctargs={gdasdata:gdasdata,obsaltitude:obsaltitude}
params=mpfitfun('tell',wavuse,datuse,erruse,parinfo=parinfo,niter=n,yfit=res,status=stat,errmsg=erm,bestnorm=chi,dof=dof,nfev=nfev,quiet=quiet,functargs=fctargs,perror=perror)
; if there was an error give error message independent of setting of silent
if stat le 0 then print,stat,erm
paramserror=perror

; create informative messages if wanted
if not keyword_set(silent) then begin
     ende=systime(/julian)
     dauer=(ende-anfang)*24*60
     print,format='(A13,F6.2,A4)','Time needed: ',dauer,' min'
     text=statusinterpreter(stat)
     print,'Status: ',stat,' ',text
     if stat le 0 then print,erm
     print,'Number of iterations: ',n
     print,'Number of calls to tell: ',nfev
     print,'Parameters of tell: smooth, v, slope, b, wat, meth, co2, o2'
     print,params
     print,perror
     print,'Bestnorm: ',chi
     print,'DOF: ',dof
endif

; return fit result of complete range if exclude was set
if keyword_set(exclude) then begin
     waveresult=wave[where(wave ge range[0] and wave le range[1])]
     result=tell(waveresult,params,gdasdata=gdasdata,obsaltitude=obsaltitude,silent=silent)
endif else begin
     waveresult=wavuse
     result=res
endelse

; recalculate the model without adjustment by straight line, i.e. get transmission spectrum convolved with Gaussian
modell=tellwithout(waveresult,params,gdasdata=gdasdata,obsaltitude=obsaltitude,silent=silent)

; remove the telluric lines 
if keyword_set(exclude) then begin
     cleaned=data[where(wave ge range[0] and wave le range[1])]/modell
     cleanederror=error[where(wave ge range[0] and wave le range[1])]/modell
endif else begin
    cleaned=datuse/modell
    cleanederror=erruse/modell
endelse

; plot result
if keyword_set(plot) then begin
    if keyword_set(titleplot) then nameplot=titleplot else nameplot=''
    loadct,39,/silent
    !P.MULTI=[0,1,2]
    ; plot original data and fit
    plot,wave,data,xra=[range[0]-10.d,range[1]+10.d],/xsty,xtitle='Wavelength',ytitle='Flux',title=nameplot+' Original data (black) and fit (red)',xtickformat='(I5)'
    oplot,waveresult,result,color=235
    ; mark the regions that where excluded
    if keyword_set(exclude) then begin
        y=mean(result)
        for da=0,n_elements(exclude)-1 do plots,exclude[da],[y-0.1*y,y+0.1*y],col=191
    endif
    ; plot cleaned data and original data
    plot,waveresult,cleaned,xra=[range[0]-10.d,range[1]+10.d],/xsty,xtitle='Wavelength',ytitle='Flux',title=nameplot+' Cleaned data (black) and original data (red)',xtickformat='(I5)'
    oplot,wave,data,color=235
    !P.MULTI=0
endif
end