pro getabundances,wavevis,specvis,errvis,wavenir,specnir,errnir,gdasdata,obsaltitude,ab_h2o,ab_co2,ab_ch4,ab_o2,ab_h2o_str,ab_co2_str,ab_ch4_str,ab_o2_str,silent=silent,plot=plot
;+
; NAME:
;	GETABUNDANCES
;
; PURPOSE:
;	This procedure determines the abundances of the main molecules (H2O,
;	CO2, CH4, O2) producing telluric lines in the X-Shooter spectral range.
;
; CALLING SEQUENCE:
;	GETABUNDANCES, Wavevis, Specvis, Errvis, Wavenir, Specnir, Errnir,
;	Gdasdata, Obsaltitude, Ab_h2o, Ab_co2, Ab_ch4, Ab_o2, Ab_h2o_str,
;	Ab_co2_str, Ab_ch4_str, Ab_o2_str
;
; INPUTS:
;	Wavevis:	Wavelength array of VIS data
;	Specvis:	Flux array of VIS data
;	Errvis:		Error array of VIS data
;	Wavenir:	Wavelength array of NIR data
;	Specnir:	Flux array of NIR data
;	Errnir:		Error array of NIR data
;	Gdasdata:	Scalar string containing absolute path to GDAS file
;	Obsaltitude:	Altitude angle observation was taken at in deg
;	
; KEYWORD PARAMETERS:
;	SILENT:		Set this keyword to suppress informational messages
;	PLOT:		Set this keyword to get a plot of the result.
;
; OUTPUTS:
;	Ab_h2o:		Abundance of H2O in this observation
;	Ab_co2:		Abundance of CO2 in this observation
;	Ab_ch4:		Abundance of CH4 in this observation
;	Ab_o2:		Abundance of O2 in this observation
;	Ab_h2o_str:	Structure containing the fit results for the
;			individual regions used to determine the abundance
;			of H20
;	Ab_co2_str:	Structure containing the fit results for the
;			individual regions used to determine the abundance
;			of CO2
;	Ab_ch4_str:	Structure containing the fit results for the
;			individual regions used to determine the abundance
;			of CH4
;	Ab_o2_str:	Structure containing the fit results for the
;			individual regions used to determine the abundance
;			of O2
;
; EXAMPLE:
;	GETABUNDANCES, Wavevis, Specvis, Errvis, Wavenir, Specnir, Errnir,
;	'/here/lies/the/Gdasdata',87.2, Ab_h2o, Ab_co2, Ab_ch4, Ab_o2,
;	Ab_h2o_str, Ab_co2_str, Ab_ch4_str, Ab_o2_str
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;	N. Rudolf, October 2014, Adapted parameter borders
;	N. Rudolf, November 2014, Added output to save the fit results of
;	the individual regions used to determine the abundances of the
;	main telluric contributors.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
if n_params() lt 8 then begin
    message,/info,'Syntax: GETABUNDANCES,wavevis,specvis,errvis,wavenir,specnir,errnir,gdasdata,obsaltitude,ab_h2o,ab_co2,ab_ch4,ab_o2,ab_h2o_str,ab_co2_str,ab_ch4_str,ab_o2_str,silent=silent,plot=plot'
    retall
endif
if not keyword_set(plot) then plot=0

abund={range:dblarr(2),p:dblarr(8),pe:dblarr(8)}
ab_h2o_str=replicate(abund,5)
ab_co2_str=replicate(abund,2)
ab_ch4_str=replicate(abund,3)
ab_o2_str=replicate(abund,2)

if not keyword_set(silent) then print,'Fitting abundances:'

; set up parinfo structure for H2O
parw = replicate({value:0.D, fixed:0, limited:[0,0],limits:[0.D,0],step:0.d},8)
; smooth, i.e. width of gaussian
parw[0].value=0.5d
parw[0].limited[0]=1
parw[0].limits[0]=1.d-2
parw[0].limited[1]=1
parw[0].limits[1]=10.d  
parw[0].step=0.1d
; v, i.e. shift in wavelength
parw[1].value=0.1d
parw[1].step=0.1d
; slope, i.e. slope of straigth line that is used to adapt to slope of observed spectrum
parw[2].value=-5.8d
parw[2].step=0.1d
; b, i.e. y-intercept of straigth line that is used to adapt to slope of observed spectrum
parw[3].value=2.d
parw[3].step=0.1d
; wat, i.e. abundance factor for water
parw[4].value=1.0d
parw[4].fixed=0
parw[4].limited=[1,1]
parw[4].limits[0]=0.4
parw[4].limits[1]=2.d
parw[4].step=0.1d
; meth, i.e. abundance factor for methane
parw[5].value=1.0d
parw[5].fixed=1
; co2, i.e. abundance factor for carbondioxide
parw[6].value=1.0d
parw[6].fixed=1
; o2, i.e. abundance factor for oxygen
parw[7].value=1.0d
parw[7].fixed=1      
if not keyword_set(silent) then print,'Fit H2O'
fittell,wavevis,specvis,errvis,[9040.,9210.],gdasdata,obsaltitude,pw1,pew1,wrw1,rw1,clw1,parinfo=parw,/silent,plot=plot,titleplot='Fit H2O:'
ab_h2o_str[0].range=[9040.,9210.]
ab_h2o_str[0].p=pw1
ab_h2o_str[0].pe=pew1

fittell,wavevis,specvis,errvis,[9270.,9510.],gdasdata,obsaltitude,pw2,pew2,wrw2,rw2,clw2,parinfo=parw,/silent,plot=plot,titleplot='Fit H2O:'
ab_h2o_str[1].range=[9270.,9510.]
ab_h2o_str[1].p=pw2
ab_h2o_str[1].pe=pew2

fittell,wavenir,specnir,errnir,[13160.,13280.],gdasdata,obsaltitude,pw3,pew3,wrw3,rw3,clw3,parinfo=parw,/silent,plot=plot,titleplot='Fit H2O:'
ab_h2o_str[2].range=[13160.,13280.]
ab_h2o_str[2].p=pw3
ab_h2o_str[2].pe=pew3

fittell,wavenir,specnir,errnir,[14470.,14685.],gdasdata,obsaltitude,pw4,pew4,wrw4,rw4,clw4,parinfo=parw,/silent,plot=plot,titleplot='Fit H2O:'
ab_h2o_str[3].range=[14470.,14685.]
ab_h2o_str[3].p=pw4
ab_h2o_str[3].pe=pew4

fittell,wavenir,specnir,errnir,[14685.,14900.],gdasdata,obsaltitude,pw5,pew5,wrw5,rw5,clw5,parinfo=parw,/silent,plot=plot,titleplot='Fit H2O:'
ab_h2o_str[4].range=[14685.,14900.]
ab_h2o_str[4].p=pw5
ab_h2o_str[4].pe=pew5

; abundance used is mean of all abundances
ab_h2o=mean([pw1[4],pw2[4],pw3[4],pw4[4],pw5[4]])

; set up parinfo structure for CO2
parco2 = replicate({value:0.D, fixed:0, limited:[0,0],limits:[0.D,0],step:0.d},8)
; smooth, i.e. width of gaussian
parco2[0].value=0.5d
parco2[0].limited[0]=1
parco2[0].limits[0]=1.d-2
parco2[0].limited[1]=1
parco2[0].limits[1]=10.d  
parco2[0].step=0.1d
; v, i.e. shift in wavelength
parco2[1].value=0.1d
parco2[1].step=0.1d
; slope, i.e. slope of straigth line that is used to adapt to slope of observed spectrum
parco2[2].value=-5.8d
parco2[2].step=0.1d
; b, i.e. y-intercept of straigth line that is used to adapt to slope of observed spectrum
parco2[3].value=2.d
parco2[3].step=0.1d
; wat, i.e. abundance factor for water
parco2[4].value=ab_h2o
parco2[4].fixed=1
; meth, i.e. abundance factor for methane
parco2[5].value=1.0d
parco2[5].fixed=1
; co2, i.e. abundance factor for carbondioxide
parco2[6].value=1.0d
parco2[6].fixed=0
parco2[6].limited=[1,1]
parco2[6].limits[0]=0.4
parco2[6].limits[1]=2.d
parco2[6].step=0.1d
; o2, ie. abundance factor for oxygen
parco2[7].value=1.0d
parco2[7].fixed=1    
if not keyword_set(silent) then print,'Fit CO2'
fittell,wavenir,specnir,errnir,[19950.,20300.],gdasdata,obsaltitude,pco21,peco21,wrco21,rco21,clco21,parinfo=parco2,/silent,plot=plot,titleplot='Fit CO2:'
ab_co2_str[0].range=[19950.,20300.]
ab_co2_str[0].p=pco21
ab_co2_str[0].pe=peco21

fittell,wavenir,specnir,errnir,[20450.,20800.],gdasdata,obsaltitude,pco22,peco22,wrco22,rco22,clco22,parinfo=parco2,/silent,plot=plot,titleplot='Fit CO2:'
ab_co2_str[1].range=[20450.,20800.]
ab_co2_str[1].p=pco22
ab_co2_str[1].pe=peco22

; abundance used is mean of all abundances
ab_co2=mean([pco21[6],pco22[6]])

; set up parinfo structure for CH4
parch4 = replicate({value:0.D, fixed:0, limited:[0,0],limits:[0.D,0],step:0.d},8)
; smooth, i.e. width of gaussian
parch4[0].value=0.5d
parch4[0].limited[0]=1
parch4[0].limits[0]=1.d-2
parch4[0].limited[1]=1
parch4[0].limits[1]=10.d  
parch4[0].step=0.1d
; v, i.e. shift in wavelength
parch4[1].value=0.1d
parch4[1].step=0.1d
; slope, i.e. slope of straigth line that is used to adapt to slope of observed spectrum
parch4[2].value=-5.8d
parch4[2].step=0.1d
; b, i.e. y-intercept of straigth line that is used to adapt to slope of observed spectrum
parch4[3].value=2.d
parch4[3].step=0.1d
; wat, i.e. abundance factor for water
parch4[4].value=ab_h2o
parch4[4].fixed=1
; meth, i.e. abundance factor for methane
parch4[5].value=1.0d
parch4[5].fixed=0
parch4[5].limited=[1,1]
parch4[5].limits[0]=0.4
parch4[5].limits[1]=2.d
parch4[5].step=0.1d
; co2, i.e. abundance factor for carbondioxide
parch4[6].value=ab_co2
parch4[6].fixed=1
; o2, ie. abundance factor for oxygen
parch4[7].value=1.d
parch4[7].fixed=1    
if not keyword_set(silent) then print,'Fit CH4'
fittell,wavenir,specnir,errnir,[22850.,23110.],gdasdata,obsaltitude,pch41,pech41,wrch41,rch41,clch41,parinfo=parch4,/silent,plot=plot,titleplot='Fit CH4:'
ab_ch4_str[0].range=[22850.,23110.]
ab_ch4_str[0].p=pch41
ab_ch4_str[0].pe=pech41

fittell,wavenir,specnir,errnir,[23370.,23630.],gdasdata,obsaltitude,pch42,pech42,wrch42,rch42,clch42,parinfo=parch4,/silent,plot=plot,titleplot='Fit CH4:'
ab_ch4_str[1].range=[23370.,23630.]
ab_ch4_str[1].p=pch42
ab_ch4_str[1].pe=pech42

fittell,wavenir,specnir,errnir,[23630.,23900.],gdasdata,obsaltitude,pch43,pech43,wrch43,rch43,clch43,parinfo=parch4,/silent,plot=plot,titleplot='Fit CH4:'
ab_ch4_str[2].range=[23630.,23900.]
ab_ch4_str[2].p=pch43
ab_ch4_str[2].pe=pech43

; abundance used is mean of all abundances
ab_ch4=mean([pch41[5],pch42[5],pch43[5]])

; set up parinfo structure for O2
paro2 = replicate({value:0.D, fixed:0, limited:[0,0],limits:[0.D,0],step:0.d},8)
; smooth, i.e. width of gaussian
paro2[0].value=0.5d
paro2[0].limited[0]=1
paro2[0].limits[0]=1.d-2
paro2[0].limited[1]=1
paro2[0].limits[1]=10.d  
paro2[0].step=0.1d
; v, i.e. shift in wavelength
paro2[1].value=0.1d
paro2[1].step=0.1d
; slope, i.e. slope of straigth line that is used to adapt to slope of observed spectrum
paro2[2].value=-5.8d
paro2[2].step=0.1d
; b, i.e. y-intercept of straigth line that is used to adapt to slope of observed spectrum
paro2[3].value=2.d
paro2[3].step=0.1d
; wat, i.e. abundance factor for water
paro2[4].value=ab_h2o
paro2[4].fixed=1
; meth, i.e. abundance factor for methane
paro2[5].value=ab_ch4
paro2[5].fixed=1
; co2, i.e. abundance factor for carbondioxide
paro2[6].value=ab_co2
paro2[6].fixed=1
paro2[6].step=0.1d
; o2, ie. abundance factor for oxygen
paro2[7].value=1.d
paro2[7].fixed=0 
paro2[7].limited=[1,1]
paro2[7].limits[0]=0.4
paro2[7].limits[1]=2.d
paro2[7].step=0.1d
if not keyword_set(silent) then print,'Fit O2'
fittell,wavevis,specvis,errvis,[7590.,7660.],gdasdata,obsaltitude,po21,peo21,wro21,ro21,clo21,parinfo=paro2,/silent,plot=plot,titleplot='Fit O2:'
ab_o2_str[0].range=[7590.,7660.]
ab_o2_str[0].p=po21
ab_o2_str[0].pe=peo21

fittell,wavenir,specnir,errnir,[12550.,12775.],gdasdata,obsaltitude,po22,peo22,wro22,ro22,clo22,parinfo=paro2,/silent,plot=plot,titleplot='Fit O2:'
ab_o2_str[1].range=[12550.,12775.]
ab_o2_str[1].p=po22
ab_o2_str[1].pe=peo22

; abundance used is mean of all abundances
ab_o2=mean([po21[7],po22[7]])
end