;+
; NAME:
;         p3d_display_grat_plot
;
;         $Id: p3d_display_grat_plot.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         This routine plots the geometrical orientation of a diffraction
;         grating.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: display routine
;
; CALLING SEQUENCE:
;         p3d_display_grat_plot,eta,alpha,blaze,forward=,ll=,ur=,lr=,ul=,
;             topwid=,verbose=,error=,/debug,/help
;
; INPUTS:
;         eta             - A decimal value specifying the collimator/camera
;                           angle [].
;         alpha           - A decimal value specifying the angle (grating
;                           normal, camera axis) [].
;         blaze           - A decimal value specifying the grating blaze angle
;                           [].
;
; KEYWORD PARAMETERS:
;         forward         - "forward" diffraction, m=+1.
;         ur              - A string specifying the text in the upper right
;                           corner.
;         lr              - A string specifying the text in the lower right
;                           corner.
;         ll              - A string specifying the text in the lower left
;                           corner.
;         ul              - A string specifying the text in the upper left
;                           corner.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         none
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; EXAMPLES:
;         p3d_display_grat_plot, 42.0, 12.4, 5.8, [/FORWARD]
;
; MODIFICATION HISTORY:
;         13.10.2008 - Converted from the original routine grat_plot of
;                      Martin Roth and Andreas Kelz. /CS
;-
PRO p3d_display_grat_plot,eta,alpha,blaze,forward=varforward, $
        ll=varll,ur=varur,lr=varlr,ul=varul, $
        topwid=topwid,verbose=verbose,error=error,help=help,debug=debug
  compile_opt hidden,IDL2

  error=0 & rname='p3d_display_grat_plot: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_display_grat_plot'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  if ~n_elements(eta) then begin
    errmsg='ETA must be set.'
    goto,error_handler
  endif

  if ~n_elements(alpha) then begin
    errmsg='ALPHA must be set.'
    goto,error_handler
  endif

  if ~n_elements(blaze) then begin
    errmsg='BLAZE must be set.'
    goto,error_handler
  endif

  if (size(eta  ,/type) ge 6L and size(eta  ,/type) le 11L) or $
     (size(alpha,/type) ge 6L and size(alpha,/type) le 11L) or $
     (size(blaze,/type) ge 6L and size(blaze,/type) le 11L) then begin
    errmsg='ETA, ALPHA, and BLAZE must be of floating point type.'
    goto,error_handler
  endif

  eta=float(eta)
  alpha=float(alpha)
  blaze=float(blaze)

  if ~n_elements(varforward) then varforward=0L
  if ~n_elements(varur) then varur=''
  if ~n_elements(varll) then varll=''
  if ~n_elements(varlr) then varlr=''
  if ~n_elements(varul) then varul=''

  ;;========================================------------------------------
  ;; Plotting:

  c=1.0 & r_blaze=!dtor*blaze
  c1=c/((tan(r_blaze))^2.0+1.0)
  h=c1*tan(r_blaze)

  window,1,xsize=700,ysize=700,retain=2
  plotrange=2.5
  plot,[-1000,-1000],xrange=[-plotrange,plotrange], $
       yrange=[-plotrange,plotrange],/xstyle,/ystyle

  if varur ne '' then xyouts, 1.0, 2.1,varur,/data,charsize=2,charthick=2
  if varll ne '' then xyouts,-2.2,-2.2,varll,/data,charsize=2,charthick=2
  if varlr ne '' then xyouts, 1.0,-2.2,varlr,/data,charsize=2,charthick=2
  if varul ne '' then xyouts,-2.2, 2.1,varul,/data,charsize=2,charthick=2


  x0=0   &   x1=c1   &   x2=c
  y0=0   &   y1=h    &   y2=0

  x_facette = [x0,x1,x2]
  y_facette = [y0,y1,y2]


  x_grat_norm = [0,0]
  y_grat_norm = [-0.9,2]

  x_surf_norm = [0,-2*h]
  y_surf_norm = [0,2*c1]

  x_surface=[x0-2,x1-2,x2-2,-x2,-x1,-x0,x0,x1,x2,-x2+2,-x1+2,-x0+2]
  y_surface=[y0,y1,y2,-y2,-y1,-y0,y0,y1,y2,-y2,-y1,-y0]

  x_backside=[-2,-2,2,2]
  y_backside=[0,-0.9,-0.9,0]

  blue=1e6 & green=1e4 & red=1e3 & blue=1e2 & yellow=1e2

  if varforward then begin
    x_surface=rotate(x_surface,2)
    x_surf_norm=x_surf_norm*[1.0,-1.0]
  endif

  ;;========================================------------------------------
  ;; Rotating arrays:

  rangle=!dtor*(alpha-90.0)

  x_grat_norm1= cos(rangle)*x_grat_norm+sin(rangle)*y_grat_norm
  y_grat_norm1=-sin(rangle)*x_grat_norm+cos(rangle)*y_grat_norm

  x_surf_norm1= cos(rangle)*x_surf_norm+sin(rangle)*y_surf_norm1
  y_surf_norm1=-sin(rangle)*x_surf_norm+cos(rangle)*y_surf_norm1

  x_surface1  = cos(rangle)*x_surface  +sin(rangle)*y_surface
  y_surface1  =-sin(rangle)*x_surface  +cos(rangle)*y_surface

  x_backside1 = cos(rangle)*x_backside +sin(rangle)*y_backside
  y_backside1 =-sin(rangle)*x_backside +cos(rangle)*y_backside

  plots,x_surface1,y_surface1,color=red,thick=2
  plots,x_backside1,y_backside1,color=green,thick=2
  plots,x_grat_norm1,y_grat_norm1,lines=3,color=green
  plots,x_surf_norm1,y_surf_norm1,lines=1,color=red

  if eta lt 45.0 then begin
    plots,[-plotrange,0,-plotrange],[0,0,plotrange*tan(!dtor*eta)]
  endif else begin
    plots,[-plotrange,0,-plotrange/tan(!dtor*eta)],[0,0,plotrange]
  endelse

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
                        verbose=verbose,/error)
  return
END ;;; procedure: p3d_display_grat_plot
