;+
; NAME:
;         p3d_display_telescl_postable
;
;         $Id: p3d_display_telescl_postable.pro 83 2010-03-05 11:33:06Z olestreicher $
;
; PURPOSE:
;         This display tool shows a spatial map of all elements of an IFU. It
;         is necessary to specify a fiber position table.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: display routine
;
; CALLING SEQUENCE:
;         p3d_display_telescl_postable,vector,xsize,ysize,orientation=, $
;             indicatecurrent=,locut=,hicut=,contours=,posfile=, $
;             rownum=,xpos=,ypos=,lens_size=,science=,sky=,shape=,nvertices=, $
;             plotmarks=,marks=,arrow=,cindex1=,cindex2=,bottom=,noerase=, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         vector          - A one-dimensional array of floating point values,
;                           which is used to calculate the color of every
;                           individual spatial element.
;         xsize           - A scalar integer specifying the x-size of the plot.
;
; OPTIONAL INPUTS:
;         ysize [xsize]   - A scalar integer specifying the y-size of the plot.
;
; KEYWORD PARAMETERS:
;         orientation [1] - A scalar integer specifying how the data is
;                           oriented. The following values are recognized:
;                             0:: up-E, left-S
;                             1:: up-N, left-E (default)
;                             2:: up-W, left-N
;                             3:: up-S, left-W
;         indicatecurrent - A one- or two-element array that indicates if the
;                           currently selected element should be marked. This
;                           parameter is only used if PLOTMARKS==0. If
;                           INDICATECURRENT is a two-element array then the
;                           second element is re-drawn, the first element is
;                           always marked.
;         locut           - A decimal scalar that is used as the lower cut in
;                           scaling colors. If neither LOCUT nor HICUT is set
;                           then these are set as the minimum and maximum
;                           values of VECTOR.
;         hicut           - A decimal scalar. For further information, see
;                           LOCUT.
;         contours        - If this keyword is set then a border is drawn
;                           around every element.
;         posfile         - is passed to p3d_misc_read_postable.
;         rownum          - A one-dimensional array of integer type, which
;                           specifies the elements in VECTOR which are used to
;                           scale colors of individual elements.
;         xpos            - A one-dimensional array of floating point type,
;                           that specifies the x-position of every element;
;                           this is the lower left corner for square shaped
;                           elements (SHAPE==0) and the center position of
;                           circular and hexagon elements (SHAPE==1||2||3).
;         ypos            - A one-dimensional array of floating point type,
;                           which specifies the y-position of every element;
;                           this is the lower left corner for square shaped
;                           elements (SHAPE==0) and the center position of
;                           circular and hexagon elements (SHAPE==1||2||3).
;         lens_size       - A decimal scalar specifying the size of circular
;                           elements; specified in the same units as XPOS and
;                           YPOS.
;         science,sky     - are passed to p3d_misc_read_postable.
;         shape [0]       - A scalar integer specifying the symbol shape:
;                             0:: square
;                             1:: circle
;                             2:: hexagon; with a vertice at angle=0deg.
;                             3:: hexagon; with a vertice at angle=30deg.
;         nvertices [15]  - The number of vertices used with circular symbols.
;         plotmarks [0]   - A scalar integer specifying if every spaxel marked
;                           in MARKS should be overplotted with a wireframe;
;                           0||1.
;         marks           - A one-dimensional array of integers specifying
;                           which spatial elements should be overplotted with a
;                           wireframe, the number of elements must be the same
;                           as in XPOS, YPOS. MARKS is used if PLOTMARKS==1.
;         arrow           - If this keyword is set, then a directional arrow
;                           is drawed on top in the following corner:
;                             orientation==0:: lower  left corner
;                             orientation==1:: lower right corner
;                             orientation==2:: upper right corner
;                             orientation==3:: upper  left corner
;         cindex1 [0]     - A color index used with wireframe overplots.
;         cindex2 [1]     - A color index used with wireframe overplots.
;         bottom [0]      - The lower index used in the color table.
;         noerase         - If this keyword is set the plot window is not
;                           erased prior to drawing a new plot.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         none
;
; COMMON BLOCKS:
;         p3d_display_telescl_postable_definearrow_common, $
;              arrowx,arrowy,narrow
;                           This common block holds the variable defining the
;                           N-E vector.
;         p3d_display_telescl_postable_definesymbol_common, $
;              cix,ciy,cix2,ciy2,savex,savey,savep
;                           This common block holds variables related to the
;                           'currently select symbol (a disk), and the plotting
;                           state.
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         06.11.2008 - Converted from the original routine telescl_postable  
;                      of Thomas Becker. /CS
;-
PRO p3d_display_telescl_postable_definearrow,arrowx_,arrowy_,narrow_
  compile_opt hidden,IDL2

  common p3d_display_telescl_postable_definearrow_common,arrowx,arrowy,narrow
  common p3d_display_telescl_postable_definesymbol_common,cix,ciy,cix2,ciy2, $
                                                          savex,savey,savep

  ;; Defining the N-E vector arrow:
  arrowx=[0.000,0.15,0.15,0.95,0.95,1.00,0.925,0.85,0.90,0.90,0.15,0.15,0.000]
  arrowy=[0.075,0.00,0.05,0.05,0.85,0.85,1.000,0.85,0.85,0.10,0.10,0.15,0.075]

  narrow=n_elements(arrowx)
  narrow_=narrow & arrowx_=arrowx & arrowy_=arrowy

  arr=findgen(10L)/(10L-1L)*(2*!PI)
  cix =0.5*sin(arr)*0.8
  ciy =0.5*cos(arr)*0.8
  cix2=cix*0.5
  ciy2=ciy*0.5

  return
END ;;; procedure: p3d_display_telescl_postable_definearrow

PRO p3d_display_telescl_postable,vector__,xsize,ysize, $
        orientation=orientation,indicatecurrent=indicatecurrent, $
        locut=locut,hicut=hicut,contours=contours, $
        posfile=posfile,rownum=rownum,xpos=xpos__,ypos=ypos__, $
        lens_size=lens_size_,science=science,sky=sky,shape=shape, $
        nvertices=nvertices,plotmarks=plotmarks,marks=marks,single=single, $
        arrow=arrow,cindex1=cindex1,cindex2=cindex2,bottom=bottom, $
        noerase=noerase,topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  common p3d_display_telescl_postable_definearrow_common,arrowx,arrowy,narrow
  common p3d_display_telescl_postable_definesymbol_common,cix,ciy,cix2,ciy2, $
                                                          savex,savey,savep

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_display_telescl_postable: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_display_telescl_postable'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(vector__)
  if s[0L] ne 1L or $
    (s[s[0l]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='VECTOR [1] must be set; to a one-dimensional array of floating' + $
           ' point type.'
    goto,error_handler
  endif
  vector=float(vector__)
  nvector=s[s[0L]+2L]

  s=size(xsize)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0l]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='XSIZE [2] must be set to a decimal scalar.'
    goto,error_handler
  endif

  if ~n_elements(ysize) then ysize=xsize
  s=size(ysize)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0l]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='YSIZE [3] must, if set, be a decimal scalar.'
    goto,error_handler
  endif

  if ~n_elements(orientation) then orientation=1
  sb=size(orientation)
  if sb[0L] ne 0L or $
    (sb[sb[0L]+1L] ge 4L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='ORIENTATION must be a scalar integer; 0<=ORIENTATION<=3.'
    goto,error_handler
  endif
  if orientation lt 0 or orientation gt 3 then begin
    errmsg='ORIENTATION must be a scalar integer; 0<=ORIENTATION<=3.'
    goto,error_handler
  endif

  if ~n_elements(locut) and ~n_elements(hicut) then begin
    locut=min(vector,max=hicut)
  endif else begin
    s=size(locut)
    if s[s[0L]+2L] ne 1L or $
      (s[s[0l]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
      errmsg='LOCUT [2] must be set; to a decimal scalar.'
      goto,error_handler
    endif

    s=size(hicut)
    if s[s[0L]+2L] ne 1L or $
      (s[s[0l]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
      errmsg='HICUT [3] must be set; to a decimal scalar.'
      goto,error_handler
    endif
  endelse

  ;; Reading the element position table if necessary values are missing:
  if ~n_elements(rownum) or ~n_elements(xpos__) or ~n_elements(ypos__) or $
     ~n_elements(lens_size_) then begin
    p3d_misc_read_postable,posfile=posfile,rownum=rownum, $
        xpos=xpos__,ypos=ypos__,lens_size=lens_size,science=science,sky=sky, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return
  endif else lens_size=lens_size_

  sr=size(rownum)
  if sr[0L] ne 1L or $
    (sr[sr[0l]+1L] ge 4L and sr[sr[0L]+1L] le 11L) then begin
    errmsg='ROWNUM must be set to a one-dimensional array of integers.'
    goto,error_handler
  endif

  s=size(xpos__)
  if s[0L] ne 1L or $
    (s[s[0l]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='XPOS must be set to a one-dimensional array of floating point' + $
           ' type.'
    goto,error_handler
  endif
  xpos=xpos__

  sb=size(ypos__)
  if sb[0L] ne 1L or (sb[sb[0l]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='YPOS must be set to a one-dimensional array of floating point' + $
           ' type.'
    goto,error_handler
  endif
  ypos=ypos__
  nspaxel=s[s[0L]+2L]

  if s[s[0L]+2L] ne sb[sb[0L]+2L] or s[s[0L]+2L] ne sr[sr[0L]+2L] then begin
    errmsg='XPOS ['  +strtrim(s [s [0L]+2L],2L)+'], ' + $
           'YPOS ['  +strtrim(sb[sb[0L]+2L],2L)+'] and ' + $
           'ROWNUM ['+strtrim(sr[sr[0L]+2L],2L)+'] ' + $
           'must have the same number of elements.'
    goto,error_handler
  endif

  if ~n_elements(plotmarks) then plotmarks=0L
  s=size(plotmarks)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0l]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='PLOTMARKS must be set to an integer scalar; 0||1.'
    goto,error_handler
  endif
  if plotmarks ne 0L and plotmarks ne 1L then begin
    errmsg='PLOTMARKS must be set to an integer scalar; 0||1, not "'+ $
           strtrim(plotmarks,2L)+'".'
    goto,error_handler
  endif

  nindicatecurrent=n_elements(indicatecurrent)
  currentcleanplot=1L
  if plotmarks then begin
    s=size(marks)
    if s[s[0L]+2L] ne sb[sb[0L]+2L] or $
      (s[s[0l]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
      errmsg='MARKS must be set to an array of integers with the same numb' + $
             'er of elements as in XPOS; value==0||1.'
      goto,error_handler
    endif
    if min(marks) lt 0L or max(marks) gt 1L then begin
      errmsg='MARKS must be set to an array of integers with the same numb' + $
             'er of elements as in XPOS; value==0||1.'
      goto,error_handler
    endif
  endif else begin
    if nindicatecurrent gt 2L then begin
      errmsg='INDICATECURRENT must have at most two elements.'
      goto,error_handler
    endif

    currentcleanplot=nindicatecurrent le 1L
    if nindicatecurrent gt 1L then begin
      if indicatecurrent[1L] eq -1L then begin
        currentcleanplot=0L
        indicatecurrent=[indicatecurrent[0L]]
        nindicatecurrent=1L
      endif
    endif

;    if max(indicatecurrent) gt max(rownum) or $
;       min(indicatecurrent) lt min(rownum) then begin
;print,'"'+strtrim(indicatecurrent,2)+'"'
;print,minmax(rownum)
;      errmsg='The value(s) of INDICATECURRENT must be within the range o' + $
;             'f ROWNUM.'
;      goto,error_handler
;      endif
  endelse

  if ~n_elements(cindex1) then cindex1=0L
  if ~n_elements(cindex2) then cindex2=1L
  cindex1=cindex1>0L<(!d.table_size-1L)
  cindex2=cindex2>0L<(!d.table_size-1L)

  if ~n_elements(bottom) then bottom=0L
  s=size(bottom)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0l]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='BOTTOM must be set to an integer scalar; >=0.'
    goto,error_handler
  endif
  if bottom gt !d.table_size-1L then begin
    errmsg='BOTTOM ['+strtrim(bottom,2L)+'] is larger than !D.TABLE_SIZE ['+ $
        strtrim(!d.table_size,2L)+'].'
    goto,error_handler
  endif

  noerase=keyword_set(noerase)

  s=size(single)
  if s[s[0L]+2L] ne 0L then begin
    if s[s[0L]+2L] ne 1L or $
      (s[s[0l]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
      errmsg='SINGLE, if set, must be an integer scalar.'
      goto,error_handler
    endif
    if single lt min(rownum) or single gt max(rownum) then begin
      errmsg='SINGLE ('+strtrim(single,2L)+'), must be in the span of ROW' + $
             'NUM ('+string(format='(i5,"-",i5)',min(rownum),max(rownum))+').'
    endif
  endif

  if ~n_elements(shape) then shape=0L
  s=size(shape)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0l]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='SHAPE must be set to an integer scalar; 0<=shape<=3.'
    goto,error_handler
  endif
  if shape lt 0L or shape gt 3L then begin
    errmsg='SHAPE must be set to an integer scalar; 0<=shape<=3, not "'+ $
           strtrim(shape,2L)+'".'
    goto,error_handler
  endif

  ;; Extra settings for circular and hexagon elements:
  if ~(shape-1L) then begin
    if ~n_elements(nvertices) then nvertices=15L
    s=size(nvertices)
    if s[s[0L]+2L] ne 1L or $
      (s[s[0l]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
      errmsg='NVERTICES must be set to a scalar integer.'
      goto,error_handler
    endif
  endif else if ~(shape-2L) or ~(shape-3L) then begin
    if ~n_elements(nvertices) then nvertices=6L
    s=size(nvertices)
    if s[s[0L]+2L] ne 1L or $
      (s[s[0l]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
      errmsg='NVERTICES must be set to a scalar integer.'
      goto,error_handler
    endif
  endif

  charsize=1.5 ;; temporary solution

  if ~n_elements(arrowx) then $
     p3d_display_telescl_postable_definearrow,arrowx,arrowy,narrow

  ;;========================================------------------------------
  ;; Setting up the symbol arrays, and the x- and y-range arrays:

  xposmin=min(xpos,max=xposmax)
  yposmin=min(ypos,max=yposmax)

  switch shape of
    1L: begin
      ;; Circular symbols:
      arr=findgen(nvertices)/nvertices*2*!PI
      x=lens_size*0.5*sin(arr)
      y=lens_size*0.5*cos(arr)

      if plotmarks then begin
        tmp=0.45d0/sqrt(2d0)*lens_size[0L]
        wx2=[-tmp,-tmp,tmp, tmp]
        wy2=[-tmp, tmp,tmp,-tmp]
        wx =wx2*0.5
        wy =wy2*0.5
      endif
      minoffset=lens_size*0.5 & maxoffset=minoffset
      break
    end
    2L:
    3L: begin
      ;; Hexagon-shaped symbols:
      arr=findgen(nvertices)/nvertices*2*!PI+(~(shape-3L)?!PI/6.0:0.0)
      x=lens_size*0.5*sin(arr)
      y=lens_size*0.5*cos(arr)

      if plotmarks then begin
        tmp=0.45d0/sqrt(2d0)*lens_size[0L]
        wx2=[-tmp,-tmp,tmp, tmp]
        wy2=[-tmp, tmp,tmp,-tmp]
        wx =wx2*0.5
        wy =wy2*0.5
      endif

      minoffset=lens_size*0.5 & maxoffset=minoffset
      break
    end
    else: begin
      ;; Quadratic symbols:
      x=[0.0,1.0,1.0,0.0,0.0]
      y=[0.0,0.0,1.0,1.0,0.0]

      if plotmarks then begin
        wx2=[0.05,0.05,0.95,0.95]
        wy2=[0.05,0.95,0.95,0.05]
        wx =[0.25,0.25,0.75,0.75]
        wy =[0.25,0.75,0.75,0.25]
      endif

      minoffset=0.0 & maxoffset=1.0
    end
  endswitch ;; shape

  xrange=[min(xposmin-minoffset),max(xposmax+maxoffset)]
  yrange=[min(yposmin-minoffset),max(yposmax+maxoffset)]

  ;; Swapping the [x:West->East, y:South->North] coordinates to get
  ;;   [x:South->North, y:West->East]:

  tmp_xrange=xrange
  tmp_xpos  =xpos
  xrange=    yrange & xpos=    ypos
  yrange=tmp_xrange & ypos=tmp_xpos

  ;;========================================------------------------------
  ;; Correcting for an aspect /= 1.0:

  xxoff=0.0 & xyoff=0.0
  xran=(xrange[1L]-xrange[0L])
  yran=(yrange[1L]-yrange[0L])
  if yran/xran lt 1.0 then begin
    xyoff=0.5*(xran-yran)
    yrange=[yrange[0L]-xyoff,yrange[1L]+xyoff]
  endif else if yran/xran gt 1.0 then begin
    xxoff=0.5*(yran-xran)
    xrange=[xrange[0L]-xxoff,xrange[1L]+xxoff]
  endif

  ;;========================================------------------------------
  ;; Setting the orientation:

  case orientation of
    0: begin ;; North is right, East is up:
      used_xpos=xpos & used_xrange=xrange
      used_ypos=ypos & used_yrange=yrange
    end
    2: begin ;; North is left, East is down:
      used_xpos=xpos & used_xrange=[xrange[1L],xrange[0L]]
      used_ypos=ypos & used_yrange=[yrange[1L],yrange[0L]]
    end
    3: begin ;; North is down, East is right:
      used_xpos=ypos & used_xrange=yrange
      used_ypos=xpos & used_yrange=[xrange[1L],xrange[0L]]
    end
    else: begin ;; Default: North is up, East is left:
      used_xpos=ypos & used_xrange=[yrange[1L],yrange[0L]]
      used_ypos=xpos & used_yrange=xrange
    end
  endcase ;; case: orientation

  ;;========================================------------------------------
  ;; Setting up the plot window:

  device,get_decomposed=decomposed
  if decomposed then device,decomposed=0L

  color=(vector[rownum]-locut)/(hicut-locut)*(!d.table_size-bottom)+bottom
  color=(color<(!d.table_size-1L))>bottom

  if (~n_elements(single) and plotmarks) or $
     (~plotmarks and currentcleanplot) then begin
    plot,[0],/nodata,noerase=noerase,/device,color=0L, $
         xstyle=1L,xrange=used_xrange, $
         ystyle=1L,yrange=used_yrange,position=[0.0,0.0,xsize,ysize]

    savex=!x
    savey=!y
    savep=!p

    ;; Plotting every element separately:
    for k=0L,nspaxel-1L do $
       polyfill,used_xpos[k]+x,used_ypos[k]+y,color=color[k]

    ;; Optionally over-plotting contours:
    if keyword_set(contours) then $
       for k=0L,nspaxel-1L do oplot,used_xpos[k]+x,used_ypos[k]+y

    if plotmarks then begin
      for k=0L,nspaxel-1L do begin
        if marks[k] then begin
          polyfill,used_xpos[k]+wx2,used_ypos[k]+wy2,color=cindex1
          polyfill,used_xpos[k]+wx ,used_ypos[k]+wy ,color=cindex2
        endif
      endfor
    endif

  endif else begin
    !x=savex
    !y=savey
    !p=savep
  endelse ;; ~n_elements(single)


  if n_elements(single) ne 0L then begin

    ;;==========-----
    ;; Optionally overplotting a single wireframe:

    if marks[single] then begin
      ;; Over-plotting a polygon:
      polyfill,used_xpos[single]+wx2,used_ypos[single]+wy2,color=cindex1
      polyfill,used_xpos[single]+wx ,used_ypos[single]+wy ,color=cindex2
    endif else begin
      ;; Re-plotting the polygon:
      polyfill,used_xpos[single]+ x ,used_ypos[single]+ y ,color=color[single]
    endelse
  endif ;; n_elements(single) ne 0L


  if nindicatecurrent gt 0L and ~plotmarks then begin

    ;;==========-----
    ;; Optionally plotting the current element:

    index=indicatecurrent[0L]
    offset=~shape?0.5:0.0
    polyfill,used_xpos[index]+offset+cix *lens_size[index], $
             used_ypos[index]+offset+ciy *lens_size[index],color=cindex1
    polyfill,used_xpos[index]+offset+cix2*lens_size[index], $
             used_ypos[index]+offset+ciy2*lens_size[index],color=cindex2

    ;; "deleting" the previous symbol:
    if nindicatecurrent eq 2L then begin
      index=indicatecurrent[1L]
      polyfill,used_xpos[index]+offset+cix*lens_size[index], $
               used_ypos[index]+offset+ciy*lens_size[index],color=color[index]
    endif
  endif ;; nindicatecurrent ne 0L

  ;;========================================------------------------------
  ;; Optionally drawing a directional double-headed arrow -
  ;;   with an indicator of the North and East directions:

  if keyword_set(arrow) then begin
    scalefactor=0.25 & offset=0.01

    case orientation of
      0: begin
        mrot=[[0,1],[-1,0]]
        arrow=transpose([transpose(arrowx),transpose(arrowy)])
        arrowx__=fltarr(narrow) & arrowy__=fltarr(narrow)
        for i=0L,narrow-1L do begin
          tmp=mrot##arrow[i,*]
          arrowx__[i]=tmp[0L]
          arrowy__[i]=tmp[1L]
        endfor
        aposx= arrowx__     *scalefactor+    offset
        aposy=(arrowy__+1.0)*scalefactor+    offset

        nalign=0.0
        nposx=1.0*scalefactor+offset & nposy=0.1*scalefactor+offset
        eposx=0.1*scalefactor+offset & eposy=1.0*scalefactor+offset
      end
      2: begin
        mrot=[[0,-1],[1,0]]
        arrow=transpose([transpose(arrowx),transpose(arrowy)])
        arrowx__=fltarr(narrow) & arrowy__=fltarr(narrow)
        for i=0L,narrow-1L do begin
          tmp=mrot##arrow[i,*]
          arrowx__[i]=tmp[0L]
          arrowy__[i]=tmp[1L]
        endfor
        aposx=(arrowx__    )*scalefactor+1.0-offset
        aposy=(arrowy__-1.0)*scalefactor+1.0-offset

        nalign=1.0
        nposx=1.0-1.0*scalefactor-offset
        nposy=1.0-0.1*scalefactor-offset-!d.y_ch_size/float(!d.y_size)*charsize
        eposx=1.0-0.1*scalefactor-offset
        eposy=1.0-1.0*scalefactor+offset-!d.y_ch_size/float(!d.y_size)*charsize
      end
      3: begin
        mrot=[[-1,0],[0,-1]]
        aposx=(-arrowx+1.0)*scalefactor+    offset
        aposy=(-arrowy    )*scalefactor+1.0-offset

        nalign=0.0
        nposx=0.1*scalefactor+offset
        nposy=1.0-1.0*scalefactor+offset-!d.y_ch_size/float(!d.y_size)*charsize
        eposx=1.0*scalefactor+offset
        eposy=1.0-0.1*scalefactor-offset-!d.y_ch_size/float(!d.y_size)*charsize
      end
      else: begin
        arrowx__=arrowx
        arrowy__=arrowy
        aposx=(arrowx__-1.0)*scalefactor+1.0-offset
        aposy= arrowy__     *scalefactor+    offset

        nalign=1.0
        nposx=1.0-0.1*scalefactor-offset
        nposy=1.0*scalefactor+offset
        eposx=1.0-1.0*scalefactor-offset
        eposy=0.1*scalefactor+offset
      end
    endcase ;; orientation

    polyfill,aposx,aposy,/normal
    plots,aposx,aposy,/normal,color=0L

    xyouts,nposx,nposy,'N',alignment=nalign,/normal,charsize=charsize, $
           charthick=charsize*2.5,color=0L
    xyouts,nposx,nposy,'N',alignment=nalign,/normal,charsize=charsize, $
           charthick=charsize*1.2
    xyouts,eposx,eposy,'E',alignment=nalign,/normal,charsize=charsize, $
           charthick=charsize*2.5,color=0L
    xyouts,eposx,eposy,'E',alignment=nalign,/normal,charsize=charsize, $
           charthick=charsize*1.2

  endif

  if decomposed then device,decomposed=decomposed

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_display_telescl_postable
