;+
; NAME:
;         p3d_flatfield
;
;         $Id: p3d_flatfield.pro 83 2010-03-05 11:33:06Z olestreicher $
;
; PURPOSE:
;         This routine prepares, and optionally normalizes, raw flat field
;         data. All elements operated upon are ensured to be >0.
;
;         If NORMALIZE is set the data is normalized:
;
;         1) First across the cross-dispersion axis (with the average of all
;           spectra). This corrects for a varying element-to-element 
;           throughput.
;         2) Secondly with the entire array in order to correct for wavelength-
;           dependent differences. If SMOWID is set the data is smoothed across
;           the dispersion-axis for each spectrum separately, using the box-car
;           width 2*SMOWID+1. If DEG is set then every spectrum is replaced with
;           a polynomial fit of the data of order DEG.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: flat field creation
;
; CALLING SEQUENCE:
;         p3d_flatfield,in,out,dout,din=,spec=,/normalize,smowid=,deg=, $
;             daxis=,topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         in              - A two-dimensional array of decimal type.
;
; KEYWORD PARAMETERS:
;         din             - An array of the same type and dimensions as IN;
;                           containing the error of IN.
;         spec            - A one-dimensional array that upon exit holds the
;                           mean spectrum of IN.
;         normalize [0]   - If this keyword is set then the data is normalized.
;         smowid [0]      - A scalar integer specifying the width of a
;                           smoothing box, which is used across the data in the
;                           dispersion direction. Only used if /NORMALIZE.
;         deg [0]         - A scalar integer specifying the order of a
;                           polynomial that is used to fit and then replace the
;                           normalized data. Only used if /NORMALIZE.
;         daxis [1]       - Defines the dispersion direction dimension of
;                           IMAGE. The default, 1, is in the x-direction.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - A two-dimensional array of decimal type. In
;                           comparison to IN the data has been masked from
;                           values <0.0 and, optionally, normalized and
;                           smoothed.
;
; OPTIONAL OUTPUTS:
;         dout            - An array of the same type and dimensions as OUT.
;                           Only returned if DIN is present.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         05.11.2008 - Based on the routine p3d_flat of Thomas Becker. /CS
;-
PRO p3d_flatfield,in,out,dout,din=din,spec=spec,normalize=normalize, $
        smowid=smowid_,deg=deg_,daxis=daxis,topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_flatfield: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_flatfield'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:
                  
  if ~n_elements(daxis) then daxis=1L
  sd=size(daxis)
  if sd[sd[0L]+2L] ne 1L or $
    (sd[sd[0L]+1L] ge 4L and sd[sd[0L]+1L] le 11L) then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  sid=daxis?2L:1L

  s=size(in)
  if ~s[s[0L]+2L] or s[0L] ne 2L or $
     (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='IN [1] must be set; to a two-dimensional array of floating poi' + $
           'nt type.'
    goto,error_handler
  endif
  flat=in
  if ~daxis then flat=transpose(flat)

  se=size(din) & ecalc=0L
  if se[se[0L]+2L] ne 0L then begin
    if se[0L] ne s[0L] or se[se[0L]+1L] ne s[s[0L]+1L] or $
       se[se[0L]+2L] ne s[s[0L]+2L] then begin
      errmsg='DIN, if set, must be of the same type and dimensions as IN.'
      goto,error_handler
    endif
    dflat=din
    if ~daxis then dflat=transpose(dflat)
    ecalc=1L
  endif ;; se[se[0L]+2L] ne 0L

  normalize=keyword_set(normalize)
  sb=size(normalize)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0L]+1L] ge 4L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='NORMALIZE must be a scalar integer; 0||1.'
    goto,error_handler
  endif
  if normalize ne 1L and normalize ne 0L then begin
    errmsg='NORMALIZE must be a scalar integer; 0||1.'
    goto,error_handler
  endif

  if normalize then begin
    if ~n_elements(smowid_) then smowid_=0L
    sb=size(smowid_)
    if sb[sb[0L]+2L] ne 1L or $
      (sb[sb[0L]+1L] ge 4L and sb[sb[0L]+1L] le 11L) then begin
      errmsg='SMOWID must be a scalar integer; >=0.'
      goto,error_handler
    endif
    if smowid_ lt 0L then begin
      errmsg='SMOWID must be a scalar integer; >=0.'
      goto,error_handler
    endif
    smowid=long(smowid_)

    if ~n_elements(deg_) then deg_=0L
    sb=size(deg_)
    if sb[sb[0L]+2L] ne 1L or $
      (sb[sb[0L]+1L] ge 4L and sb[sb[0L]+1L] le 11L) then begin
      errmsg='DEG must be a scalar integer; >=0.'
      goto,error_handler
    endif
    if deg_ lt 0L then begin
      errmsg='DEG must be a scalar integer; >=0.'
      goto,error_handler
    endif
    deg=long(deg_)
  endif ;; normalize


  ;;========================================------------------------------
  ;; Normalizing the data:

  mask=flat gt 0d0
  msg=['Masking the data of elements which are <0.']
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  if normalize then begin

    msg='[Normalize 1/1] Cross-dispersion axis.'
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

    ;;==============================--------------------
    ;; Averaging the data across the cross-dispersion axis; correcting the data
    ;;   for varying throughput fiber-to-fiber:

    spec=float(total(flat*mask,sid,/double)/(total(mask,sid,/double)>1d0))


    ;; Logging information (loglevel=2):
    msg=['   i) Normalizing the flat field for varying fiber-to-fiber thro' + $
         'ughput.']
    nextstr='  ii) '
    if smowid gt 0L then begin
       msg=[msg,'  ii) Smoothed every spectrum with a box-car of width 2*S' + $
                'MOWID+1='+strtrim(2L*smowid+1L,2L)+'.']
       nextstr=' iii) '
    endif
    if deg gt 0L then begin
       msg=[msg,nextstr+'Replaced every spectrum with a fitted polynomial ' + $
            'of order DEG='+strtrim(deg,2L)+'.']
    endif


    ;;==============================--------------------
    ;; Looping over all spectra:

    k=-1L & while ++k lt s[sid] do begin
      pos=where(mask[*,k],count)
      if count ne 0L then begin
        flat[pos,k]/=spec[pos]
        if ecalc then dflat[pos,k]/=spec[pos]

        ;;==============================--------------------
        ;; Smoothing the data across the dispersion axis with a box-car:

        if smowid gt 0L then begin
          p3d_misc_smooth_1d,flat[pos,k],smowid,tmp,topwid=topwid, $
              logunit=logunit,verbose=verbose,error=error,debug=debug
          if error ne 0 then return
          flat[pos,k]=tmp

          if ecalc then begin
            p3d_misc_smooth_1d,dflat[pos,k],smowid,tmp,topwid=topwid, $
                logunit=logunit,verbose=verbose,error=error,debug=debug
            if error ne 0 then return
            dflat[pos,k]=tmp
          endif ;; ecalc
        endif ;; smowid gt 0L

        ;;==============================--------------------
        ;; Fitting and replacing the data with a polynomial:

        if deg gt 0L then begin
          par=poly_fit(pos,flat[pos,k],deg,/double)
          flat[*,k]=0d0 & for L=0L,deg do flat[pos,k]+=par[L]*double(pos)^L

          if ecalc then begin
            par=poly_fit(pos,dflat[pos,k],deg,/double)
            dflat[*,k]=0d0 & for L=0L,deg do dflat[pos,k]+=par[L]*double(pos)^L
          endif ;; ecalc
        endif ;; deg gt 0L

      endif ;; count ne 0L
    endwhile ;; ++k lt s[sid]

    error=p3d_misc_logger(msg,logunit,loglevel=2L,rname=rname, $
        verbose=verbose ge 2)

  endif ;; normalize


  ;;========================================------------------------------
  ;; Creating the flat field data set:

  out=flat*mask
  if ecalc then dout=dflat*mask


  ;;========================================------------------------------
  ;; Normalizing the data a second time with the average of the full array:

  if normalize then begin
    pos=where(mask,count)
    tmp=total(out[pos])/count
    out[pos]/=tmp & if ecalc then dout[pos]/=tmp

    msg=[msg, $
         '[Normalize 2/2] Dividing the flat field with the mean of the ful' + $
         'l array.']
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
  endif ;; normalize

  ;;==============================--------------------
  ;; Transposing the output data if daxis==2:

  if ~daxis then begin
    out=transpose(out)
    if ecalc then dout=transpose(dout)
  endif ;; ~daxis

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_flatfield
