;+
; NAME:
;         p3d_gui
;
;         $Id: p3d_gui.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         This is the p3d GUI core routine. This routine should not be used by
;         itself, but is called by the wrapper p3d (in p3d.pro).
;
; AUTHOR:
;         Christer Sandin and Joris Gerssen
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: main routine
;
; CALLING SEQUENCE:
;         p3d_gui,parfile,cwd=,path=,title=,detector=,userparfile=, $
;             displib=,box=,colortable=,bottom=,cindex=,/tracking_events, $
;             logfile=,loglevel=,/mpfit,verbose=,error=,/debug,/help, $
;             /restart
;
; INPUTS:
;         parfile         - A string specifying the filename of an existing
;                           file containing instrument-specific parameters,
;                           which are used by p3d to perform semi-automatic
;                           reductions.
;
; KEYWORD PARAMETERS:
;         cwd             - This keyword can be set in two ways:
;                           o If CWD is a scalar string then the string is
;                             assumed to be the path where all data is saved.
;                           o If CWD is used as /CWD (i.e. an integer==1) then
;                             the current directory is used to save all output
;                             data.
;         path            - A scalar string that specifies where the raw data
;                           is placed. If it is set then the data path is set
;                           to PATH (after checking if PATH is a directory). If
;                           PATH is not set then the data path is set to
;                           !p3d_data_path.
;         title           - A scalar string that is used in the title of the
;                           p3d GUI.
;         detector        - This scalar integer specifies the detector to start
;                           with for instruments which have more than one (such
;                           as, e.g., VIMOS or SPIRAL).
;         userparfile     - A scalar string that specifies the name of an
;                           optional file with user parameters. This file is
;                           used by various routines in the p3d package (please
;                           see the separate routines for its use).
;         displib         - N/A.
;         box             - If this keyword is set then p3d is setup to use
;                           shift boxes (NOTE! This option is currently not
;                           implemented so far that it can be used).
;         colortable      - A scalar integer that specifies the colortable to
;                           use. If COLORTABLE==-1 then the "Sauron colormap"
;                           is used (cf. sauron_colormap.pro). If
;                           0<=COLORTABLE<=40 then the corresponding IDL
;                           standard colortable is used. If COLORTABLE is a
;                           string, then the file with that filename is loaded
;                           by loadct (cf. p3d_misc_colortable).
;         bottom          - A scalar integer that specifies the lower index of
;                           the colortable that is used to scale the colormap
;                           (the upper index is the maximum possible; cf.
;                           p3d_misc_colortable.pro).
;         cindex          - An array of integers specifying which color indices
;                           to use for reserved colors. It doesn't make sense
;                           to use more than 4 elements in the CINDEX array.
;                           The indices should also be set as low as possible
;                           in order to allow the remaining indices to be used
;                           by the (scaled) COLORTABLE
;                           (cf. p3d_misc_colortable).
;         tracking_events [1]
;                         - If this keyword is set then (some) p3d tools, which
;                           have a GUI, show a status line with updated
;                           information on what various widgets do, and what is
;                           going on.
;         logfile         - A scalar string that specifies the name of a
;                           logfile. If a file already exists, with the name
;                           LOGFILE then it is overwritten. Use this keyword to
;                           save the output of the various routine of p3d. In
;                           order to only print information at the prompt, use
;                           VERBOSE instead. 
;         loglevel        - A scalar integer that specifies the level of
;                           messages that should be recorded in a logfile
;                           LOGFILE. LOGLEVEL can be set to 1-3, where 3
;                           results in most information being written to the
;                           logfile.
;         mpfit           - If this keyword is set then it is assumed that the
;                           fitting routines mpfit.pro and mpcurvefit.pro (cf.
;                           http://purl.com/net/mpfit) exist in the path. These
;                           routines are then used in various routines of p3d
;                           where curve fitting is required. (MPFIT has more
;                           options to control than, e.g., GAUSSFIT has.)
;         verbose [0]     - Set this variable to 0<=VERBOSE<=3 to provide
;                           different levels of output to the prompt
;                           (VERBOSE==0 results in no output at all). In order
;                           to save the messages, use the LOGFILE and LOGLEVEL
;                           keywords instead.
;         error           - Returns an error code if set. A 0 is returned if
;                           operations are successful.
;         debug           - If this keyword is set (as /DEBUG) then no error
;                           handlers are setup. This is useful when debugging
;                           the code.
;         help            - If this keyword is set (as /HELP) then this routine
;                           documentation is shown, and the routine exits.
;         restart         - This keyword is used by p3d internally if p3d is
;                           restarted (with the menu option).
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         Requires IDL version 6.2 or better.
;
; MODIFICATION HISTORY:
;         03.09.2009 - All reduction operation tasks have been moved to their
;                      own routines. /CS
;         08.11.2008 - The is a converted version of the original routine
;                      p3d_online_extended of Thomas Becker and Petra Bhm. /CS
;-
FUNCTION p3d_gui_readdr,logfile,offset=offset
  compile_opt hidden,IDL2

  if n_elements(offset) eq 0L then offset=0L

  ;; Opening the file:
  openr,unit,logfile,/get_lun

  ;; Skipping forward OFFSET lines:
  point_lun,unit,offset

  ;; Reading as many rows of text as is possible:
  tmp=''
  readf,unit,tmp
  text=[tmp]
  while ~eof(unit) do begin
    tmp=''
    readf,unit,tmp
    text=[text,tmp]
  endwhile
  point_lun,-unit,offset

  ;; Closing the file:
  free_lun,unit

  return,text
END ;; function: p3d_gui_readdr


PRO p3d_gui_timerdr,state,event
  compile_opt hidden,IDL2

  ;; Re-reading the log file, if the log file window is open:
  if widget_info((*state).logfiletid,/valid_id) then begin
    if (*state).logfileoffset gt 0L then begin

      ;; Only reading entire lines (re-reading the last line):
      nrows=(*state).logfilenrows
      n=strlen((*(*state).logfiletext)[nrows-1L])
      (*state).logfileoffset-=n+1L ;; +2L for WINDOWS!!!
      offset=(*state).logfileoffset

      ;; Reading the new part in the text:
      text=p3d_gui_readdr((*state).logfile,offset=offset)
      (*state).logfileoffset=offset
      (*state).logfilenrows+=n_elements(text)-1L

      text=[(*(*state).logfiletext)[0L:nrows-2L],text]
      ptr_free,(*state).logfiletext
      (*state).logfiletext=ptr_new(text)
    
      ;; Updating the logfile viewing text widget with the new text:
      widget_control,(*state).logfiletid,set_value=text, $
          set_text_select=[offset,0L]

      ;; Issuing a new timer event:
      if (*state).logfileusetimer then $
         widget_control,event.id,timer=(*state).logfiletimer

    endif
  endif else begin

    ;; The window has been closed, again allowing the log file to be opened:
    widget_control,(*state).wix.blogr,sensitive=1L
    widget_control,(*state).wix.blogt[0L],sensitive=1L

  endelse ;; widget_info((*state).logfiletid,/valid_id)

  return
END ;; procedure: p3d_gui_timerdr


PRO p3d_gui_openedlog,state,close=close
  compile_opt hidden,IDL2

  if keyword_set(close) then begin
    msg=['The log file was closed on: '+systime()]
  endif else begin
    msg=['!p3d_path='+!p3d_path, $
         '!p3d_data_path='+!p3d_data_path, $
         'The log file was opened on: '+systime()]
  endelse

  error=p3d_misc_logger(msg,(*state).logunit,rname='p3d_gui: ', $
            verbose=(*state).verbose ge 2L)

  return
END ;; procedure: p3d_gui_openedlog


PRO p3d_gui_editheader_event,event
  compile_opt hidden,IDL2

  rname='p3d_gui_editheader_event: '

  ;;========================================------------------------------
  ;; Get the state information:

  widget_control,event.top,get_uvalue=state

  ;;========================================------------------------------
  ;; Get the user value of the event structure:

  widget_control,event.id,get_uvalue=uval,get_value=val

  case uval of
    'edit': begin
      if ptr_valid((*state).localtext) then ptr_free,(*state).localtext
      (*state).localtext=ptr_new(val)
    end

    'close': begin

      i=(*state).localnumber
      j=(*state).localtype
      d=(*(*state).ostate).d

      if ptr_valid((*(*state).ostate).hdr[d,j,i]) then $
          ptr_free,(*(*state).ostate).hdr[d,j,i]
      (*(*state).ostate).hdr[d,j,i]=ptr_new(*(*state).localtext)

      widget_control,event.top,/destroy
    end

    'cancel': widget_control,event.top,/destroy
  endcase ;; uval

  return
end ;;; procedure: p3d_gui_editheader_event


PRO p3d_gui_editheader_cleanup,id
  compile_opt hidden,IDL2

  ;; Get the state information:
  widget_control,id,get_uvalue=state

  if (*state).verbose ge 3 then print,'p3d_gui_editheader: cleanup.'

  if ptr_valid((*state).localtext)  then ptr_free,(*state).localtext
  if ptr_valid(state) then ptr_free,state

  return
END ;;; procedure: p3d_gui_editheader_cleanup


PRO p3d_gui_editheader,state,topwid,error=error
  compile_opt hidden,IDL2

  error=0 & rname='p3d_gui_editheader: '

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      return
    endif
  endif ;; ~(*state).debug

  ;;========================================------------------------------
  ;; Setting up the widget tool:

  if ptr_valid((*(*state).ostate).hhdr[(*(*state).ostate).d]) then $
      ptr_free,(*(*state).ostate).hhdr[(*(*state).ostate).d]

  i=(*state).selectpos
  j=(*state).localtype
  stype=(*(*state).ostate).stype[j]
  d=(*(*state).ostate).d

  (*(*state).ostate).hhdr[d]=ptr_new(*(*(*state).ostate).hdr[d,j,i])

  s=n_elements((*(*state).ostate).hhdr[d])
  localtext=*(*(*state).ostate).hhdr[d]

  screensize=get_screen_size()
  ysize=long(0.9*screensize[1L])
  xsize=80*!d.x_ch_size+40L

  ;; Creating a widget tool:
  bbasewid=widget_base(/column,title='hdr_'+stype + $
      strtrim((*state).selectpos,2L),group_leader=topwid,/base_align_center, $
      /tlb_frame_attr)
  ttextwid=widget_text(bbasewid,value=*(*(*state).ostate).hhdr[d], $
      ysize=s,scr_ysize=ysize,scr_xsize=xsize,/wrap,/scroll,/editable, $
      uvalue='edit',/all_events)

  bbutbwid=widget_base(bbasewid,/row,/base_align_center,/grid_layout)
  bcloswid=widget_button(bbutbwid,value='Set Header and close',uval='close')
  bcancwid=widget_button(bbutbwid,value='Cancel',uval='cancel')

  ;; Realizing the tool:
  widget_control,bbasewid,/realize

  sstate={localtext:ptr_new(localtext,/no_copy), $
          localtype:(*state).localtype,localnumber:(*state).selectpos, $
          ostate:(*state).ostate,verbose:(*state).verbose,debug:(*state).debug}

  ;; Storing the state information in the widget tree:
  pstate=ptr_new(sstate,/no_copy)
  widget_control,bbasewid,set_uvalue=pstate

  xmanager,'p3d_gui_editheader',bbasewid, $
           cleanup='p3d_gui_editheader_cleanup'

  return
END ;;; procedure: p3d_gui_editheader


PRO p3d_gui_set_shiftfile_event,event
  compile_opt hidden,IDL2

  rname='p3d_gui_set_shiftfile_event: '

  ;;========================================------------------------------
  ;; Get the state information:

  widget_control,event.top,get_uvalue=state

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=event.top
      catch,/cancel
      return
    endif
  endif ;; ~(*state).debug

  ;;========================================------------------------------
  ;; Get the user value of the event structure:

  widget_control,event.id,get_uvalue=uval

     d=(*(*state).ostate).d
  path=(*(*state).ostate).sepwpath?(*(*state).ostate).outputpath: $
                                   (*(*state).ostate).inputpath

  case uval of

    'text': begin
      name=tag_names(event,/structure_name)
      if name ne 'widget_text_CH' then return
      widget_control,event.id,get_value=val
      (*state).lname=strtrim(val[0L],2L)
    end

    'browse': begin
      filename=dialog_pickfile(path=path,/must_exist,filter='*.fits', $
          dialog_parent=event.top,title='Select Image:')
      if filename ne '' then begin
;;      name=filesplit(filename) ;; must be exchanged!
        (*state).lpath=name[0L]+path_sep()
        (*(*state).ostate).inputpath=(*state).lpath
        (*state).lname=name[1L]
        widget_control,(*state).ttextwid,set_value=(*state).lname
      endif
    end

    'default': begin
      case (*state).ltype of
        'trc': begin
          j=(*(*state).ostate).ntr
          i=(*(*state).ostate).cur[d,j]
          (*state).lpath=(*(*state).ostate).path[d,j,i]
          dotpos=strpos((*(*state).ostate).file[d,j,i],'.',/reverse_search)
          fsfx=str
          (*state).lname=strmid((*(*state).ostate).file[d,j,i],0L,dotpos)+ $
                         '.fits'
        end
        'createdispmask': begin
          j=(*(*state).ostate).ndi
          i=(*(*state).ostate).cur[d,j]
          (*state).lpath=(*(*state).ostate).path[d,j,i]
          dotpos=strpos((*(*state).ostate).file[d,j,i],'.',/reverse_search)
          (*state).lname=strmid((*(*state).ostate).file[d,j,i],0L,dotpos)+ $
                         '.fits'
        end
      endcase
      widget_control,(*state).ttextwid,set_value=(*state).lname
    end

    'close': begin
      case (*state).ltype of
        'trc': begin
          j=(*(*state).ostate).ntsh
          i=(*(*state).ostate).cur[d,(*(*state).ostate).ntr]
          (*(*state).ostate).file[d,j,i]=(*state).lname
          (*(*state).ostate).path[d,j,i]=(*state).lpath
        end
        'createdispmask': begin
          j=(*(*state).ostate).ndsh
          i=(*(*state).ostate).cur[d,(*(*state).ostate).ndi]
          (*(*state).ostate).file[d,j,i]=(*state).lname
          (*(*state).ostate).path[d,j,i]=(*state).lpath
        end
      endcase
      widget_control,event.top,/destroy
    end

  endcase ;; uval

  return
END ;;; procedure: p3d_gui_set_shiftfile_event


PRO p3d_gui_shiftfile_cleanup,id
  compile_opt hidden,IDL2

  ;; Get the state information:
  widget_control,id,get_uvalue=state

  if (*state).verbose ge 3 then print,'p3d_gui_shiftfile: cleanup.'

  if ptr_valid(state) then ptr_free,state

  return
END ;;; procedure: p3d_gui_shiftfile_cleanup


PRO p3d_gui_set_shiftfile,type,state,topwid
  compile_opt hidden,IDL2

  error=0 & rname='p3d_gui_set_shiftfile: '

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      return
    endif
  endif ;; ~(*state).debug

  ;;========================================------------------------------
  ;; Setting up a widget hierarchy:

  d=(*state).d

  ltype=type
  case type of
    'trc': begin
      j=(*state).ntsh & i=(*state).cur[d,(*state).ntr]
      lpath   = (*state).path[d,j,i]
      lname   = (*state).file[d,j,i]
      applied = (*state).file[d,(*state).ntr,i]
    end
    'createdispmask': begin
      j=(*state).ndsh & i=(*state).cur[d,(*state).ndi]
      lpath   = (*state).path[d,j,i]
      lname   = (*state).file[d,j,i]
      applied = (*state).file[d,(*state).nds,i]
    end
    else: begin
      errmsg='Invalid TYPE.'
      goto,error_handler
    end
  endcase

  bbasewid=widget_base(title='Set Shift Reference File for '+applied,/column, $
      group_leader=topwid,/base_align_center,/tlb_frame_attr)
  llabewid=widget_label(bbasewid,value='Current File:',/align_left)

  bfilewid=widget_base(bbasewid,/row,/base_align_center)
  ttextwid=widget_text(bfilewid,/editable,xsize=20L,value=lname,uvalue='text')
  bfilewid=widget_button(bfilewid,value='Browse',uvalue='browse')
  bdefawid=widget_button(bfiledid,value='Set Default',uvalue='default')

  bcloswid=widget_button(bbasewid,value='Close',uvalue='close')

  ;; Realizing the tool:
  widget_control,bbasewid,/realize

  sstate={lpath:lpath,lname:lname,ltype:ltype,ttextwid:ttextwid, $
          file:(*state).file,path:(*state).path, $
          ostate:state,verbose:(*state).verbose,debug:(*state).debug}

  ;; Storing the state information in the widget tree:
  pstate=ptr_new(sstate,/no_copy)
  widget_control,bbasewid,set_uvalue=pstate

  xmanager,'p3d_gui_set_shiftfile',bbasewid, $
      cleanup='p3d_gui_set_shiftfile_cleanup'

  return

error_handler:
  error=p3d_misc_logger(errmsg,(*state).logunit,rname=rname,topwid=topwid, $
            verbose=(*state).verbose,/error)
  return
END ;;; procedure: p3d_gui_set_shiftfile


PRO p3d_gui_select_file_event,event
  compile_opt hidden,IDL2

  rname='p3d_gui_select_file_event: '

  ;;========================================------------------------------
  ;; Get the state information:

  widget_control,event.top,get_uvalue=state

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=event.top
      catch,/cancel
      return
    endif
  endif ;; ~(*state).debug

  ;;========================================------------------------------
  ;; Get the user value of the event structure:

  widget_control,event.id,get_uval=uval

  case uval of

    'header': begin
      p3d_gui_editheader,state,event.top,error=error
      if error ne 0 then return
    end

    'set': begin
      name=tag_names(event,/structure_name)
      if name ne 'widget_text_CH' then return
      widget_control,event.id,get_val=val
    end

    'select_table': begin
      if event.type ne 4 then return
      if event.sel_top lt 0 then return
      val=strtrim(event.sel_top,2L)
    end

    'close': begin

      i=(*state).selectpos
      j=(*state).localtype
      d=(*(*state).ostate).d

      if j eq (*(*state).ostate).ndi then begin
        (*(*state).ostate).ccmin[d]=(*(*state).ostate).cmin[d,i]
        (*(*state).ostate).ccmax[d]=(*(*state).ostate).cmax[d,i]
        (*(*state).ostate).curb[d] =(*(*state).ostate).column[d,i]
      endif

      ;; Setting information about the current data set:
      (*(*state).ostate).cur[d,j]=i
      if ptr_valid((*(*state).ostate).curdata[d,j]) then $
          ptr_free,(*(*state).ostate).curdata[d,j]
      (*(*state).ostate).curdata[d,j]= $
         ptr_new(*(*(*state).ostate).data[d,j,(*(*state).ostate).cur[d,j]])

      (*(*state).ostate).sf_cancel=0L
      widget_control,event.top,/destroy
      return
    end ;; case: 'close'

    'cancel': begin
      (*(*state).ostate).sf_cancel=1L ;; BEWARE, this will not be saved!
      widget_control,event.top,/destroy
      return
    end

    'save': begin
      widget_control,(*state).bsavewid,sensitive=0L

      d=(*(*state).ostate).d

      if ptr_valid((*(*state).ostate).hhdr[d]) then $
          ptr_free,(*(*state).ostate).hhdr[d]

      i=(*state).selectpos
      j=(*state).localtype

      ;; Retrieving the current data set:
      (*(*state).ostate).hhdr[d]=ptr_new(*(*(*state).ostate).hdr[d,j,i])
      image=*(*(*state).ostate).data[d,j,i]

      ;; Saving the data:
      fxwrite,(*state).paths[(*state).selectpos] + $
              (*state).filenames[(*state).selectpos], $
              *(*(*state).ostate).hhdr[d],image
      widget_control,(*state).bsavewid,sensitive=1L

    end ;; case: 'save'

    'setlib': begin
      widget_control,(*state).blibawid,sensitive=0L

      d=(*(*state).ostate).d

      if ptr_valid((*(*state).ostate).hhdr[d]) then $
          ptr_free,(*(*state).ostate).hhdr[d]

      i=(*state).selectpos
      j=(*state).localtype

      ;; Retrieving the current data set:
      (*(*state).ostate).hhdr[d]=ptr_new(*(*(*state).ostate).hdr[d,j,i])
      image=*(*(*state).ostate).data[d,j,i]

      print,rname+'write ' + !p3d_dispmask_path + (*state).libname + $
            path_sep() + (*state).libfilename + (*state).libextname

      fxwrite,!p3d_dispmask_path + (*state).libname + path_sep() + $
              (*state).libfilename + (*state).libextname, $
              *(*(*state).ostate).hhdr[d],image

      print,rname+'done'

      widget_control,(*state).blibawid,sensitive=1L
    end

    'setlibname': begin
      widget_control,event.id,get_value=tmp
      (*state).libfilename=tmp
    end

  endcase ;; uval

  if uval eq 'set' or uval eq 'select_table' then begin

    i=val[0L] & istr=strtrim(i,2L)
    if max((*state).existpos eq val[0L]) ne 0L then begin
      (*state).selectpos=val[0L]

      widget_control,(*state).bheadwid,/sensitive
      widget_control,(*state).bsetfwid,/sensitive
      widget_control,(*state).bsavewid,/sensitive

      d=(*(*state).ostate).d
      j=(*state).localtype
      case j of

        (*(*state).ostate).nbx: begin
          widget_control,(*(*state).ostate).wid[d].tboxf, $
              set_value= (*(*state).ostate).file[d,(*(*state).ostate).nbx,i]
          widget_control,(*(*state).ostate).wid[d].lboxi,set_value=istr
        end ;; case: (*(*state).ostate).nbx

        (*(*state).ostate).nbi: begin
          widget_control,(*(*state).ostate).wid[d].tbiaf, $
              set_value= (*(*state).ostate).file[d,(*(*state).ostate).nbi,i]
          widget_control,(*(*state).ostate).wid[d].lbiai,set_value=istr
        end ;; case: (*(*state).ostate).nbi

        (*(*state).ostate).ntr: begin
          widget_control,(*(*state).ostate).wid[d].ttraf, $
              set_value= (*(*state).ostate).file[d,(*(*state).ostate).ntr,i]
          widget_control,(*(*state).ostate).wid[d].ltrai,set_value=istr
        end ;; case: (*(*state).ostate).ntr

        (*(*state).ostate).ndi: begin

;         if ptr_valid((*(*state).ostate).hhdr[d]) then $
;             ptr_free,(*(*state).ostate).hhdr[d]
;         (*(*state).ostate).hhdr[d]=ptr_new( $
;             *(*(*state).ostate).hdr[d,(*state).ndi,(*state).selectpos])
;
;         ;; this is not portable:
;         tmp=p3d_misc_get_hdr_kwrd((*(*state).ostate).lkwrdlist,'GRATNAME', $
;                 topwid=event.top,verbose=(*state).verbose,error=error, $
;                 debug=(*state).debug)
;         if error ne 0 then return
;         gratname=fxpar(*(*(*state).ostate).hhdr[d],tmp)
;
;         (*state).libname=strtrim(gratname,2L)
;         p3d_wavecal_set_linelist, $
;             (*(*state).ostate).path[d,(*(*state).ostate).ndi,i] + $
;             (*(*state).ostate).file[d,(*(*state).ostate).ndi,i], $
;             linelists,lamps,topwid=event.top,verbose=verbose,error=error, $
;             debug=(*state).debug
;         (*state).libfilename=lamps[0L]
;
;         for k=1L,n_elements(lamps)-1L do (*state).libfilename+='_'+lamps[k]
;
;         tmp=p3d_misc_get_hdr_kwrd((*(*state).ostate).lkwrdlist,'GROTPOS', $
;                 topwid=event.top,verbose=(*state).verbose,error=error, $
;                 debug=(*state).debug)
;         if error ne 0 then return
;         grotpos=abs(fxpar(*(*(*state).ostate).hhdr[d],tmp))
;
;         grotpos=fix(grotpos*100)
;         grotfix=strtrim(fix(grotpos/100),2L)
;         grotrest=strtrim(grotpos-grotfix*100,2L)
;         if grotrest eq '0' then grotrest=''
;         grotname=grotfix+grotrest
;         (*state).libextname='_'+grotname+'.dmk'
;
          tmp=(*(*state).ostate).file[d,(*(*state).ostate).ndi,i]
          widget_control,(*(*state).ostate).wid[d].tdisf,set_value=tmp
          widget_control,(*(*state).ostate).wid[d].ldiai,set_value=istr
;         tmp=(*state).libname+path_sep()
;         widget_control,(*(*state).ostate).tlibawid,set_value=tmp
;         widget_control,(*(*state).ostate).tlibnwid, $
;             set_value= (*state).libfilename
;         widget_control,(*(*state).ostate).tlibewid, $
;             set_value= (*state).libextname
;         widget_control,(*(*state).ostate).blibawid,sensitive=(*state).libok
;         widget_control,(*(*state).ostate).tlibawid,sensitive=(*state).libok
;         widget_control,(*(*state).ostate).tlibnwid,sensitive=(*state).libok
;         widget_control,(*(*state).ostate).tlibewid,sensitive=(*state).libok
        end ;; case: (*(*state).ostate).ndi

        (*(*state).ostate).nfl: begin
          widget_control,(*(*state).ostate).wid[d].tflaf, $
              set_value= (*(*state).ostate).file[d,(*(*state).ostate).nfl,i]
          widget_control,(*(*state).ostate).wid[d].lflai,set_value=istr
        end ;; case: (*(*state).ostate).nfl

        (*(*state).ostate).nob: begin
          tmp=(*(*state).ostate).file[d,(*(*state).ostate).nob,i]
          widget_control,(*(*state).ostate).wid[d].tobjf,set_value=tmp
          widget_control,(*(*state).ostate).wid[d].lobji,set_value=istr
          tmp=strtrim((*(*state).ostate).cmin[d,i],2L)
          widget_control,(*(*state).ostate).wid[d].tcmin,set_value=tmp
          tmp=strtrim((*(*state).ostate).cmax[d,i],2L)
          widget_control,(*(*state).ostate).wid[d].tcmax,set_value=tmp
          tmp=strtrim((*(*state).ostate).column[d,i],2L)
          widget_control,(*(*state).ostate).wid[d].twave,set_value=tmp

          ;; Reloading the data:
          file=(*(*state).ostate).file[d,(*(*state).ostate).nob,i]
          path=(*(*state).ostate).path[d,(*(*state).ostate).nob,i]
          data=readfits(path+file,header, $
                        silent=(*(*state).ostate).verbose lt 3,/no_unsigned)

          image=reform((*(*state).ostate).daxis? $
                  data[(*(*state).ostate).column[d,i],*]: $
                  data[*,(*(*state).ostate).column[d,i]])

          ;; Retrieving position table properties:
          idx=d*(*(*state).ostate).nblocks+(*(*state).ostate).block
          posfile =(*(*(*state).ostate).pos[idx]).file
          shape   =(*(*(*state).ostate).pos[idx]).shape
          rownum  =(*(*(*state).ostate).pos[idx]).fpos
          xpos    =(*(*(*state).ostate).pos[idx]).fpx
          ypos    =(*(*(*state).ostate).pos[idx]).fpy
          lenssize=(*(*(*state).ostate).pos[idx]).fles

          wset,(*(*state).ostate).wid[d].dleft
          p3d_display_telescl_postable,image,(*(*state).ostate).leftsize, $
              orientation=(*(*state).ostate).orientation[d,i],/science, $
              posfile=posfile,rownum=rownum,xpos=xpos,ypos=ypos, $
              lens_size=lenssize,shape=shape, $
              bottom=(*(*state).ostate).cbottom,/noerase,topwid=event.top, $
              logunit=(*(*state).ostate).logunit,verbose=(*state).verbose, $
              error=error, $
              debug=(*state).debug
          if error ne 0 then return
        end ;; case: (*(*state).ostate).nob

      endcase ;; j

    endif ;; end condition: max((*state).existpos eq val[0]) ne 0

    widget_control,(*state).tchoowid,set_value=istr
    if i gt 0L then $
       widget_control,(*state).tpostwid,set_table_select=[0L,i,0L,i]

  endif ;; end condition: uval eq 'set' or uval eq 'select_table'

  return
END ;;; procedure: p3d_gui_select_file_event


PRO p3d_gui_select_file_cleanup,id
  compile_opt hidden,IDL2

  ;; Get the state information:
  widget_control,id,get_uvalue=state

  if (*state).verbose ge 3 then print,'p3d_gui_select_file: cleanup.'

  if ptr_valid(state) then ptr_free,state

  return
END ;;; procedure: p3d_gui_select_file_cleanup


PRO p3d_gui_select_file,type,state,error=error
  compile_opt hidden,IDL2

  error=0 & rname='p3d_gui_select_file: '

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=state.top
      catch,/cancel
      return
    endif
  endif ;; ~(*state).debug

  ;;========================================------------------------------
  ;; Setting up a widget hierarchy:

  libok=0L & localtype=type
  stype=(*state).stype[type]

  filenames=(*state).file[(*state).d,type,*]
  paths    =(*state).path[(*state).d,type,*]

;  if type eq (*state).ndi then begin
;    if strpos(strtrim(!p3d_dispmask_path,2L),strtrim(!p3d_path,2L)) $
;       ne 0L then libok=1
;  endif

  existpos=where(strtrim(filenames,2L) ne '',count)
  existpos=~count?-1L:strtrim(existpos,2L)

  table=reform(filenames,1L,n_elements(filenames))

  ;; Create the widget hierarchy:
  bbasewid=widget_base(title='Select one of the loaded data sets:',/column, $
      /base_align_center,/tlb_frame_attr,group_leader=(*state).top)
  ltitlwid=widget_label(bbasewid,value='Current Image Settings:')
  colw=50L*!d.x_ch_size
  tpostwid=widget_table(bbasewid,value=table,column_label=['Filename:'], $
      row_labels=stype+strtrim(lindgen((*state).n),2L),column_widths=colw, $
      /all_events,uvalue='select_table')
  bchoowid=widget_base(bbasewid,/row,/base_align_center,xpad=0L,ypad=0L, $
      space=10L)
  llabewid=widget_label(bchoowid,value='Choose Mask: '+stype)
  tchoowid=widget_text(bchoowid,value='',xsize=1,/editable,/all_events, $
      uvalue='set')
  bheadwid=widget_button(bchoowid,value='Edit header',uvalue='header')
  bsavewid=widget_button(bbasewid,value='Write to file',uvalue='save')

  blibawid=0L & btextwid=0L
  tlibawid=0L & tlibnwid=0L & tlibewid=0L
;  if type eq (*state).ndi then begin
;    bsetlwid=widget_base(basewid,/column,/frame,/base_align_center)
;
;    blibawid=widget_button(bsetlwid,value='ADD TO LIBRARY',uvalue='setlib')
;
;    btextwid=widget_base(bsetlwid,/row,/base_align_center)
;    tlibawid=widget_text(btextwid,value='',xsize=6L)
;    tlibnwid=widget_text(btextwid,value='',uvalue='setlibname',/editable, $
;        xsize=12L)
;    tlibewid=widget_text(btextwid,value='',editable=0L,xsize=9L)
;
;    widget_control,tlibawid,sensitive=0L
;    widget_control,tlibewid,sensitive=0L
;    widget_control,blibawid,sensitive=0L
;  endif ;; type eq (*state).ndi

  bbutrwid=widget_base(bbasewid,/row,/base_align_center,/grid_layout, $
      xpad=0L,ypad=0L,space=20L)
  bsetfwid=widget_button(bbutrwid,value='OK',uvalue='close')
  bcancwid=widget_button(bbutrwid,value=' Cancel ',uvalue='cancel')

  widget_control,bsetfwid,sensitive=0L
  widget_control,bsavewid,sensitive=0L
  widget_control,bheadwid,sensitive=0L

  ;; Realizing the tool:
  widget_control,bbasewid,/realize

  ;; Centering the tool:
  tmp=widget_info(bbasewid,/geometry)
  ttp=widget_info((*state).top,/geometry)
  xpos=ttp.xoffset+ttp.xsize/2L-tmp.xsize/2L
  ypos=ttp.yoffset+ttp.ysize/2L-tmp.ysize/2L
  widget_control,bbasewid,tlb_set_xoffset=xpos
  widget_control,bbasewid,tlb_set_yoffset=ypos

  sstate={libname:'',libfilename:'',libextname:'', $
          existpos:existpos,filenames:filenames,paths:paths, $
          bheadwid:bheadwid,bsavewid:bsavewid,blibawid:blibawid, $
          bsetfwid:bsetfwid,tpostwid:tpostwid,tchoowid:tchoowid, $
          tlibawid:tlibawid,tlibnwid:tlibnwid,tlibewid:tlibewid, $
          libok:libok,localtype:localtype,selectpos:-1L, $
          ostate:state,verbose:(*state).verbose,debug:(*state).debug}

  ;; Storing the state information in the widget tree:
  pstate=ptr_new(sstate,/no_copy)
  widget_control,bbasewid,set_uvalue=pstate

  ;; Handing over the event handling to xmanager:
  xmanager,'p3d_gui_select_file',bbasewid, $
      cleanup='p3d_gui_select_file_cleanup'

  return

error_handler:
  error=p3d_misc_logger(errmsg,(*state).logunit,rname=rname, $
      topwid=(*state).top,verbose=(*state).verbose,/error)
  return
END ;;; procedure: p3d_gui_select_file


PRO p3d_gui_position_new_file_event,event
  compile_opt hidden,IDL2

  rname='p3d_gui_position_new_file_event: '

  ;;========================================------------------------------
  ;; Get the state information:

  widget_control,event.top,get_uval=state

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=event.top
      catch,/cancel
      return
    endif
  endif ;; ~(*state).debug

  ;;========================================------------------------------
  ;; Get the user value of the event structure:

  widget_control,event.id,get_uval=uval

  case uval of

    'set': begin
      name=tag_names(event,/structure_name)
      if name ne 'widget_text_CH' then return
      widget_control,event.id,get_val=val
    end

    'select_table': begin
      if event.type ne 4L then return
      if event.sel_top lt 0L then return
      val=strtrim(event.sel_top,2L)
    end

    'close': begin

      ;; Setting various data variables:
      j=(*state).localtype
      d=(*(*state).ostate).d
      (*(*state).ostate).file[d,j,*]=transpose(strtrim((*state).table,2L))
      (*(*state).ostate).path[d,j,(*state).pos]=(*state).path
      (*(*state).ostate).cur[d,j]=(*state).pos

      if ptr_valid((*(*state).ostate).curdata[d,j]) then $
          ptr_free,(*(*state).ostate).curdata[d,j]
      (*(*state).ostate).curdata[d,j]=ptr_new((*state).data_array)

      ;; (*state).localtype:
      case (*state).localtype of

        ;; Shift boxes:
        (*(*state).ostate).nbx: begin
          if (*(*state).ostate).cur[d,(*(*state).ostate).ntr] ne -1L and $
             (*(*state).ostate).cur[d,(*(*state).ostate).nob] ne -1L then $
             widget_control,(*(*state).ostate).wix.bshiv,/sensitive
        end

        ;; Trace mask:
        (*(*state).ostate).ntr: begin
          k=(*(*state).ostate).ntsh
          (*(*state).ostate).path[d,k,(*state).pos]=(*state).path

          ok=widget_info((*(*state).ostate).wix.bshiv,/valid_id)
          if (*(*state).ostate).cur[d,(*(*state).ostate).nob] ne -1L and $
             (*(*state).ostate).cur[d,(*(*state).ostate).nbx] ne -1L and $
             ok then widget_control,(*(*state).ostate).wix.bshiv,/sensitive
        end

        ;; Dispersion Mask:
        (*(*state).ostate).ndi: begin
          k=(*(*state).ostate).ndsh
          (*(*state).ostate).path[d,k,(*state).pos]=(*state).path
        end

        ;; Object:
        (*(*state).ostate).nob: begin
          ;; Must be rewritten!
          tmp=(*state).path+ $
              (*(*state).ostate).file[d,j,(*(*state).ostate).cur[d,j]]
          hdr=headfits(tmp)

          num=(*(*state).ostate).cur[d,j]
          if ptr_valid((*(*state).ostate).hdr[d,j,num]) then $
              ptr_free,(*(*state).ostate).hdr[d,j,num]
          (*(*state).ostate).hdr[d,j,num]=ptr_new(hdr)

          k=(*(*state).ostate).nosh
          (*(*state).ostate).path[d,k,num]=(*state).path
          (*(*state).ostate).file[d,k,num]=fxpar(hdr,'Shiftref')

;          if (*(*state).ostate).e3dactive then $
;             widget_control,(*(*state).ostate).wix.be3dv,/sensitive

          ok=widget_info((*(*state).ostate).wix.bshiv,/valid_id)
          if (*(*state).ostate).cur[d,(*(*state).ostate).ntr] ne -1L and $
             (*(*state).ostate).cur[d,(*(*state).ostate).nbx] ne -1L and $
          ok then widget_control,(*(*state).ostate).wix.bshiv,/sensitive
        end

        else: ;; nothing
      endcase ;; (*state).localtype

      (*state).sdone=1L

      widget_control,event.top,/destroy
      return
    end

    'cancel': begin
      (*(*state).ostate).nf_cancel=1L ;; BEWARE, this will not be saved
      widget_control,event.top,/destroy
      return
    end

  endcase ;; uval

  if uval eq 'set' or uval eq 'select_table' then begin
    widget_control,(*state).tnumbwid,set_value=strtrim((*state).pos,2L)

    if max(lindgen((*(*state).ostate).n) eq val[0L]) ne 0L then begin
      (*state).pos=val[0L]
      (*state).table=transpose((*state).filenames)
      (*state).table[(*state).pos]=(*state).localfile
      widget_control,(*state).tposfwid,set_value=(*state).table

      if (*state).localtype eq (*(*state).ostate).nob then begin
        tmp=strtrim((*state).pos,2L)
        widget_control,(*(*state).ostate).wid[(*(*state).ostate).d].lobji, $
                       set_value=tmp
      endif

      widget_control,(*state).bcloswid,/sensitive
    endif

    widget_control,(*state).tnumbwid,set_value=strtrim((*state).pos,2L)
    if (*state).pos gt 0L then widget_control,(*state).tposfwid, $
         set_table_select=[0L,(*state).pos,0L,(*state).pos]
  endif ;; uval eq 'set' or uval eq 'select_table'

  return
END ;; procedure: p3d_gui_position_new_file_events


PRO p3d_gui_position_new_file_cleanup,id
  compile_opt hidden,IDL2

  ;; Get the state information:
  widget_control,id,get_uvalue=state

  if (*state).verbose ge 3 then $
     print,'p3d_gui_position_new_file: cleanup.'

  ;; Setting this in order to not set any variables in case the window
  ;; is closed and nothing was done:
  if ~(*state).sdone then (*(*state).ostate).nf_cancel=1L

  if ptr_valid(state) then ptr_free,state

  return
END ;;; procedure: p3d_gui_position_new_file_cleanup


PRO p3d_gui_position_new_file,type,newfile,data_array,state, $
        useinputpath=useinputpath,error=error,debug=debug
  compile_opt hidden,IDL2

  error=0 & rname='p3d_gui_position_new_file: '
  topwid=(*state).top

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      return
    endif
  endif

  ;;========================================------------------------------
  ;; Checking the input arguments:

  (*state).nf_cancel=0L

  filenames=(*state).file[(*state).d,type,*]
  localtype=type
  stype=(*state).stype[type]

  pos=where(strtrim(filenames,2L) eq '',count)
  pos=pos[0L]
;  if count eq 0 then pos='' else pos=strtrim(pos+1,2)

  table=reform(filenames,1L,n_elements(filenames))

  ;;========================================------------------------------
  ;; Setting up a widget tool:

  bbasewid=widget_base(title='Set New Data Variable:',/column, $
      xoffset=640,group_leader=topwid,/base_align_center,/tlb_frame_attr)

  ltitlwid=widget_label(bbasewid,value='Current Data Settings:')
  tposfwid=widget_table(bbasewid,value=table,column_label=['Filename:'], $
      row_label=stype+strtrim(indgen((*state).n),2L),column_widths=180, $
      /all_events,uvalue='select_table')

  lspacwid=widget_label(bbasewid,value='',ysize=20)

  bnewbwid=widget_base(bbasewid,/row,/base_align_center)
  llabewid=widget_label(bnewbwid,value='New Image: '+newfile)

  bnew2wid=widget_base(bbasewid,/row,/base_align_center)
  llabewid=widget_label(bnew2wid,value='Set Image to: '+stype)
  tnumbwid=widget_text(bnew2wid,value='',xsize=1,/editable, $
                       /all_events,uvalue='set')

  bbutrwid=widget_base(bbasewid,/row,/base_align_center,/grid_layout)
  bcloswid=widget_button(bbutrwid,value='OK',uvalue='close')
  bcancwid=widget_button(bbutrwid,value='Cancel',uvalue='cancel')

  widget_control,bcloswid,sensitive=0L
  ;; Realizing the tool:
  widget_control,bbasewid,/realize

  ;; Centering the tool:
  tmp=widget_info(bbasewid,/geometry)
  ttp=widget_info((*state).top,/geometry)
  xpos=ttp.xoffset+ttp.xsize/2L-tmp.xsize/2L
  ypos=ttp.yoffset+ttp.ysize/2L-tmp.ysize/2L
  widget_control,bbasewid,tlb_set_xoffset=xpos
  widget_control,bbasewid,tlb_set_yoffset=ypos

  path=keyword_set(useinputpath)?(*state).inputpath: $
       ((*state).sepwpath?(*state).outputpath:(*state).inputpath)

  sstate={localtype:localtype,localfile:newfile,filenames:filenames, $
          path:path,table:table,pos:pos,data_array:data_array, $
          tposfwid:tposfwid,tnumbwid:tnumbwid,bcloswid:bcloswid, $
          ostate:state,sdone:0L,verbose:(*state).verbose,debug:debug}

  ;; Storing the state information in the widget tree:
  pstate=ptr_new(sstate,/no_copy)
  widget_control,bbasewid,set_uvalue=pstate,/no_copy

  ;; Handing over the event handling to xmanager:
  xmanager,'p3d_gui_position_new_file',bbasewid, $
           cleanup='p3d_gui_position_new_file_cleanup'

  return

error_handler:
  error=p3d_misc_logger(errmsg,(*state).logunit,rname=rname,topwid=topwid, $
            verbose=(*state).verbose,/error)
  return
END ;;; procedure: p3d_gui_position_new_file


PRO p3d_gui_suffixes,state,error=error
  ;; Multi-block detectors: adding an extra suffix to only show the first
  ;; block filenames:
  error=0

  if (*state).userparfile ne '' then $
     readcol,(*state).userparfile,uparname,uparvalue,format='a,a', $
         comment=';',silent=(*state).verbose lt 3,delimiter=' '

  (*state).xxsfx=''
  if (*state).nblocks gt 1L then begin
    p3d_misc_read_params,*(*state).parname,*(*state).parvalue, $
        'blocksfx_1',tmp,/must_exist,uparname=uparname, $
        uparvalue=uparvalue,topwid=(*state).top,logunit=(*state).logunit, $
        verbose=(*state).verbose,error=error,debug=(*state).debug
    if error ne 0 then return

    (*state).xxsfx=strtrim(tmp,2L)
  endif ;; (*state).nblocks gt 1L

  ;; General and Cosmic ray mask filename suffix, output file compression:
  (*state).sfx='.fits'
  (*state).crsfx='.crmk'
  (*state).icsfx='_imcmb'
  (*state).mbsfx='_mbias'
  (*state).trsfx='_trace'
  (*state).dmsfx='_dmask'
  (*state).ffsfx='_flatf'
  (*state).obsfx='_oextr'
  (*state).compress=0L
  if n_elements(uparname) ne 0L then begin
    p3d_misc_read_params,uparname,uparvalue,'sfx',sfx,/upo, $
        topwid=(*state).top,logunit=(*state).logunit, $
        verbose=(*state).verbose,error=error,debug=(*state).debug
    if error ne 0 then return
    if n_elements(sfx) ne 0L then begin
      tmp2=strpos(sfx,'.',/reverse_search)
      sfx=tmp2 ne -1L?strmid(sfx,tmp2):'.'+sfx
      if sfx ne '.' then (*state).sfx=strtrim(sfx,2L)
    endif

    p3d_misc_read_params,uparname,uparvalue,'crsfx',crsfx,/upo, $
        topwid=(*state).top,logunit=(*state).logunit, $
        verbose=(*state).verbose,error=error,debug=(*state).debug
    if error ne 0 then return
    if n_elements(crsfx) ne 0L then begin
      tmp2=strpos(crsfx,'.',/reverse_search)
      crsfx=tmp2 ne -1L?strmid(crsfx,tmp2):'.'+crsfx
      if crsfx ne '.' then (*state).crsfx=strtrim(crsfx,2L)
    endif

    p3d_misc_read_params,uparname,uparvalue,'icsfx',icsfx,/upo, $
        topwid=(*state).top,logunit=(*state).logunit, $
        verbose=(*state).verbose,error=error,debug=(*state).debug
    if error ne 0 then return
    if n_elements(icsfx) ne 0L then begin
      tmp2=strpos(icsfx,'_',/reverse_search)
      icsfx=tmp2 ne -1L?strmid(icsfx,tmp2):'_'+icsfx
      if icsfx ne '.' then (*state).icsfx=strtrim(icsfx,2L)
    endif

    p3d_misc_read_params,uparname,uparvalue,'mbsfx',mbsfx,/upo, $
        topwid=(*state).top,logunit=(*state).logunit, $
        verbose=(*state).verbose,error=error,debug=(*state).debug
    if error ne 0 then return
    if n_elements(mbsfx) ne 0L then begin
      tmp2=strpos(mbsfx,'_',/reverse_search)
      mbsfx=tmp2 ne -1L?strmid(mbsfx,tmp2):'_'+mbsfx
      if mbsfx ne '.' then (*state).mbsfx=strtrim(mbsfx,2L)
    endif

    p3d_misc_read_params,uparname,uparvalue,'trsfx',trsfx,/upo, $
        topwid=(*state).top,logunit=(*state).logunit, $
        verbose=(*state).verbose,error=error,debug=(*state).debug
    if error ne 0 then return
    if n_elements(trsfx) ne 0L then begin
      tmp2=strpos(trsfx,'_',/reverse_search)
      trsfx=tmp2 ne -1L?strmid(trsfx,tmp2):'_'+trsfx
      if trsfx ne '.' then (*state).trsfx=strtrim(trsfx,2L)
    endif

    p3d_misc_read_params,uparname,uparvalue,'dmsfx',dmsfx,/upo, $
        topwid=(*state).top,logunit=(*state).logunit, $
        verbose=(*state).verbose,error=error,debug=(*state).debug
    if error ne 0 then return
    if n_elements(dmsfx) ne 0L then begin
      tmp2=strpos(dmsfx,'_',/reverse_search)
      dmsfx=tmp2 ne -1L?strmid(dmsfx,tmp2):'_'+dmsfx
      if dmsfx ne '.' then (*state).dmsfx=strtrim(dmsfx,2L)
    endif

    p3d_misc_read_params,uparname,uparvalue,'ffsfx',ffsfx,/upo, $
        topwid=(*state).top,logunit=(*state).logunit, $
        verbose=(*state).verbose,error=error,debug=(*state).debug
    if error ne 0 then return
    if n_elements(ffsfx) ne 0L then begin
      tmp2=strpos(ffsfx,'_',/reverse_search)
      ffsfx=tmp2 ne -1L?strmid(ffsfx,tmp2):'_'+ffsfx
      if ffsfx ne '.' then (*state).ffsfx=strtrim(ffsfx,2L)
    endif

    p3d_misc_read_params,uparname,uparvalue,'obsfx',obsfx,/upo, $
        topwid=(*state).top,logunit=(*state).logunit, $
        verbose=(*state).verbose,error=error,debug=(*state).debug
    if error ne 0 then return
    if n_elements(obsfx) ne 0L then begin
      tmp2=strpos(obsfx,'_',/reverse_search)
      obsfx=tmp2 ne -1L?strmid(obsfx,tmp2):'_'+obsfx
      if obsfx ne '.' then (*state).obsfx=strtrim(obsfx,2L)
    endif

    p3d_misc_read_params,uparname,uparvalue,'compress',cmr,/upo, $
        topwid=(*state).top,logunit=(*state).logunit, $
        verbose=(*state).verbose,error=error,debug=(*state).debug
    if error ne 0 then return
    (*state).compress=~n_elements(cmr)?0L:(strlowcase(cmr) eq 'yes'?1L:0L)
  endif

  return
END ;;; procedure: p3d_gui_suffixes


PRO p3d_gui_extract_menu_event,event
  compile_opt hidden,IDL2

  rname='p3d_gui_extract_menu_event: '

  ;;========================================------------------------------
  ;; Get the state information:

  widget_control,event.top,get_uvalue=state

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=event.top
      catch,/cancel
      return
    endif
  endif ;; ~(*state).debug

  tmpstrname=strlowcase(tag_names(event,/structure_name))
  if tmpstrname eq 'widget_kbrd_focus' then if event.enter then return

  ;;========================================------------------------------
  ;; Get the user value of the event structure:

  widget_control,event.id,get_uvalue=uval

  case uval of

    'browse': begin
      xfx=(*(*state).ostate).xxsfx+(*(*state).ostate).sfx
      yfx=(*(*state).ostate).icsfx+(*(*state).ostate).sfx
      filter=['*'+xfx+';*'+xfx+'.gz;*'+xfx+'.Z','*'+xfx+['','.gz','.Z'], $
              '*'+yfx+';*'+yfx+'.gz;*'+yfx+'.Z','*'+yfx+['','.gz','.Z']]
      filename=dialog_pickfile(path=(*state).ex.path,/must_exist, $
          /multiple_files,filter=filter, $
          dialog_parent=event.top,title='Please select an object image file')
      if filename[0L] ne '' then begin
        n=n_elements(filename)
        (*state).ex.path=file_dirname(filename[0L],/mark_directory)

        if ptr_valid((*state).ex.file) then ptr_free,(*state).ex.file
        (*state).ex.file=ptr_new(file_basename(filename))

        ;; Updating widgets:
        tmp=strjoin(*(*state).ex.file,', ')
        widget_control,(*state).tfilewid,set_value=tmp
        widget_control,(*state).bexecwid,/sensitive

        widget_control,(*state).bcosmwid,sensitive=n eq 1L
        tmp=n eq 1L and (*state).ex.crsub
        widget_control,(*state).bcosbwid,sensitive=tmp
        tmp=tmp and (*state).ex.cruex and (*state).sexmethod eq 'optimal'
        widget_control,(*state).bcoscwid,sensitive=tmp

        if (*state).assignobj then begin
          (*state).ex.shiftrefpath=(*state).ex.path
          (*state).ex.shiftreffile=(*(*state).ex.file)[0L]
          widget_control,(*state).treftwid,set_value=(*state).ex.shiftreffile
        endif

      endif ;; filename[0L] ne ''
    end ;; case: 'browse'

    'filetext': begin
      name=tag_names(event,/structure_name)
      if name ne 'widget_text_CH' then return
      widget_control,event.id,get_value=val

      tmp=strtrim(strsplit(val[0L],',',/extract,count=count),2L)
      if ptr_valid((*state).ex.file) then ptr_free,(*state).ex.file
      (*state).ex.file=ptr_new(file_basename(tmp))

      ptmp=file_dirname(tmp[0L],/mark_directory)
      (*state).ex.path=ptmp ne '.'+path_sep()?ptmp:(*(*state).ostate).inputpath

      ;; Updating widgets:
      widget_control,(*state).bcosmwid,sensitive=n eq 1L
      tmp=n eq 1L and (*state).ex.crsub
      widget_control,(*state).bcosbwid,sensitive=tmp
      tmp=tmp and (*state).ex.cruex and (*state).sexmethod eq 'optimal'
      widget_control,(*state).bcoscwid,sensitive=tmp

      if (*state).assignobj then begin
        (*state).ex.shiftrefpath=(*state).ex.path
        (*state).ex.shiftreffile=(*(*state).ex.file)[0L]
        widget_control,(*state).treftwid,set_value=(*state).ex.shiftreffile
      endif

      ;; Only sensitizing the extract button if the file exists:
      sensarr=lonarr(count)
      for i=0L,count-1L do begin
        tmp=(*state).ex.path+(*(*state).ex.file)[i]
        sensarr[i]=file_test(tmp,/regular,/read)
      endfor
      sensitive=total(sensarr)/count
      widget_control,(*state).bexecwid,sensitive=sensitive

    end ;; case: 'filetext'

    'refbrowse': begin
      sfx=(*(*state).ostate).sfx & xxsfx=(*(*state).ostate).xxsfx
      filter=['*'+xxsfx+sfx+';*'+xxsfx+sfx+'.gz;*'+xxsfx+sfx+'.Z', $
              '*'+xxsfx+sfx+['','.gz','.Z']]
      filename=dialog_pickfile(path=(*state).ex.path, $
          /must_exist,filter=filter,dialog_parent=event.top, $
          title='Please select an object image file')
      if filename[0L] ne '' then begin
        (*state).ex.shiftrefpath=file_dirname(filename,/mark_directory)
        (*state).ex.path=(*state).ex.shiftrefpath
        (*state).ex.shiftreffile= $
           strmid(filename,strlen((*state).ex.shiftrefpath))

        ;; Updating widgets:
        widget_control,(*state).treftwid,set_value=(*state).ex.shiftreffile
      endif ;; filename[0L] ne ''

    end ;; case: 'refbrowse'

    'reftext': begin
      name=tag_names(event,/structure_name)
      if name ne 'widget_text_CH' then return
      widget_control,event.id,get_value=val

      (*state).ex.shiftreffile=strtrim(val[0L],2L)
;     (*(*state).ostate).shiftrefpath=(*(*state).ostate).inputpath
    end ;; case: 'reftext'

    'setref': begin
      (*state).assignobj=((*state).assignobj+1L) mod 2L

      if (*state).assignobj then begin
        (*state).ex.shiftrefpath=(*state).ex.path
        (*state).ex.shiftreffile=(*(*state).ex.file)[0L]
        widget_control,(*state).treftwid,set_value=(*state).ex.shiftreffile
      endif

      ;; Updating widgets:
      widget_control,(*state).bsetrwid,set_button= (*state).assignobj
      widget_control,(*state).treftwid,editable  =~(*state).assignobj
      widget_control,(*state).brefbwid,sensitive =~(*state).assignobj
    end ;; case: 'setref'

    'traceshift': (*(*state).ostate).dotraceshif= $
       ~(*(*state).ostate).dotraceshif

    'bias':   (*state).ex.bias =~(*state).ex.bias

    'cosmic': begin
       (*state).ex.crsub=~(*state).ex.crsub
       if (*state).ex.crsub then begin
         tmp=*(*state).ex.file
         tmp=tmp[0L] ne '' and n_elements(tmp) eq 1L
         widget_control,(*state).bcosbwid,sensitive=tmp
         tmp=tmp and (*state).sexmethod eq 'optimal' and (*state).ex.cruex
         widget_control,(*state).bcoscwid,sensitive=tmp
       endif else begin
         widget_control,(*state).bcosbwid,sensitive=0L
         widget_control,(*state).bcoscwid,sensitive=0L
       endelse ;; (*state).ex.crsub
    end ;; case: 'cosmic'

    'crbrowse': begin
      sfx=(*(*state).ostate).crsfx
      filter=['*'+sfx+';*'+sfx+'.gz;*'+sfx+'.Z','*'+sfx+['','.gz','.Z'], $
              '*'+['','.gz','.Z']]
      filename=dialog_pickfile(path=(*state).ex.opath,/must_exist, $
          filter=filter,dialog_parent=event.top, $
          title='Please select a cosmic ray mask file')
      if filename[0L] ne '' then begin
        (*state).ex.crfile=filename[0L]
        (*state).ex.crfileset=1L

        ;; Updating widgets:
        tmp=file_basename((*state).ex.crfile)
        widget_control,(*state).tcfilwid,set_value=tmp
      endif else begin ;; filename[0L] ne ''
        (*state).ex.cruex=0L
        widget_control,(*state).tcfilwid,set_value=''
        widget_control,(*state).bcosswid,set_button=0L
        widget_control,(*state).bcoscwid,sensitive=0L
      endelse ;; filename[0L] ne ''
    end ;; case: 'crbrowse'

    'excrmask': begin
      (*state).ex.cruex=~(*state).ex.cruex
      widget_control,(*state).bcoscwid,sensitive=(*state).ex.cruex

      if ~(*state).ex.crfileset and (*state).ex.cruex then begin
        crfilename=*(*state).ex.file
        if (*(*state).ostate).nblocks gt 1L then begin
          if (*(*state).ostate).userparfile ne '' then $
            readcol,(*state).userparfile,uparname,uparvalue,format='a,a', $
                comment=';',silent=(*(*state).ostate).verbose lt 3,delimiter=' '

          p3d_misc_read_params,*(*(*state).ostate).parname, $
              *(*(*state).ostate).parvalue,'blocksfx_1',bsfx,/must_exist, $
              uparname=uparname,uparvalue=uparvalue,topwid=event.top, $
              logunit=(*(*state).ostate).logunit, $
              verbose=(*(*state).ostate).verbose,error=error, $
              debug=(*(*state).ostate).debug
          if error ne 0 then return
          bsfx=strtrim(bsfx,2L)

          tmp=strpos(crfilename,bsfx+(*(*state).ostate).sfx,/reverse_search)
          crfilename=strmid(crfilename,0L,tmp)+'_imcmb'
        endif else begin
          tmp=strpos(crfilename,(*(*state).ostate).sfx,/reverse_search)
          if tmp ne -1L then crfilename=strmid(crfilename,0L,tmp)
        endelse ;; (*(*state).ostate).nblocks gt 1L

        crfilename+=(*(*state).ostate).crsfx

        fileexists=0L
        tarr=['','.gz','.Z']
        tmp1=file_test((*state).ex.opath+crfilename+tarr,/read,/regular)
        tmp2=file_test((*state).ex.path +crfilename+tarr,/read,/regular)
        if max(tmp1) eq 1L then begin
          idx=where(tmp1)
          crfilename=(*state).ex.opath+crfilename+tarr[idx[0L]]
          fileexists=1L
        endif else if max(tmp2) eq 1L then begin
          idx=where(tmp2)
          crfilename=(*state).ex.path +crfilename+tarr[idx[0L]]
          fileexists=1L
        endif ;; max(tmp1) eq 1L

        if fileexists then begin
          (*state).ex.crfile=crfilename
          (*state).ex.crfileset=1L

          ;; Updating widgets:
          tmp=file_basename(crfilename)
          widget_control,(*state).tcfilwid,set_value=tmp
        endif else begin
          widget_control,(*state).tcfilwid,set_value='Please select an' + $
                         ' existing file.'

          sfx=(*(*state).ostate).crsfx
          filter=['*'+sfx+';*'+sfx+'.gz;*'+sfx+'.Z','*'+sfx+['','.gz','.Z'], $
                  '*'+(*(*state).ostate).sfx+['','.gz','.Z']]
          filename=dialog_pickfile(path=(*state).ex.opath,/must_exist, $
              filter=filter,dialog_parent=event.top, $
              title='Please select a cosmic ray mask file')
          if filename[0L] ne '' then begin
            (*state).ex.crfile=filename[0L]
            (*state).ex.crfileset=1L

            ;; Updating widgets:
            tmp=file_basename((*state).ex.crfile)
            widget_control,(*state).tcfilwid,set_value=tmp
          endif else begin
            (*state).ex.cruex=0L
            widget_control,(*state).bcosswid,set_button=0L
            widget_control,(*state).bcoscwid,sensitive=0L
            widget_control,(*state).tcfilwid,set_value=''
          endelse ;; filename[0L] ne ''
        endelse ;; file_test(crfilename,/read,/regular)
      endif ;; ~(*state).ex.crfileset and (*state).ex.cruex
    end ;; case: 'excrmask'

    'crmkfile': begin
      name=strlowcase(tag_names(event,/structure_name))
      case name of
        'widget_text_ch': if event.ch ne 10b then return ;; Pressed return
        'widget_kbrd_focus': if event.enter then return  ;; Widget lost focus
        else: return
      endcase
      widget_control,event.id,get_value=val

      ;; Checking if only one filename was selected:
      tmp=strtrim(strsplit(val[0L],',',/extract,count=count),2L)
      if count gt 1L then begin
        tmp=file_basename((*state).ex.crfile)
        widget_control,(*state).tcfilwid,set_value=tmp
        return
      endif

      ;; Checking if the entered filename exists:
      tmp=strpos(val[0L],path_sep(),/reverse_search)
      if tmp eq -1L then tmp=(*state).ex.path+val[0L]
      tmp2=(*state).ex.opath+file_basename(tmp)
      if ~file_test(tmp ,/read,/regular) and $
         ~file_test(tmp2,/read,/regular) then begin
        tmp=file_basename((*state).ex.crfile)
        widget_control,(*state).tcfilwid,set_value=tmp
        return
      endif

      ;; Prioritizing a file in the output directory:
      if file_test(tmp2,/read,/regular) then tmp=tmp2

      (*state).ex.crfile=tmp
      (*state).ex.crfileset=1L

      tmp=file_basename(tmp)
      widget_control,(*state).tcfilwid,set_value=tmp
    end ;; case: 'crmkfile'

    'wave': begin
      (*state).ex.wave=~(*state).ex.wave
      widget_control,(*state).bwavewid,sensitive=(*state).ex.wave
      widget_control,(*state).bbatewid,sensitive=(*state).ex.wave
    end

    'dispshift': (*(*state).ostate).dodispshift=~(*(*state).ostate).dodispshift
    'promptwave':    (*state).ex.waveprompt     =~(*state).ex.waveprompt
    'flat':          (*state).ex.flat           =~(*state).ex.flat
    'e3d':           (*state).ex.savee3d        =~(*state).ex.savee3d
    'telluric_none': (*(*state).ostate).skyalign=''
    'telluric_lo':   (*(*state).ostate).skyalign=(*state).tello
    'telluric_hi':   (*(*state).ostate).skyalign=(*state).telhi

    'execute': begin
      (*state).ex.object=1L
      widget_control,event.top,/destroy
      widget_control,/hourglass
      return
    end

    'cancel': begin
      (*state).ex.object=0L
      widget_control,event.top,/destroy
      return
    end

  endcase ;; uval

  return
END ;;; end of: p3d_gui_extract_menu_event


PRO p3d_gui_extract_menu,state,topwid,ex,sexmethod,exfilename,crfilename
  compile_opt hidden,IDL2

  d=(*state).d

  ex={path:(*state).inputpath, $
      opath:(*state).sepwpath?(*state).outputpath:(*state).inputpath, $
      file:ptr_new(['']),crsub:0L,waveprompt:0L, $
      bias:0L,flat:0L,wave:0L,object:0L,savee3d:0L, $
      cruex:0L,crfile:'',crfileset:0L, $
      shiftreffile:'',shiftrefpath:''}

;  if (*state).cur[d,(*state).nob] ne -1L then begin
    ;; If there a loaded/extracted object file? Then that file is used
    ;; for starters:

;    num=(*state).cur[d,(*state).nob]
;    file=(*state).file[d,(*state).nob,num]
    ;[d,(*state).nob,num]

;    tmppos=strpos(file,'.',/reverse_search)
;    fsfx=strmid(file,tmppos)
;
;    tmppos=strpos(file,'_oextr')
;    (*state).extractfile=file
; endif; else begin
;    (*state).extractfile=''
;    (*state).extractpath=''
;  endelse
  ex.shiftreffile=(*ex.file)[0L]
  ex.shiftrefpath=ex.path

  assign_obj=1L

  bmofile=filepath(subdir=['resource','bitmaps'],'open.bmp')

  ;; Setting up the widget hierarchy:
  bbasewid=widget_base(title='p3d: object data extraction options',/column, $
      /base_align_center,group_leader=topwid,/modal,resource_name='p3d', $
      xpad=2L,ypad=2L,space=10L)

  boverwid=widget_base(bbasewid,/column,/base_align_left, $
      space=2L,xpad=0L,ypad=0L)

  bfilewid=widget_base(boverwid,/row,/base_align_center, $
      space=2L,xpad=0L,ypad=0L)
  lfilewid=widget_label(bfilewid,value='Extract File:    ',/align_left)
  tmp=strjoin(*ex.file,', ')
  tfilewid=widget_text(bfilewid,value=tmp,uvalue='filetext', $
      /editable,/all_events,xsize=40L,resource_name='field',/kbrd_focus_events)
  bfbrowid=widget_button(bfilewid,value=bmofile,uvalue='browse',/bitmap)

  brefewid=widget_base(boverwid,/row,/base_align_left, $
      space=2L,xpad=0L,ypad=0L)
  lrefewid=widget_label(brefewid,value='Shift Reference: ',/align_left)
  tmp=(*ex.file)[0L]
  treftwid=widget_text(brefewid,value=tmp,uvalue='reftext', $
      /editable,/all_events,xsize=40L,resource_name='field',/kbrd_focus_events)
  brefbwid=widget_button(brefewid,value=bmofile,uvalue='refbrowse',/bitmap)

  bsetrwid=widget_base(brefewid,/row,/nonexclusive,/base_align_center)
  bsetrwid=widget_button(bsetrwid,value=' ',uvalue='setref')


  llabewid=widget_label(bbasewid, $
      value='Perform the following operations on the data:')

  bbuttwid=widget_base(bbasewid,/column,/frame,/base_align_left, $
      resource_name='field',space=10L)
  bbut2wid=widget_base(bbuttwid,/column,/nonexclusive)
  sensitive=(*state).cur[(*state).d,(*state).nbi] ne -1L
  bbiaswid=widget_button(bbut2wid,value='Subtract the master bias', $
      uvalue='bias',sensitive=sensitive)
  trcshift=widget_button(bbut2wid,value='The trace mask is spatially shi' + $
      'fted',uvalue='traceshift')

  bbutdwid=widget_base(bbuttwid,/column,xpad=0L,ypad=0L,space=0L)
  bbutqwid=widget_base(bbutdwid,/column,/nonexclusive,xpad=0L,ypad=0L,space=0L)
  crsens=0L
  bcosmwid=widget_button(bbutqwid,value='Remove cosmic ray hits (when 1 f' + $
      'ile is selected)',uvalue='cosmic',sensitive=crsens)
  bcosbwid=widget_base(bbutdwid,/row,/base_align_center, $
      xpad=0L,ypad=0L,space=0L,sensitive=sexmethod eq 'optimal' and crsens)
  lspacwid=widget_label(bcosbwid,value='',xsize=15L)
  bbut5wid=widget_base(bcosbwid,/column,/nonexclusive,xpad=0L,ypad=0L,space=0L)
  bcosswid=widget_button(bbut5wid,value='Use an existing cosmic ray mask', $
      uvalue='excrmask')
  bcoscwid=widget_base(bbutdwid,/row,/base_align_center,xpad=0L,ypad=0L, $
      sensitive=sexmethod eq 'optimal' and crsens)
  lspacwid=widget_label(bcoscwid,value='',xsize=15L)
  tcfilwid=widget_text(bcoscwid,value=tmp,uvalue='crmkfile', $
      /editable,/all_events,xsize=43L,resource_name='crfield', $
      /kbrd_focus_events)
  bcbrowid=widget_button(bcoscwid,value=bmofile,uvalue='crbrowse',/bitmap)

  bbutdwid=widget_base(bbuttwid,/column,xpad=0L,ypad=0L,space=0L)
  bbut6wid=widget_base(bbutdwid,/column,/nonexclusive,xpad=0L,ypad=0L,space=0L)
  sensitive=(*state).cur[(*state).d,(*state).ndi] ne -1L
  bwavcwid=widget_button(bbut6wid,value='Wavelength calibrate using the di' + $
      'spersion mask',uvalue='wave',sensitive=sensitive)

  bwavbwid=widget_base(bbutdwid,/row,/base_align_center, $
      xpad=0L,ypad=0L,space=0L)
  lspacwid=widget_label(bwavbwid,value='',xsize=15L)
  bbut3wid=widget_base(bwavbwid,/column,/nonexclusive,xpad=0L,ypad=0L,space=0L)
  bwavswid=widget_button(bbut3wid,value='The dispersion mask is spatially' + $
      ' shifted',uvalue='dispshift')
  bwavewid=widget_button(bbut3wid,value='Prompt for wavelength-related pro' + $
      'perties',uvalue='promptwave')

  tello=!p3d_path+'data'+path_sep()+'tables'+path_sep()+'linelists'+ $
            path_sep()+'telluric_lines_lores.dat'
  telhi=!p3d_path+'data'+path_sep()+'tables'+path_sep()+'linelists'+ $
            path_sep()+'telluric_lines_hires.dat'

  bbatewid=widget_base(bbutdwid,/column,xpad=0L,ypad=0L,space=0L)
  batmpwid=widget_base(bbatewid,/row,/base_align_center, $
      xpad=0L,ypad=0L,space=0L)
  lspacwid=widget_label(batmpwid,value='',xsize=15L)
  llab1wid=widget_label(batmpwid,value='Align the disp.mask with telluric' + $
      ' lines:')
  batmpwid=widget_base(bbatewid,/row,xpad=0L,ypad=0L,space=0L)
  lspacwid=widget_label(batmpwid,value='',xsize=15L)
  batmpwid=widget_base(batmpwid,/row,/exclusive, $
      xpad=0L,ypad=0L,space=0L)
  bnonewid=widget_button(batmpwid,value='none||user par.file', $
      uvalue='telluric_none')
  blowrwid=widget_button(batmpwid,value='Lo-res',uvalue='telluric_lo')
  bhigrwid=widget_button(batmpwid,value='Hi-res',uvalue='telluric_hi')

  bbut4wid=widget_base(bbuttwid,/column,/nonexclusive,/base_align_center)
  sensitive=(*state).cur[(*state).d,(*state).nfl] ne -1L
  bflatwid=widget_button(bbut4wid,value='Normalize the resulting spectra ' + $
      'using the flat field',uvalue='flat',sensitive=sensitive)

  bbut4wid=widget_base(bbuttwid,/column,/nonexclusive,/base_align_center)
  bse3dwid=widget_button(bbut4wid,value='Export the resulting spectra (als' + $
      'o) in the E3D-format',uvalue='e3d')

  bexebwid=widget_base(bbasewid,/row,/base_align_center,space=20L,/grid_layout)
  bexecwid=widget_button(bexebwid,value='  OK  ',uvalue='execute')
  bcancwid=widget_button(bexebwid,value='Cancel',uvalue='cancel')

  widget_control,brefbwid,sensitive=1L-assign_obj
  widget_control,treftwid,editable= 1L-assign_obj
  widget_control,bsetrwid,set_button=assign_obj
  widget_control,bbiaswid,set_button=ex.bias,sensitive=(*state).pisreadybias
  widget_control,trcshift,set_button=(*state).dotraceshif, $
                 sensitive=(*state).pisreadybox
  widget_control,bcosmwid,set_button=ex.crsub
  widget_control,bwavcwid,set_button=ex.wave,sensitive=(*state).pisreadydisp
  widget_control,bwavswid,set_button=(*state).dodispshift, $
                 sensitive=(*state).pisreadybox*(*state).pisreadydisp* $
                 (1L-(*state).displibset)
  widget_control,bwavewid,set_button=ex.waveprompt, $
                 sensitive=ex.wave*(*state).pisreadydisp
  widget_control,bflatwid,set_button=ex.flat,sensitive=(*state).pisreadyflat
  if (*state).skyalign eq '' then begin
    widget_control,bnonewid,/set_button
  endif else if (*state).skyalign eq tello then begin
    widget_control,blowrwid,/set_button
  endif else if (*state).skyalign eq telhi then begin
    widget_control,bhigrwid,/set_button
  endif
  widget_control,bbatewid,sensitive=ex.wave

  widget_control,bexecwid,sensitive=(*ex.file)[0L] ne ''
  widget_control,bbasewid,/realize

  ;; Centering the tool:
  tmp=widget_info(bbasewid,/geometry)
  ttp=widget_info((*state).top,/geometry)
  xpos=ttp.xoffset+ttp.xsize/2L-tmp.xsize/2L
  ypos=ttp.yoffset+ttp.ysize/2L-tmp.ysize/2L
  widget_control,bbasewid,tlb_set_xoffset=xpos
  widget_control,bbasewid,tlb_set_yoffset=ypos

  sstate={assignobj:assign_obj,tello:tello,telhi:telhi,sexmethod:sexmethod, $
          bexecwid:bexecwid,brefbwid:brefbwid,bsetrwid:bsetrwid, $
          bwavewid:bwavewid,bbatewid:bbatewid,bcosswid:bcosswid, $
          bcosmwid:bcosmwid,bcoscwid:bcoscwid,bcosbwid:bcosbwid, $
          tcfilwid:tcfilwid,tfilewid:tfilewid,treftwid:treftwid,ex:ex, $
          ostate:state,verbose:(*state).verbose,debug:(*state).debug}

  ;; Storing the state information in the widget tree:
  pstate=ptr_new(sstate,/no_copy)
  widget_control,bbasewid,set_uvalue=pstate

  ;; Handing over the event handling to xmanager:
  xmanager,'p3d_gui_extract_menu',bbasewid

  ex=(*pstate).ex
  if ptr_valid(ex.file) then begin
    if ex.crsub then begin
      if ex.cruex then begin
        if file_test(ex.crfile,/read,/regular) then begin
          ;; Testing if the file is a fits file:
          tmp=readfits(ex.crfile,silent=(*state).verbose lt 3,/no_unsigned)
          if size(tmp,/n_dimensions) eq 0L and tmp[0L] eq -1L then begin
            ret=dialog_message(['The selected cosmic ray mask file is no', $ $
                                'fits file:',file_basename(ex.crfile)], $
                               /center,dialog_parent=(*state).top,/information)
            ex.object=0L ;; Canceling the extraction procedure
          endif else begin
            crfilename=ex.crfile
          endelse
        endif
      endif else begin
        ret=dialog_message(['An interactive cosmic ray hit removal', $
                            'is not yet implemented. Please select', $
                            'an already created cosmic ray mask.'], $
                           /center,dialog_parent=(*state).top,/information)
        ex.object=0L ;; Canceling the extraction procedure
      endelse
    endif ;; ex.crsub

    exfilename=*(ex.file)
    ptr_free,ex.file
  endif ;; ptr_valid(ex.file)

  widget_control,/hourglass

  return
END ;;; end of: p3d_gui_extract_menu


PRO p3d_gui_flatfield_menu_event,event
  compile_opt hidden,IDL2

  rname='p3d_gui_flatfield_menu_event: '

  ;;========================================------------------------------
  ;; Get the state information:

  widget_control,event.top,get_uvalue=state

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=event.top
      catch,/cancel
      return
    endif
  endif ;; ~(*state).debug

  tmpstrname=strlowcase(tag_names(event,/structure_name))
  if tmpstrname eq 'widget_kbrd_focus' then if event.enter then return

  ;;========================================------------------------------
  ;; Get the user value of the event structure:

  widget_control,event.id,get_uvalue=uval

  case uval of

    'ffbrowse': begin
      xfx=(*(*state).ostate).xxsfx+(*(*state).ostate).sfx
      yfx=(*(*state).ostate).icsfx+(*(*state).ostate).sfx
      filter=['*'+xfx+';*'+xfx+'.gz;*'+xfx+'.Z','*'+xfx+['','.gz','.Z'], $
              '*'+yfx+';*'+yfx+'.gz;*'+yfx+'.Z','*'+yfx+['','.gz','.Z']]
      filename=dialog_pickfile(path=(*(*state).ostate).inputpath, $
          /must_exist,/multiple_files,filter=filter,get_path=tmppath, $
          dialog_parent=event.top,title='p3d: please select one or three ' + $
          'or more flatfield image files')
      if filename[0L] ne '' then begin
        if ptr_valid((*state).filename) then ptr_free,(*state).filename
        (*state).filename=ptr_new(filename)
        (*state).path=tmppath

        ;; Updating widgets:
        tmp=strjoin(file_basename(filename),', ')
        widget_control,(*state).tfilewid,set_value=tmp
        widget_control,(*state).bexecwid,/sensitive
      endif ;; filename ne ''
    end ;; case: 'browse'

    'trbrowse': begin
      sfx=(*(*state).ostate).sfx & str=(*(*state).ostate).trsfx
      filter=['*'+str+'*'+sfx+';*'+str+'*'+sfx+'.gz;*'+str+'*'+sfx+'.Z', $
              '*'+str+'*'+sfx+['','.gz','.Z'],'*'+sfx+['','.gz','.Z']]
      filename=dialog_pickfile(path=(*(*state).ostate).inputpath, $
          /must_exist,filter=filter,dialog_parent=event.top, $
          title='Please select a trace mask')
      if filename ne '' then begin
        (*state).tracemask=filename
        ;; Updating widgets:
        widget_control,(*state).tfiltwid,set_value=file_basename(filename)
      endif ;; filename ne ''
    end ;; case: 'browse'

    'dmbrowse': begin
      sfx=(*(*state).ostate).sfx & str=(*(*state).ostate).dmsfx
      filter=['*'+str+'*'+sfx+';*'+str+'*'+sfx+'.gz;*'+str+'*'+sfx+'.Z', $
              '*'+str+'*'+sfx+['','.gz','.Z'],'*'+sfx+['','.gz','.Z']]
      filename=dialog_pickfile(path=(*(*state).ostate).inputpath, $
          /must_exist,filter=filter,dialog_parent=event.top, $
          title='Please select a dispersion mask')
      if filename ne '' then begin
        (*state).dispmask=filename

        ;; Updating widgets:
        widget_control,(*state).tfildwid,set_value=file_basename(filename)
      endif ;; filename ne ''
    end ;; case: 'browse'

    'self': begin
      (*state).self=~(*state).self
      widget_control,(*state).bftrswid,set_button=(*state).self

      widget_control,(*state).bfiltwid,sensitive=~(*state).self
    end ;; case: 'self'

    'bdmask': begin
      (*state).bdmask=~(*state).bdmask
      widget_control,(*state).bftrdwid,set_button=~(*state).bdmask

      widget_control,(*state).bfildwid,sensitive=(*state).bdmask
    end ;; case: 'bdmask'

    'execute': begin
      (*state).extractflat=1L
      widget_control,event.top,/destroy
      widget_control,/hourglass
      return
    end

    'cancel': begin
      (*state).extractflat=0L
      widget_control,event.top,/destroy
      return
    end

  endcase ;; uval

  return
END ;;; end of: p3d_gui_flatfield_menu_event


PRO p3d_gui_flatfield_menu,state,ret,tmppath, $
        filename=filename,tracemask=otracemask,dispmask=dispmask
  compile_opt hidden,IDL2

  num=(*state).cur[(*state).d,(*state).ntr]
  if num ne -1L then $
     tracemask=reform((*state).path[(*state).d,(*state).ntr,num] + $
                      (*state).file[(*state).d,(*state).ntr,num])

  bmofile=filepath(subdir=['resource','bitmaps'],'open.bmp')

  ;; Setting up the widget hierarchy:
  bbasewid=widget_base(title='p3d: Flat field extraction options',/column, $
      /base_align_center,group_leader=(*state).top,/modal, $
      resource_name='p3d',xpad=2L,ypad=2L,space=10L)

  boverwid=widget_base(bbasewid,/column,/base_align_left, $
      space=2L,xpad=0L,ypad=0L)

  ;; Raw flat field images:
  bfilewid=widget_base(boverwid,/row,/base_align_center, $
      space=2L,xpad=0L,ypad=0L)
  lfilewid=widget_label(bfilewid,value='Flat field:      ',/align_left)
  tfilewid=widget_text(bfilewid,value='',uvalue='filetext', $
      xsize=40L,resource_name='field')
  bfbrewid=widget_button(bfilewid,value=bmofile,uvalue='ffbrowse',/bitmap)

  ;; Trace mask:
  btrmpwid=widget_base(boverwid,/column,/base_align_left,/frame, $
      space=2L,xpad=0L,ypad=0L)

  batmpwid=widget_base(btrmpwid,/row,/nonexclusive,/base_align_center, $
      xpad=0L,ypad=0L,space=0L)
  bftrswid=widget_button(batmpwid,value='Use the flat field raw data for' + $
      ' finding spectrum traces',uvalue='self')
  bfiltwid=widget_base(btrmpwid,/row,/base_align_center, $
      space=2L,xpad=0L,ypad=0L,sensitive=0L)
  lfiltwid=widget_label(bfiltwid,value='Trace mask:      ',/align_left)
  tfiltwid=widget_text(bfiltwid,value=file_basename(tracemask), $
      uvalue='tracetext',xsize=40L,resource_name='field')
  bfbrtwid=widget_button(bfiltwid,value=bmofile,uvalue='trbrowse',/bitmap)

  ;; Dispersion mask:
  btrmpwid=widget_base(boverwid,/column,/base_align_left,/frame, $
      space=2L,xpad=0L,ypad=0L)

  batmpwid=widget_base(btrmpwid,/row,/nonexclusive,/base_align_center, $
      xpad=0L,ypad=0L,space=0L)
  bftrdwid=widget_button(batmpwid,value='Do not use a dispersion mask w' + $
      'ith the flat field data',uvalue='bdmask')

  bfildwid=widget_base(btrmpwid,/row,/base_align_left, $
      space=2L,xpad=0L,ypad=0L,sensitive=0L)
  lfildwid=widget_label(bfildwid,value='Dispersion mask: ',/align_left)
  tfildwid=widget_text(bfildwid,value='', $
      uvalue='dispmtext',xsize=40L,resource_name='field')
  bdbrmwid=widget_button(bfildwid,value=bmofile,uvalue='dmbrowse',/bitmap)

  bexebwid=widget_base(bbasewid,/row,/base_align_center,space=20L,/grid_layout)
  bexecwid=widget_button(bexebwid,value='  OK  ',uvalue='execute')
  bcancwid=widget_button(bexebwid,value='Cancel',uvalue='cancel')

  widget_control,bftrswid,/set_button
  widget_control,bftrdwid,/set_button
  widget_control,bbasewid,/realize

  ;; Centering the tool:
  tmp=widget_info(bbasewid,/geometry)
  ttp=widget_info((*state).top,/geometry)
  xpos=ttp.xoffset+ttp.xsize/2L-tmp.xsize/2L
  ypos=ttp.yoffset+ttp.ysize/2L-tmp.ysize/2L
  widget_control,bbasewid,tlb_set_xoffset=xpos
  widget_control,bbasewid,tlb_set_yoffset=ypos

  sstate={tracemask:tracemask,dispmask:'',filename:ptr_new(), $
          self:1L,bdmask:0L, $
          bexecwid:bexecwid,bfiltwid:bfiltwid,bfildwid:bfildwid, $
          tfilewid:tfilewid,bftrswid:bftrswid,bftrdwid:bftrdwid, $
          tfiltwid:tfiltwid,tfildwid:tfildwid, $
          path:'',extractflat:0L, $
          ostate:state,verbose:(*state).verbose,debug:(*state).debug}

  ;; Storing the state information in the widget tree:
  pstate=ptr_new(sstate,/no_copy)
  widget_control,bbasewid,set_uvalue=pstate

  ;; Handing over the event handling to xmanager:
  xmanager,'p3d_gui_flatfield_menu',bbasewid

  ;; Saving the chosen values:
  if (*pstate).extractflat then begin
    ret=1L
    tmppath=(*pstate).path
    filename=*(*pstate).filename
    if ~(*pstate).self then otracemask=(*pstate).tracemask
    if (*pstate).bdmask and (*pstate).dispmask ne '' then $
       dispmask=(*pstate).dispmask
  endif else ret=0L

  ptr_free,(*pstate).filename
  ptr_free,pstate

  widget_control,/hourglass

  return
END ;;; end of: p3d_gui_flatfield_menu


PRO p3d_gui_savedata,type,i,hdr,state
  compile_opt hidden,IDL2

  if ptr_valid((*state).data[(*state).d,type,i]) then $
      ptr_free,(*state).data[(*state).d,type,i]
  (*state).data[(*state).d,type,i]=ptr_new(*(*state).curdata[(*state).d,type])

  if ptr_valid((*state).hdr[(*state).d,type,i]) then $
      ptr_free,(*state).hdr[(*state).d,type,i]
  (*state).hdr[(*state).d,type,i]=ptr_new(hdr)

  return
END ;;; procedure: p3d_gui_savedata


PRO p3d_gui_draw,i,state,topwid,data, $
        left=left,right=right,arrow=arrow,auto=auto
  compile_opt hidden,IDL2

   left=keyword_set(left)
  right=keyword_set(right)
  arrow=keyword_set(arrow)
  d=(*state).d

  if ~n_elements(data) then begin
    tmp=(*state).curb[d]
    image=reform((*state).daxis? $
                 (*(*state).curdata[d,(*state).nob])[tmp,*]: $
                 (*(*state).curdata[d,(*state).nob])[*,tmp])
  endif else image=data

  if ~keyword_set(auto) then begin
    minval=(*state).ccmin[d]
    maxval=(*state).ccmax[d]
  end 

  ;; Retrieving position table properties:
  idx=d*(*state).nblocks;+(*state).block ;; 10.10.2009 /CS
  posfile =(*(*state).pos[idx]).file
  shape   =(*(*state).pos[idx]).shape
  rownum  =(*(*state).pos[idx]).fpos
  xpos    =(*(*state).pos[idx]).fpx
  ypos    =(*(*state).pos[idx]).fpy
  lenssize=(*(*state).pos[idx]).fles

  ;; Loading the colors that are used with this tool:
  tvlct,red,green,blue,/get
  tvlct,(*state).colors.red,(*state).colors.green,(*state).colors.blue

  ;; Updating the left object image on the main panel:
  if left then begin
    wset,(*state).wid[d].dleft
    p3d_display_telescl_postable,image,(*state).leftsize, $
        orientation=(*state).orientation[d,i], $
        locut=minval,hicut=maxval,/science,noerase=0L,posfile=posfile, $
        rownum=rownum,xpos=xpos,ypos=ypos,arrow=arrow, $
        lens_size=lenssize,shape=shape,bottom=(*state).cbottom, $
        topwid=topwid,logunit=(*state).logunit,verbose=(*state).verbose, $
        error=error,debug=(*state).debug
    if error ne 0 then return
  endif ;; left

  ;; Updating the right object image on the main panel:
  if right then begin
     wset,(*state).wid[(*state).d].dright[i]
     p3d_display_telescl_postable,image,(*state).rightsize, $
         orientation=(*state).orientation[(*state).d,i], $
         locut=minval,hicut=maxval,/science,noerase=0L,posfile=posfile, $
         rownum=rownum,xpos=xpos,ypos=ypos, $
         lens_size=lenssize,shape=shape,bottom=(*state).cbottom, $
         topwid=topwid,logunit=(*state).logunit,verbose=(*state).verbose, $
         error=error,debug=(*state).debug
     if error ne 0 then return
   endif ;; right

  ;; Resetting the colors:
  tvlct,red,green,blue

  ;; Re-sensitizing the colortable menu button if xloadct is closed:
  if ~xregistered('xloadct') then $
     widget_control,(*state).wix.bcolo,sensitive=1L

  return
END ;;; procedure: p3d_gui_draw


PRO p3d_gui_setwave,i,state,updateuname=updateuname
  compile_opt hidden,IDL2

  d=(*state).d
  if ~(*state).crvd[d,0L,i] then begin
    ;; Not showing any value:
    widget_control,(*state).wid[d].lwave,set_value=''

    if keyword_set(updateuname) then $
       widget_control,(*state).wid[d].lwave, $
           set_uname='When the data is wavelngth calibrated this field sh' + $
           'ows the wavelength.'
  endif else begin
    ;; Using the wavelength calibrated value:
    wavelength=(*state).crvd[d,0L,i]+ $
               (*state).crvd[d,1L,i]*((*state).curb[d]+0.5)
    tmp=string(wavelength,format='(f8.2,"")')
    widget_control,(*state).wid[d].lwave,set_value=strtrim(tmp,2L)

    if keyword_set(updateuname) then $
       widget_control,(*state).wid[d].lwave,set_uname='This field shows t' + $
           'he wavelength of the currently selected wavelength in [].'
  endelse

  return
END ;;; procedure: p3d_gui_setwave


PRO p3d_gui_xloadct,data=state
  compile_opt hidden,IDL2

  if ptr_valid((*state).curdata[(*state).d,(*state).nob]) then begin
    i=(*state).cur[(*state).d,(*state).nob]

    ;; It would be nice if one could set (*state).colortable here:

    ;; Loading the colors to use:
    tvlct,red,green,blue,/get
    (*state).colors={red:red,green:green,blue:blue}

    ;; Redrawing the left and right object images on the panel:
    p3d_gui_draw,i,state,(*state).top,image,/left,/right
  endif

  return
END ;;; procedure: p3d_gui_xloadct


PRO p3d_gui_event,event
  compile_opt hidden,IDL2

  rname='p3d_gui_event: '

  ;;========================================------------------------------
  ;; Retrieving the widget state structure:

  widget_control,event.top,get_uvalue=state

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=event.top
      catch,/cancel
      return
    endif
  endif ;; ~(*state).debug

  tmpstrname=strlowcase(tag_names(event,/structure_name))

  ;;========================================------------------------------
  ;; Handling widget timer events first:

  if tmpstrname eq 'widget_timer' then begin
    ;; Issuing a reload of the data reduction log file viewer text widget:
    p3d_gui_timerdr,state,event

    return
  endif ;; tmpstrname eq 'widget_timer'

  ;;========================================------------------------------
  ;; Handling widget tracking and context events second:

  if tmpstrname eq 'widget_tracking' then begin
    if event.enter then begin  
      infostr=widget_info(event.id,/uname)
      widget_control,(*state).wix.lstat,set_value=infostr
    endif else widget_control,(*state).wix.lstat,set_value=''

    ;; Showing a directional arrow in the object window when entering
    ;; the widget:
    if event.id eq (*state).wid[(*state).d].dobjl then begin
      if ptr_valid((*state).curdata[(*state).d,(*state).nob]) then begin
        i=(*state).cur[(*state).d,(*state).nob]
        p3d_gui_draw,i,state,event.top,/left,arrow=event.enter
      endif
    endif ;; end: event.id eq (*state).wid[(*state).d].dobjl

    return ;; Nothing more to do, returning:
  endif ;; strlowcase(tag_names(event,/structure_name)) eq 'widget_tracking'

  if event.id eq (*state).wid[(*state).d].dobjl then begin
    if event.release eq 4b then widget_displaycontextmenu, $
       event.id,event.x,event.y,(*state).wid[(*state).d].bdrco
  endif ;; 'widget_context'

  if tmpstrname eq 'widget_kbrd_focus' then if event.enter then return

  ;;========================================------------------------------
  ;; Retrieving the widget user value trying to see if there is an index also:

  widget_control,event.id,get_uvalue=uval

  index=-1L
  on_ioerror,noindexed_event
  tmppos=strpos(uval,'_',/reverse_search)
  index=long(strmid(uval,tmppos+1L))
  uval=strmid(uval,0L,tmppos)

  noindexed_event:
  on_ioerror,NULL
  
  ;;========================================------------------------------
  ;; Handling all remaining events:

  widget_control,/hourglass

  case uval of

    ;;========================================------------------------------
    ;;========================================------------------------------
    ;; Shift boxes:

    'boxes': begin
      filename=dialog_pickfile(path=(*state).inputpath, $
          get_path=tmppath,/must_exist,filter='*.fits', $
          dialog_parent=event.top,title='p3d: please select a shift refere' + $
          'nce image file')
      if filename ne '' then begin
        (*state).inputpath=tmppath
        image=readfits(filename,silent=(*state).verbose lt 3,/no_unsigned)
;;        p3d_display_set_box,image,boxes,topwid=event.top, $
;;                            verbose=(*state).verbose,error=error
;;        if error ne 0 then return

        if n_tags(boxes) ne 0 then begin
          filepath=file_dirname(filename,/mark_directory)
          slen=strlen(filepath)
          boxname=strmid(filename,slen)

          pos=strpos(boxname,'.fits')
          boxname=strmid(boxname,0,pos)
          d=(*state).d

          filename=dialog_pickfile(path=(*state).inputpath,get_path=tmppath, $
              file=boxname+'.box',dialog_parent=event.top,title='p3d: plea' + $
              'se elect a filename for writing boxes')

          if filename ne '' then begin
;;          p3d_utils_writebox,filename,boxes ;; not yet prepared

            slen=strlen((*state).inputpath)
            boxname=strmid(filename,slen)

            p3d_gui_position_new_file,(*state).nbx,boxname,boxes, $
                state,error=error,debug=(*state).debug
            if error ne 0 then return

            if ~(*state).nf_cancel then begin
              num=(*state).cur[d,(*state).nbx]
              widget_control,(*state).wid[d].tboxf, $
                  set_value=(*state).file[d,(*state).nbx,num]
              widget_control,(*state).wid[d].lboxi, $
                             set_value=strtrim(num,2L)
              (*state).data[d,(*state).nbx,(*state).cur[d,(*state).nbx]]= $
                      *(*state).curdata[d,(*state).nbx]

              ;; this below cannot be correct:
              (*state).curdata[d,(*state).nfl]=ptr_new((*(*state).flat) $
                   [(*state).cur[d,(*state).nfl]])

              if ptr_valid((*state).hdr[d,(*state).nbx,num]) then $
                  ptr_free,(*state).hdr[d,(*state).nbx,num]
              (*state).hdr[d,(*state).nbx,num]= $
                 ptr_new(headfits(boxpath+boxname,exten=1))

              (*state).pisreadybox=1L
            endif ;; ~(*state).nf_cancel
          endif ;; filename ne ''
        endif ;; n_tags(boxes) ne 0
      endif ;; filename ne ''
    end ;; case: 'boxes'

    ;;========================================------------------------------
    ;; Browsing for an already reduced file:

    'browsebox': begin
      filename=dialog_pickfile(path=(*state).inputpath,get_path=tmppath, $
                   /must_exist,filter='*.box',dialog_parent=event.top, $
                   title='p3d: please select a reduced box file')
      if filename ne '' then begin
        (*state).inputpath=tmppath

;;        p3d_tool_readbox,filename,box,hdr, $
;;            topwid=event.top,verbose=(*state).verbose,error=error
;;        if error ne 0 then return

        boxname=strmid(filename,strlen((*state).inputpath))
        d=(*state).d

        p3d_gui_position_new_file,(*state).nbx,boxname,box, $
            state,/useinputpath,error=error,debug=(*state).debug
        if error ne 0 then return

        if ~(*state).nf_cancel then begin
          i=(*state).cur[d,(*state).nbx]
          widget_control,(*state).wid[d].tboxf, $
                         set_value=(*state).file[(*state).d,(*state).nbx,i]
          widget_control,(*state).wid[d].lboxi,set_value=strtrim(i,2L)

          if ptr_valid((*state).data[d,(*state).nbx,i]) then $
              ptr_free,(*state).data[d,(*state).nbx,i]
          (*state).data[d,(*state).nbx,i]= $
             ptr_new(*(*state).curdata[d,(*state).nbx])

          if ptr_valid((*state).hdr[d,(*state).nbx,i]) then $
              ptr_free,(*state).hdr[d,(*state).nbx,i]
          (*state).hdr[d,(*state).nbx,i]=ptr_new(hdr)

          (*state).pisreadybox=1L

        endif ;; ~(*state).nf_cancel
      endif ;; filename ne ''

    end ;; case: 'browsebox'

    ;;========================================------------------------------
    ;; Select an already loaded file:

    'selectbox': begin

      p3d_gui_select_file,(*state).nbx,state,error=error

      i=(*state).cur[(*state).d,(*state).nbx]
      if i ne -1L then begin
        widget_control,(*state).wid[(*state).d].tboxf, $
                       set_value=(*state).file[(*state).d,(*state).nbx,i]
        widget_control,(*state).wid[(*state).d].lboxi,set_value=strtrim(i,2L)
      endif

    end ;; case: 'selectbox'

    ;;========================================------------------------------
    ;;========================================------------------------------
    ;; Bias:

    'createmasterbias': begin
      p3d_gui_suffixes,state,error=error
      if error ne 0 then return
      xxsfx=(*state).xxsfx+(*state).sfx
      filter=['*'+xxsfx+';*'+xxsfx+'.gz;*'+xxsfx+'.Z', $
              '*'+xxsfx+['','.gz','.Z']]
      if (*state).xxsfx ne '' then begin
        filter=[filter, $
                '*'+(*state).sfx+';*'+(*state).sfx+'.gz;*'+(*state).sfx+'.Z', $
                '*'+(*state).sfx+['','.gz','.Z']]
      endif ;; (*state).xxsfx ne ''
      filename=dialog_pickfile(path=(*state).inputpath,get_path=tmppath, $
          /must_exist,/multiple_files,filter=filter, $
          dialog_parent=event.top,resource_name='pickfile', $
          title='p3d: please select three or more bias image files')
      if filename[0L] ne '' then begin
        (*state).inputpath=tmppath

        opath=(*state).sepwpath?(*state).outputpath:(*state).inputpath

        ;; Calling the master bias creation script:
        p3d_cmbias,filename,(*state).parfile,masterbias,icstr=(*state).icsfx, $
            ostr=(*state).mbsfx, $
	    userparfile=(*state).userparfile,ofilename=ofilename, $
            opath=opath,detector=(*state).d,sfx=(*state).sfx, $
            compress=(*state).compress,stawid=(*state).wix.lstat, $
	    topwid=event.top,logunit=(*state).logunit, $
	    verbose=(*state).verbose,error=error,debug=(*state).debug
        if error ne 0 then return

        ;; saving the data in the widget tool state variable:
        p3d_gui_position_new_file,(*state).nbi,file_basename(ofilename), $
	    masterbias,state,error=error,debug=(*state).debug
        if error ne 0 then return

        if ~(*state).nf_cancel then begin
          d=(*state).d
          i=(*state).cur[d,(*state).nbi]
          tmp=(*state).file[d,(*state).nbi,i]
          widget_control,(*state).wid[d].tbiaf,set_value=tmp,update=0L
          widget_control,(*state).wid[d].tbiaf, $
                         set_text_select=[strlen(tmp),0L]
          widget_control,(*state).wid[d].lbiai,set_value=strtrim(i,2L)
          widget_control,(*state).wid[d].bbivi,sensitive=i ne -1L

          ;; New flat/dispersion mask/object widgets:
          tmp=(*state).cur[d,(*state).ntr] ne -1L
          widget_control,(*state).wid[d].bdicr,sensitive=tmp
          widget_control,(*state).wid[d].bflcr,sensitive=tmp
          widget_control,(*state).wix.bdisp   ,sensitive=tmp
          widget_control,(*state).wix.bflat   ,sensitive=tmp

          tmp=(*state).cur[d,(*state).ntr] ne -1L
          widget_control,(*state).wid[d].bobcr,sensitive=tmp
          widget_control,(*state).wix.bextr   ,sensitive=tmp

          ;; Saving the data:
          hdr=headfits(ofilename)
          p3d_gui_savedata,(*state).nbi,i,hdr,state

          ;; The select button is made active if there are >=2 loaded objects:
          num=long(total(ptr_valid((*state).data[d,(*state).nbi,*])))
          widget_control,(*state).wid[d].bbise,sensitive=num ge 2L

          widget_control,event.top,update=1L

          (*state).pisreadybias=1L
        endif ;; ~(*state).nf_cancel
      endif ;; filename ne ''

    end ;; case: 'createmasterbias'


    ;;========================================------------------------------
    ;; Browsing for an already reduced file:

    'browsemasterbias': begin
      p3d_gui_suffixes,state,error=error
      path=(*state).sepwpath?(*state).outputpath:(*state).inputpath
      sfx=(*state).sfx & str=(*state).mbsfx
      filter=['*'+str+'*'+sfx+';*'+str+'*'+sfx+'.gz;*'+str+'*'+sfx+'.Z', $
              '*'+str+'*'+sfx+['','.gz','.Z'], $
              '*'+sfx+';*'+sfx+'.gz;*'+sfx+'.Z','*'+sfx+['','.gz','.Z']]
      filename=dialog_pickfile(path=path,get_path=tmppath,/must_exist, $
          filter=filter,dialog_parent=event.top, $
          title='p3d: please select a reduced masterbias image file')
      if filename ne '' then begin
        (*state).inputpath=tmppath

        ;; Checking if the DETECTOR of the file is the same as the used one:
        p3d_misc_checkfile,filename[0L],(*state).nd,(*state).d, $
            (*state).lkwrdlist,topwid=event.top,logunit=(*state).logunit, $
            verbose=(*state).verbose,error=error,debug=(*state).debug
        if error ne 0 then return

        bias=readfits(filename,hdr,silent=(*state).verbose lt 3,/no_unsigned)

        p3d_gui_position_new_file,(*state).nbi,file_basename(filename),bias, $
            state,/useinputpath,error=error,debug=(*state).debug
        if error ne 0 then return

        if ~(*state).nf_cancel then begin
          d=(*state).d
          i=(*state).cur[d,(*state).nbi]
          tmp=(*state).file[d,(*state).nbi,i]
          widget_control,(*state).wid[d].tbiaf,set_value=tmp
          widget_control,(*state).wid[d].tbiaf,set_text_select=[strlen(tmp),0L]
          widget_control,(*state).wid[d].lbiai,set_value=strtrim(i,2L)
          widget_control,(*state).wid[d].bbivi,sensitive=i ne -1L

          ;; New flat/dispersion mask/object widgets:
          tmp=(*state).cur[d,(*state).ntr] ne -1L
          widget_control,(*state).wid[d].bflcr,sensitive=tmp
          widget_control,(*state).wid[d].bdicr,sensitive=tmp
          widget_control,(*state).wix.bdisp   ,sensitive=tmp
          widget_control,(*state).wix.bflat   ,sensitive=tmp

          tmp=(*state).cur[d,(*state).ntr] ne -1L
          widget_control,(*state).wid[d].bobcr,sensitive=tmp
          widget_control,(*state).wix.bextr   ,sensitive=tmp

          ;; View the bias data:

          ;; Saving the data:
          p3d_gui_savedata,(*state).nbi,i,hdr,state

          ;; The select button is made active if there are >=2 loaded objects:
          num=long(total(ptr_valid((*state).data[d,(*state).nbi,*])))
          widget_control,(*state).wid[d].bbise,sensitive=num ge 2L

          (*state).pisreadybias=1L
        endif ;; ~(*state).nf_cancel
      endif ;; filename ne ''

    end ;; case: 'browsemasterbias'


    ;;========================================------------------------------
    ;; Select an already loaded file:

    'selectmasterbias': begin

      p3d_gui_select_file,(*state).nbi,state,error=error
      if error ne 0 then return

      d=(*state).d & nbi=(*state).nbi & i=(*state).cur[d,nbi]
      if i ne -1L then begin
        widget_control,(*state).wid[d].tbiaf,set_value=(*state).file[d,nbi,i]
        widget_control,(*state).wid[d].lbiai,set_value=strtrim(i,2L)
      endif
      widget_control,(*state).wid[d].bbivi,sensitive=i ne -1L

    end ;; case: 'selectmasterbias'

    ;;========================================------------------------------
    ;; View the bias data:

    'viewbias': begin

      num=(*state).cur[(*state).d,(*state).nbi]
      biasfile=reform((*state).path[(*state).d,(*state).nbi,num] + $
                      (*state).file[(*state).d,(*state).nbi,num])

      title='p3d: [bias] "'+biasfile+'"'
      p3d_imv,biasfile,title=title,bottom=(*state).cbottom, $
          xscrsize=xscrsize,yscrsize=yscrsize,group_leader=event.top, $
          verbose=(*state).verbose,error=error,debug=(*state).debug
      if error ne 0 then return

    end ;; case: 'viewbias'


    ;;========================================------------------------------
    ;;========================================------------------------------
    ;; Trace mask:

    'createtracemask': begin
      p3d_gui_suffixes,state,error=error
      if error ne 0 then return
      xfx=(*state).xxsfx+(*state).sfx
      yfx=(*state).icsfx+(*state).sfx
      filter=['*'+xfx+';*'+xfx+'.gz;*'+xfx+'.Z','*'+xfx+['','.gz','.Z'], $
              '*'+yfx+';*'+yfx+'.gz;*'+yfx+'.Z','*'+yfx+['','.gz','.Z']]
      filename=dialog_pickfile(path=(*state).inputpath,get_path=tmppath, $
          /must_exist,/multiple_files,filter=filter, $
          dialog_parent=event.top,title='p3d: please select one, or three ' + $
          'or more image files for tracing.')
      if filename[0L] ne '' then begin
        (*state).inputpath=tmppath

        opath=(*state).sepwpath?(*state).outputpath:(*state).inputpath

        num=(*state).cur[(*state).d,(*state).nbi]
        if num ne -1L then $
           masterbias=reform((*state).path[(*state).d,(*state).nbi,num] + $
                             (*state).file[(*state).d,(*state).nbi,num])

        ;; Calling the trace mask creation script:
        p3d_ctrace,filename,(*state).parfile,tracemask,icstr=(*state).icsfx, $
            ostr=(*state).trsfx, $
            masterbias=masterbias,userparfile=(*state).userparfile, $
            ofilename=ofilename,opath=opath,detector=(*state).d, $
            sfx=(*state).sfx,compress=(*state).compress, $
            stawid=(*state).wix.lstat,topwid=event.top, $
            logunit=(*state).logunit,verbose=(*state).verbose, $
            error=error,debug=(*state).debug
        if error ne 0 then return

        ;;=============================--------------------
        ;; Saving the data in the widget tool state variable:

        p3d_gui_position_new_file,(*state).ntr,file_basename(ofilename), $
	    tracemask,state,error=error,debug=(*state).debug
        if error ne 0 then return

        if ~(*state).nf_cancel then begin
          d=(*state).d
          i=(*state).cur[d,(*state).ntr]
          tmp=(*state).file[d,(*state).ntr,i]
          widget_control,(*state).wid[d].ttraf,set_value=tmp,update=0L
          widget_control,(*state).wid[d].ttraf, $
                         set_text_select=[strlen(tmp),0L]
          widget_control,(*state).wid[d].ltrai,set_value=strtrim(i,2L)
          widget_control,(*state).wid[d].btrvi,sensitive=i ne -1L

;          tmp=(*state).file[d,(*state).ntr,i]
;          widget_control,(*state).wix.bextr,sensitive=strtrim(tmp,2L) ne ''

          ;; New trace-rel/flat/dispersion mask/object widgets:
          ;;   trace shift
          if widget_info((*state).wid[d].btras,/valid_id) then $
             widget_control,(*state).wid[d].btras,sensitive=1L
;         widget_control,(*state).wix.btfoc,sensitive=1L ;; telescope focus

          tmp=(*state).cur[d,(*state).nbi] ne -1L
          widget_control,(*state).wid[d].bflcr,sensitive=tmp
          widget_control,(*state).wix.bflat   ,sensitive=tmp

          tmp=~(*state).displibset ;; que?
          widget_control,(*state).wid[d].bdicr,sensitive=tmp
          widget_control,(*state).wix.bdisp   ,sensitive=tmp

          tmp=(*state).cur[d,(*state).nbi] ne -1L
          widget_control,(*state).wid[d].bobcr,sensitive=tmp
          widget_control,(*state).wix.bextr   ,sensitive=tmp

          ;; Saving the data:
          hdr=headfits(ofilename)
          p3d_gui_savedata,(*state).ntr,i,hdr,state

          ;; The select button is made active if there are >=2 loaded objects:
          num=long(total(ptr_valid((*state).data[d,(*state).ntr,*])))
          widget_control,(*state).wid[d].btrse,sensitive=num ge 2L

          widget_control,event.top,update=1L
        endif ;; ~(*state).nf_cancel
      endif ;; filename ne ''

    end ;; case: 'createtracemask'


    ;;========================================------------------------------
    ;; Browsing for an already reduced file:

    'browsetracemask': begin
      p3d_gui_suffixes,state,error=error
      path=(*state).sepwpath?(*state).outputpath:(*state).inputpath
      sfx=(*state).sfx & str=(*state).trsfx
      filter=['*'+str+'*'+sfx+';*'+str+'*'+sfx+'.gz;*'+str+'*'+sfx+'.Z', $
              '*'+str+'*'+sfx+['','.gz','.Z'], $
              '*'+sfx+';*'+sfx+'.gz;*'+sfx+'.Z','*'+sfx+['','.gz','.Z']]
      filename=dialog_pickfile(path=path,get_path=tmppath,/must_exist, $
          filter=filter,dialog_parent=event.top, $
          title='p3d: please select a reduced trace mask file')
      if filename ne '' then begin
        (*state).inputpath=tmppath

        ;; Checking if the detector of the file is the same as the used one:
        p3d_misc_checkfile,filename,(*state).nd,(*state).d, $
            (*state).lkwrdlist,topwid=event.top,logunit=(*state).logunit, $
            verbose=(*state).verbose,error=error,debug=(*state).debug
        if error ne 0 then return

        traces=readfits(filename,hdr,silent=(*state).verbose lt 3,/no_unsigned)
        tracefile=file_basename(filename)

        p3d_gui_position_new_file,(*state).ntr,tracefile,traces, $
            state,/useinputpath,error=error,debug=(*state).debug
        if error ne 0 then return

        if ~(*state).nf_cancel then begin
          d=(*state).d
          i=(*state).cur[d,(*state).ntr]
          tmp=(*state).file[d,(*state).ntr,i]
          widget_control,(*state).wid[d].ttraf,set_value=tmp
          widget_control,(*state).wid[d].ttraf,set_text_select=[strlen(tmp),0L]
          widget_control,(*state).wid[d].ltrai,set_value=strtrim(i,2L)
          widget_control,(*state).wid[d].btrvi,sensitive=i ne -1L

          ;; New trace-rel/flat/dispersion mask/object widgets:
          ;;   trace shift
          if widget_info((*state).wid[d].btras,/valid_id) then $
             widget_control,(*state).wid[d].btras,sensitive=1L
                            
;         widget_control,(*state).wix.btfoc,sensitive=1L ;; telescope focus

          tmp=(*state).cur[d,(*state).nbi] ne -1L
          widget_control,(*state).wid[d].bflcr,sensitive=tmp
          widget_control,(*state).wix.bflat   ,sensitive=tmp

          tmp=~(*state).displibset ;; que?
          widget_control,(*state).wid[d].bdicr,sensitive=tmp
          widget_control,(*state).wix.bdisp   ,sensitive=tmp

          tmp=(*state).cur[d,(*state).nbi] ne -1L
          widget_control,(*state).wid[d].bobcr,sensitive=tmp
          widget_control,(*state).wix.bextr   ,sensitive=tmp

          ;; Saving the data:
          p3d_gui_savedata,(*state).ntr,i,hdr,state

          ;; The select button is made active if there are >=2 loaded objects:
          num=long(total(ptr_valid((*state).data[d,(*state).ntr,*])))
          widget_control,(*state).wid[d].btrse,sensitive=num ge 2L

        endif ;; ~(*state).nf_cancel
      endif ;; filename ne ''

    end ;; case: 'browsetracemask'


    ;;========================================------------------------------
    ;; Trace mask shiftfile:

    'trcshift': begin
      p3d_gui_set_shiftfile,'trc',state,event.top,error=error
      if error ne 0 then return
    end ;; case: 'trcshift'


    ;;========================================------------------------------
    ;; Select an already loaded file:

    'selecttracemask': begin

      p3d_gui_select_file,(*state).ntr,state,error=error
      if error ne 0 then return

      d=(*state).d & ntr=(*state).ntr & i=(*state).cur[d,ntr]
      if i ne -1L then begin
        widget_control,(*state).wid[d].ttraf,set_value=(*state).file[d,ntr,i]
        widget_control,(*state).wid[d].ltrai,set_value=strtrim(i,2L)
      endif
      widget_control,(*state).wid[d].btrvi,sensitive=i ne -1L

    end ;; case: 'selecttracemask'

    ;;========================================------------------------------
    ;; View the trace mask data, on top of the raw data and as an overlay:

    'viewtrace': begin

      num=(*state).cur[(*state).d,(*state).ntr]
      tracefile=reform((*state).path[(*state).d,(*state).ntr,num] + $
                       (*state).file[(*state).d,(*state).ntr,num])

      title='p3d: [trace mask] "'+tracefile+'"'
      p3d_imv,tracefile,title=title,bottom=(*state).cbottom, $
          xscrsize=xscrsize,yscrsize=yscrsize,group_leader=event.top, $
          verbose=(*state).verbose,error=error,debug=(*state).debug
      if error ne 0 then return

    end ;; case: 'viewtrace'


    ;;========================================------------------------------
    ;;========================================------------------------------
    ;; Dispersion mask:

    'createdispmask': begin
      p3d_gui_suffixes,state,error=error
      if error ne 0 then return
      xfx=(*state).xxsfx+(*state).sfx
      yfx=(*state).icsfx+(*state).sfx
      filter=['*'+xfx+';*'+xfx+'.gz;*'+xfx+'.Z','*'+xfx+['','.gz','.Z'], $
              '*'+yfx+';*'+yfx+'.gz;*'+yfx+'.Z','*'+yfx+['','.gz','.Z']]
      filename=dialog_pickfile(path=(*state).inputpath,get_path=tmppath, $
          /must_exist,/multiple_files,dialog_parent=event.top, $
          filter=filter,title='p3d: please select one, two, or more arc' + $
          ' lamp image files.')
      if filename[0L] ne '' then begin
        (*state).inputpath=tmppath

        opath=(*state).sepwpath?(*state).outputpath:(*state).inputpath

        ;; A master bias is used if one is loaded:
        num=(*state).cur[(*state).d,(*state).nbi]
        if num ne -1L then $
           masterbias=reform((*state).path[(*state).d,(*state).nbi,num] + $
                             (*state).file[(*state).d,(*state).nbi,num])

        num=(*state).cur[(*state).d,(*state).ntr]
        tracemask=reform((*state).path[(*state).d,(*state).ntr,num] + $
                         (*state).file[(*state).d,(*state).ntr,num])

        p3d_cdmask,filename,(*state).parfile,saved,dispmask, $
            icstr=(*state).icsfx,ostr=(*state).dmsfx,oostr=(*state).obsfx, $
            masterbias=masterbias,tracemask=tracemask, $
            userparfile=(*state).userparfile, $
            ofilename=ofilename,opath=opath,detector=(*state).d, $
            sfx=(*state).sfx,compress=(*state).compress,dbin=(*state).dbin, $
            track=(*state).track,stawid=(*state).wix.lstat,topwid=event.top, $
            logunit=(*state).logunit, $
            verbose=(*state).verbose,error=error,debug=(*state).debug, $
            colortable=(*state).colortable,cindex=(*state).cindex, $
            cindv=(*state).cindv,bottom=(*state).cbottom
        if error ne 0 then return
          
        if saved then begin

          ;;==============================---------------
          ;; Writing information to the status line:

          tmpstr='[wavecal] Wrote the dispersion mask file "'+ $
            file_basename(ofilename)+'".'
          widget_control,(*state).wix.lstat,set_value=tmpstr
          widget_control,/hourglass

          ;;==============================---------------
          ;; Saving the data in the GUI's state structure:

          p3d_gui_position_new_file,(*state).ndi, $
              file_basename(ofilename),dispmask,state, $
              error=error,debug=(*state).debug
          if error ne 0 then return

          if ~(*state).nf_cancel then begin
            d=(*state).d
            i=(*state).cur[d,(*state).ndi]
            tmp=(*state).file[d,(*state).ndi,i]
            widget_control,(*state).wid[d].tdisf,set_value=tmp
            widget_control,(*state).wid[d].tdisf, $
                           set_text_select=[strlen(tmp),0L]
            widget_control,(*state).wid[d].ldiai,set_value=strtrim(i,2L)
            widget_control,(*state).wid[d].bdivi[0L],sensitive=i ne -1L
            widget_control,(*state).wid[d].bdivi[1L],sensitive=i ne -1L
 
;            ;; New object widgets:
;            tmp=(*state).cur[d,(*state).ntr] ne -1L and $
;                (*state).cur[d,(*state).nbi] ne -1L
;            widget_control,(*state).wid[d].bobcr,sensitive=tmp
;            widget_control,(*state).wix.bextr   ,sensitive=tmp

            if widget_info((*state).wid[d].bdiss,/valid_id) then begin
              tmp=~(*state).displibset
              widget_control,(*state).wid[d].bdiss,sensitive=tmp
            endif

            ;; Saving the data:
            hdr=headfits(ofilename)
            p3d_gui_savedata,(*state).ndi,i,hdr,state

            ;; The select button is made active if >=2 objects are loaded:
            num=long(total(ptr_valid((*state).data[d,(*state).ndi,*])))
            widget_control,(*state).wid[d].bdise,sensitive=num ge 2L

            (*state).pisreadydisp=1L
          endif ;; ~(*state).nf_cancel
        endif else begin ;; saved

          ;;==============================---------------
          ;; Writing information to the status line:

          tmpstr='[wavecal] No new dispersion mask was created.'
          widget_control,(*state).wix.lstat,set_value=tmpstr
          widget_control,/hourglass

        endelse ;; saved

	if ptr_valid(saved) then ptr_free,saved

      endif ;; filename ne ''
    end ;; case: 'createdispmask'


    ;;========================================------------------------------
    ;; Browsing for an already reduced file:

    'browsedispmask': begin
      path=(*state).sepwpath?(*state).outputpath:(*state).inputpath
      sfx=(*state).sfx & str=(*state).dmsfx
      filter=['*'+str+'*'+sfx+';*'+str+'*'+sfx+'.gz;*'+str+'*'+sfx+'.Z', $
              '*'+str+'*'+sfx+['','.gz','.Z'], $
              '*'+sfx+';*'+sfx+'.gz;*'+sfx+'.Z','*'+sfx+['','.gz','.Z']]
      filename=dialog_pickfile(path=path,get_path=tmppath,/must_exist, $
          filter=filter,dialog_parent=event.top, $
          title='p3d: please select a reduced dispersion mask file')
      if filename ne '' then begin
        (*state).inputpath=tmppath

        ;; Checking if the DETECTOR of the file is the same as the used one:
        p3d_misc_checkfile,filename,(*state).nd,(*state).d, $
            (*state).lkwrdlist,topwid=event.top,logunit=(*state).logunit, $
            verbose=(*state).verbose,error=error,debug=(*state).debug
        if error ne 0 then return

        disp=readfits(filename,hdr,silent=(*state).verbose lt 3,/no_unsigned)
        dispfile=file_basename(filename)

        p3d_gui_position_new_file,(*state).ndi,dispfile,disp, $
            state,/useinputpath,error=error,debug=(*state).debug
        if error ne 0 then return

        if ~(*state).nf_cancel then begin
          d=(*state).d
          i=(*state).cur[d,(*state).ndi]
          tmp=(*state).file[d,(*state).ndi,i]
          widget_control,(*state).wid[d].tdisf,set_value=tmp
          widget_control,(*state).wid[d].tdisf, $
                         set_text_select=[strlen(tmp),0L]
          widget_control,(*state).wid[d].ldiai,set_value=strtrim(i,2L)
          widget_control,(*state).wid[d].bdivi[0L],sensitive=i ne -1L
          widget_control,(*state).wid[d].bdivi[1L],sensitive=i ne -1L

;          ;; New object widgets:
;          tmp=(*state).cur[d,(*state).ntr] ne -1L and $
;              (*state).cur[d,(*state).nbi] ne -1L
;          widget_control,(*state).wid[d].bobcr,sensitive=tmp
;          widget_control,(*state).wix.bextr   ,sensitive=tmp

          if widget_info((*state).wid[d].bdiss,/valid_id) then begin
            tmp=~(*state).displibset
            widget_control,(*state).wid[d].bdiss,sensitive=tmp
          endif

          ;; Saving the data:
          p3d_gui_savedata,(*state).ndi,i,hdr,state

          ;; The select button is made active if >=2 objects are loaded:
          num=long(total(ptr_valid((*state).data[d,(*state).ndi,*])))
          widget_control,(*state).wid[d].bdise,sensitive=num ge 2L

          (*state).pisreadydisp=1L
        endif ;; ~(*state).nf_cancel
      endif ;; filename ne ''

    end ;; case: 'browsedispmask'

    ;;========================================------------------------------
    ;; Dispersion mask shift file:

    'dispshift': begin
      p3d_gui_set_shiftfile,'createdispmask',state,event.top,error=error
      if error ne 0 then return
    end ;; case: 'dispshift'

    ;;========================================------------------------------
    ;; Select an already loaded file:

    'selectdispmask': begin

      p3d_gui_select_file,(*state).ndi,state,error=error
      if error ne 0 then return

      d=(*state).d & ndi=(*state).ndi & i=(*state).cur[d,ndi]
      if i ne -1L then begin
        widget_control,(*state).wid[d].tdisf,set_value=(*state).file[d,ndi,i]
        widget_control,(*state).wid[d].ldiai,set_value=strtrim(i,2L)
      endif
      widget_control,(*state).wid[d].bdivi[0L],sensitive=i ne -1L
      widget_control,(*state).wid[d].bdivi[1L],sensitive=i ne -1L

    end ;; case: 'selectdispmask'

    ;;========================================------------------------------
    ;; View the dispersion mask raw data, and with the overlayed trace mask:

    'viewdispmask': begin

      d=(*state).d
      num=(*state).cur[d,(*state).ndi]
      dispfile=reform((*state).path[d,(*state).ndi,num] + $
                      (*state).file[d,(*state).ndi,num])
      p3d_gui_suffixes,state,error=error

      ;; Checking if the information about the extracted data has been stored
      ;; in the file:
      fitpar=readfits(dispfile,dhdr,silent=(*state).verbose lt 3,/no_unsigned)
      dispxfile=fxpar(dhdr,'IMEXTR')
      i=1L & more=1L
      while more do begin
        tmp=fxpar(dhdr,'IMEXTR'+strtrim(i++,2L))
        if tmp ne '0' then dispxfile+=tmp else more=0L
      endwhile
      dispxfile=strtrim(dispxfile,2L)

      ;; Also attempting the DISPFILE directory, and its parent:
      if ~file_test(dispxfile,/read,/regular) then $
         dispxfile=reform((*state).path[d,(*state).ndi,num])+ $
                   file_basename(dispxfile)
      if ~file_test(dispxfile,/read,/regular) then $
         dispxfile=file_dirname(reform((*state).path[d,(*state).ndi,num]), $
                                /mark_directory)+file_basename(dispxfile)

      isplitbin=fxpar(dhdr,'IMSPBIN')  & isplitbin=long(isplitbin)
      imageorder=fxpar(dhdr,'IMORDER') & imageorder=long(imageorder)

      if ~file_test(strtrim(dispxfile,2L),/regular,/read) then begin
        sfx=(*state).sfx & str=(*state).obsfx
        filter=['*'+str+'*'+sfx+';*'+str+'*'+sfx+'.gz;*'+str+'*'+sfx+'.Z', $
                '*'+str+'*'+sfx+['','.gz','.Z'],'*'+sfx+['','.gz','.Z']]
        dispxfile=dialog_pickfile(path=(*state).inputpath,/must_exist, $
            filter=filter,dialog_parent=event.top, $
            title='p3d: please select the disp.mask extracted file') 
      endif ;; ~file_test(strtrim(dispxfile,2L),/regular,/read)

      if dispxfile ne '' and file_test(dispxfile,/regular,/read) then begin
        title='[dispersion mask, extracted] "'+dispxfile+'"'

        if ~index then begin
          p3d_imv,dispxfile,title=title,bottom=(*state).cbottom, $
              isplitbin=isplitbin,imageorder=imageorder,daxis=daxis, $
              xscrsize=xscrsize,yscrsize=yscrsize,group_leader=event.top, $
              verbose=(*state).verbose,error=error,debug=(*state).debug
          if error ne 0 then return
        endif else begin
          data=readfits(dispxfile,hdr,silent=(*state).verbose lt 3, $
                        /no_unsigned)

          ;; Attempting to read the errors file:
          dff=0L
          dfile=fxpar(hdr,'IMERR')
          if dfile ne '0' then begin
            i=1L & more=1L
            while more do begin
              tmp=fxpar(hdr,'IMERR'+strtrim(i++,2L))
              if tmp ne '0' then dfile+=strtrim(tmp,2L) else more=0L
            endwhile

            dff=file_test(dfile,/read,/regular)
          endif ;; dfile ne '0'

          if ~dff then begin
            ;; Making another attempt, with a slightly different filename:
            sfx=(*state).sfx & str=(*state).ffsfx
            tmp=strpos(dispxfile,sfx,/reverse_search)
            fffx=strmid(dispxfile,tmp)
            dfile=strmid(dispxfile,0L,tmp)+'_err'+fffx
            if file_test(dfile,/read,/regular) then dff=1L
          endif
          if dff then $
             ddata=readfits(dfile,silent=(*state).verbose lt 3,/no_unsigned)

          ;; Writing information to the status line:
          msg='Applying the dispersion correction to the extracted data file.'
          widget_control,(*state).wix.lstat,set_value=msg

          ;; Creating a two-dimensional wavelength array using the polynomial
          ;; fitting parameters which were returned by p3d_wavecal_fit_maskwd:
          sid=long(fxpar(dhdr,'NAXIS1'))
          pdeg=long(fxpar(dhdr,'NAXIS2'))
          nwav=long(fxpar(hdr,'NAXIS'+strtrim((*state).daxis,2L)))
          x=dindgen(nwav)
          template=double(data)*0d0
          if (*state).daxis then begin
            for k=0L,sid-1L do for L=0L,pdeg-1L do $
               template[*,k]+=fitpar[k,L]*x^L
          endif else begin
            for k=0L,sid-1L do for L=0L,pdeg-1L do $
               template[k,*]+=transpose(fitpar[k,L]*x^L)
          endelse ;; (*state).daxis

          ;; Calculating the dispersion correction of the spectrum image,
          ;; using the wavelength template:
          p3d_wavecal_dispersion_correction,data,template,crval,cdelt,odata, $
              oddata,dstack=ddata,daxis=(*state).daxis,topwid=(*state).top, $
              logunit=(*state).logunit,verbose=(*state).verbose,error=error, $
              debug=(*state).debug
          if error ne 0 then return

          ;; Adding the proper information to the data header:
          fxaddpar,hdr,'CRVAL'+strtrim((*state).daxis,2L),crval
          fxaddpar,hdr,'CDELT'+strtrim((*state).daxis,2L),cdelt

          ;; Writing information to the status line:
          msg='Loading the spectrum viewer.'
          widget_control,(*state).wix.lstat,set_value=msg

          ;; Starting the spectrum viewer application:
          idx=d*(*state).nblocks;+(*state).block ;; 10.10.2009 /CS
;        colorcut=[(*state).ccmin[d],(*state).ccmax[d]]
          htitle='Fits Header: "'+dispxfile+'"'

          tvlct,(*state).colors.red,(*state).colors.green,(*state).colors.blue
          p3d_rss,odata,(*(*state).pos[idx]).file,dimage=oddata,hdr=hdr, $
              orientation=(*state).orientation[d,num], $
              colortable=(*state).colortable, $
              bottom=(*state).cbottom,cindex=(*state).cindex, $
              cindv=(*state).cindv,title=title,fitsheadertitle=htitle, $
              daxis=(*state).daxis,group_leader=event.top, $
              verbose=(*state).verbose,error=error,debug=(*state).debug
       
        endelse ;; ~index
      endif ;; dispxfile ne '' and file_test(dispxfile,/regular,/read)

    end ;; case: 'viewdispmask'


    ;;========================================------------------------------
    ;;========================================------------------------------
    ;; Fiber flatfield:

    'createflatfield': begin
      ;; Launching a widget with flat field extraction options:
      p3d_gui_suffixes,state,error=error
      if error ne 0 then return
      p3d_gui_flatfield_menu,state,ret,tmppath, $
          filename=filename,tracemask=tracemask,dispmask=dispmask

      if ret then begin
        (*state).inputpath=tmppath
        opath=(*state).sepwpath?(*state).outputpath:(*state).inputpath

        num=(*state).cur[(*state).d,(*state).nbi]
        if num ne -1L then $
          masterbias=reform((*state).path[(*state).d,(*state).nbi,num] + $
                            (*state).file[(*state).d,(*state).nbi,num])

        ;;==============================--------------------
        ;; Calling the flatfield extraction program:

        p3d_cflatf,filename,(*state).parfile,finalflat, $
            icstr=(*state).icsfx,ostr=(*state).ffsfx,tostr=(*state).trsfx, $
            masterbias=masterbias,tracemask=tracemask,dispmask=dispmask, $
            userparfile=(*state).userparfile,ofilename=ofilename, $
            opath=opath,detector=(*state).d,sfx=(*state).sfx, $
            compress=(*state).compress,stawid=(*state).wix.lstat, $
            topwid=event.top,logunit=(*state).logunit, $
            verbose=(*state).verbose,error=error,debug=(*state).debug
	if error ne 0 then return

        ;;==============================--------------------
        ;; Displaying a wavelength collapsed image of the flat-field data:

        flatim=total(finalflat,(*state).daxis)

        p3d_gui_position_new_file,(*state).nfl, $
            file_basename(ofilename),finalflat,state, $
            error=error,debug=(*state).debug
        if error ne 0 then return

        i=(*state).cur[(*state).d,(*state).nfl]
        p3d_gui_draw,i,state,event.top,flatim,/left,/auto

        if ~(*state).nf_cancel then begin
          d=(*state).d
          tmp=(*state).file[d,(*state).nfl,i]
          widget_control,(*state).wid[d].tflaf,set_value=tmp
          widget_control,(*state).wid[d].tflaf, $
                         set_text_select=[strlen(tmp),0L]
          widget_control,(*state).wid[d].lflai,set_value=strtrim(i,2L)
          widget_control,(*state).wid[d].bflvi[0L],sensitive=i ne -1L
          widget_control,(*state).wid[d].bflvi[1L],sensitive=i ne -1L

          ;; Saving the data:
          p3d_gui_savedata,(*state).nfl,i,hdr,state

          ;; The select button is made active if >=2 objects are loaded:
          num=long(total(ptr_valid((*state).data[d,(*state).nfl,*])))
          widget_control,(*state).wid[d].bflse,sensitive=num ge 2L

          (*state).pisreadyflat=1L
        endif ;; ~(*state).nf_cancel

      endif ;; filename ne ''
    end ;; case: 'createflatfield'


    ;;========================================------------------------------
    ;; Browsing for an already reduced flat field file:

    'browseflatfield': begin
      path=(*state).sepwpath?(*state).outputpath:(*state).inputpath
      sfx=(*state).sfx & str=(*state).ffsfx
      filter=['*'+str+'*'+sfx+';*'+str+'*'+sfx+'.gz;*'+str+'*'+sfx+'.Z', $
              '*'+str+'*'+sfx+['','.gz','.Z'], $
              '*'+sfx+';*'+sfx+'.gz;*'+sfx+'.Z','*'+sfx+['','.gz','.Z']]
      filename=dialog_pickfile(path=path,get_path=tmppath,/must_exist, $
          filter=filter,dialog_parent=event.top, $
          title='p3d: please select a reduced flatfield image file')
      if filename ne '' then begin
        (*state).inputpath=tmppath

        ;; Checking if the DETECTOR of the file is the same as the used one:
        p3d_misc_checkfile,filename,(*state).nd,(*state).d, $
            (*state).lkwrdlist,topwid=event.top,logunit=(*state).logunit, $
            verbose=(*state).verbose,error=error,debug=(*state).debug
        if error ne 0 then return

        flat=readfits(filename,hdr,silent=(*state).verbose lt 3,/no_unsigned)
        flatfile=file_basename(filename)

        p3d_gui_position_new_file,(*state).nfl,flatfile,flat, $
            state,/useinputpath,error=error,debug=(*state).debug
        if error ne 0 then return

        if ~(*state).nf_cancel then begin
          d=(*state).d
          i=(*state).cur[d,(*state).nfl]
          tmp=(*state).file[d,(*state).nfl,i]
          widget_control,(*state).wid[d].tflaf,set_value=tmp
          widget_control,(*state).wid[d].tflaf,set_text_select=[strlen(tmp),0L]
          widget_control,(*state).wid[d].lflai,set_value=strtrim(i,2L)
          widget_control,(*state).wid[d].bflvi[0L],sensitive=i ne -1L
          widget_control,(*state).wid[d].bflvi[1L],sensitive=i ne -1L

          ;; Saving the data:
          p3d_gui_savedata,(*state).nfl,i,hdr,state

          ;; The select button is made active if >=2 objects are loaded:
          num=long(total(ptr_valid((*state).data[d,(*state).nfl,*])))
          widget_control,(*state).wid[d].bflse,sensitive=num ge 2L

          (*state).pisreadyflat=1L
        endif ;; ~(*state).nf_cancel
      endif ;; filename ne ''

    end ;; case: 'browseflatfield'

    ;;========================================------------------------------
    ;; Select an already loaded flat field file:

    'selectflatfield': begin

      p3d_gui_select_file,(*state).nfl,state,error=error
      if error ne 0 then return

      d=(*state).d & nfl=(*state).nfl & i=(*state).cur[d,nfl]
      if i ne -1L then begin
        widget_control,(*state).wid[d].tflaf,set_value=(*state).file[d,nfl,i]
        widget_control,(*state).wid[d].lflai,set_value=strtrim(i,2L)
      endif
      widget_control,(*state).wid[d].bflvi[0L],sensitive=i ne -1L
      widget_control,(*state).wid[d].bflvi[1L],sensitive=i ne -1L

    end ;; case: 'selectflatfield'

    ;;========================================------------------------------
    ;; View the flatfield raw data, and with the overlayed trace mask:

    'viewflatfield': begin

      d=(*state).d
      num=(*state).cur[d,(*state).nfl]
      flatfile=reform((*state).path[d,(*state).nfl,num] + $
                      (*state).file[d,(*state).nfl,num])

      title='[flat field] "'+flatfile+'"'

      if ~index then begin
        p3d_imv,flatfile,title=title,bottom=(*state).cbottom, $
            xscrsize=xscrsize,yscrsize=yscrsize,group_leader=event.top, $
            verbose=(*state).verbose,error=error,debug=(*state).debug
        if error ne 0 then return
      endif else begin
        data=readfits(flatfile,hdr,silent=(*state).verbose lt 3,/no_unsigned)

        ;; Attempting to read the errors file:
        dff=0L
        dfile=fxpar(hdr,'IMERR')
        if dfile ne '0' then begin
          i=1L & more=1L
          while more do begin
            tmp=fxpar(hdr,'IMERR'+strtrim(i++,2L))
            if tmp ne '0' then dfile+=strtrim(tmp,2L) else more=0L
          endwhile

          dff=file_test(dfile,/read,/regular)
        endif ;; dfile ne '0'

        if ~dff then begin
          ;; Making another attempt, with a slightly different filename:
          sfx=(*state).sfx & str=(*state).ffsfx
          tmp=strpos(flatfile,sfx,/reverse_search)
          fffx=strmid(flatfile,tmp)
          dfile=strmid(flatfile,0L,tmp)+'_err'+fffx
          if file_test(dfile,/read,/regular) then dff=1L
        endif
        if dff then $
           ddata=readfits(dfile,silent=(*state).verbose lt 3,/no_unsigned)

        numd=(*state).cur[d,(*state).ndi]

        if numd ne -1L then begin
          ;; Writing information to the status line:
          msg='Applying the dispersion correction to the extracted data file.'
          widget_control,(*state).wix.lstat,set_value=msg

          ;; The following information should be read from the file...
          dispfile=reform((*state).path[d,(*state).ndi,numd] + $
                          (*state).file[d,(*state).ndi,numd])

          ;; Checking if the information about the extracted data has
          ;; been stored in the file:
          fitpar=readfits(dispfile,dhdr,silent=(*state).verbose lt 3, $
                          /no_unsigned)

          ;; Creating a two-dimensional wavelength array using the polynomial 
          ;; fitting parameters which were returned by p3d_wavecal_fit_maskwd:
          sid=long(fxpar(dhdr,'NAXIS1'))
          pdeg=long(fxpar(dhdr,'NAXIS2'))
          nwav=long(fxpar(hdr,'NAXIS'+strtrim((*state).daxis,2L)))
          x=dindgen(nwav)
          template=double(data)*0d0
          if (*state).daxis then begin
            for k=0L,sid-1L do for L=0L,pdeg-1L do $
              template[*,k]+=fitpar[k,L]*x^L
          endif else begin
            for k=0L,sid-1L do for L=0L,pdeg-1L do $
               template[k,*]+=transpose(fitpar[k,L]*x^L)
          endelse ;; (*state).daxis

          ;; Calculating the dispersion correction of the spectrum image,
          ;; using the wavelength template:
          p3d_wavecal_dispersion_correction,data,template,crval,cdelt, $
              odata,oddata,dstack=ddata,daxis=(*state).daxis, $
              topwid=(*state).top,logunit=(*state).logunit, $
              verbose=(*state).verbose,error=error,debug=(*state).debug
          if error ne 0 then return

          ;; Adding the proper information to the data header:
          fxaddpar,hdr,'CRVAL'+strtrim((*state).daxis,2L),crval
          fxaddpar,hdr,'CDELT'+strtrim((*state).daxis,2L),cdelt
        endif else begin
           odata=data
        endelse ;; numd ne -1L

        ;; Writing information to the status line:
        msg='Loading the spectrum viewer.'
        widget_control,(*state).wix.lstat,set_value=msg

        ;; Starting the spectrum viewer application:
        idx=d;*(*state).nblocks;+(*state).block ;; 10.10.2009 /CS
;        colorcut=[(*state).ccmin[d],(*state).ccmax[d]]
        htitle='Fits Header: "'+flatfile+'"'

        tvlct,(*state).colors.red,(*state).colors.green,(*state).colors.blue
        p3d_rss,odata,(*(*state).pos[idx]).file,dimage=oddata,hdr=hdr, $
            orientation=(*state).orientation[d,num], $
            colortable=(*state).colortable, $
            bottom=(*state).cbottom,cindex=(*state).cindex, $
            cindv=(*state).cindv,title=title,fitsheadertitle=htitle, $
            daxis=(*state).daxis,group_leader=event.top, $
            verbose=(*state).verbose,error=error,debug=(*state).debug
      endelse ;; ~index

    end ;; case: 'viewflatfield'


    ;;========================================------------------------------
    ;;========================================------------------------------
    ;; Object data:

    'createobject': begin
      ;; Launching a widget showing several options:
      method=p3d_misc_extract_method((*state).userparfile,topwid=event.top, $
          logunit=(*state).logunit,verbose=(*state).verbose, $
          error=error,debug=(*state).debug)
      if error ne 0 then return
      p3d_gui_suffixes,state,error=error
      if error ne 0 then return
      p3d_gui_extract_menu,state,event.top,ex,method,exfilename,crfilename

      if ex.object then begin
        (*state).inputpath=ex.path
        filename=ex.path+exfilename

        opath=(*state).sepwpath?(*state).outputpath:(*state).inputpath

        d=(*state).d
        if ex.bias then begin
          num=(*state).cur[d,(*state).nbi]
          if num ne -1L then $
            masterbias=reform((*state).path[d,(*state).nbi,num] + $
                              (*state).file[d,(*state).nbi,num])
        endif ;; ex.bias

        num=(*state).cur[d,(*state).ntr]
        tracemask=reform((*state).path[d,(*state).ntr,num] + $
                         (*state).file[d,(*state).ntr,num])

        if ex.wave then begin
          num=(*state).cur[d,(*state).ndi]
          if num ne -1L then $
             dispmask=reform((*state).path[d,(*state).ndi,num] + $
                             (*state).file[d,(*state).ndi,num])
        endif ;; ex.wave

        if ex.flat then begin
          num=(*state).cur[d,(*state).nfl]
          if num ne -1L then $
             flatfield=reform((*state).path[d,(*state).nfl,num] + $
                              (*state).file[d,(*state).nfl,num])
        endif ;; ex.flat

        ;; Shall a trace shift file be used?:
        if (*state).dotraceshif then traceshiftfile=reform( $
           (*state).path[d,(*state).ntsh,(*state).cur[d,(*state).ntr]] + $
           (*state).file[d,(*state).ntsh,(*state).cur[d,(*state).ntr]])

        ;; Shall a dispersion mask shift file be used?:
        if (*state).dodispshift then dispshiftfile=reform( $
           (*state).path[d,(*state).ndsh,(*state).cur[d,(*state).ndi]] + $
           (*state).file[d,(*state).ndsh,(*state).cur[d,(*state).ndi]])

        if (*state).dodispshift or (*state).dotraceshif then objectshiftfile= $
           ex.shiftrefpath+ex.shiftreffile

        ;;==============================--------------------
        ;; Optionally cleaning the input image of cosmic ray hits, or creating
        ;; a cosmic ray mask:

        ; This is work in progress...

        ;;==============================--------------------
        ;; Calling the object extraction program:

        p3d_cobjex,filename,(*state).parfile,extract,bsw_extract, $
            icstr=(*state).icsfx,ostr=(*state).obsfx,masterbias=masterbias, $
            tracemask=tracemask,dispmask=dispmask,flatfield=flatfield, $
            crmask=crfilename,waveprompt=ex.waveprompt, $
            skyalign=(*state).skyalign,savee3d=ex.savee3d,sbsw=bsw, $
            shiftreffile=ex.shiftreffile,trcshiftfile=traceshiftfile, $
            dispshiftfile=dispshiftfile,objshiftfile=objectshiftfile, $
            userparfile=(*state).userparfile, $
            ofilename=ofilename,bsw_ofilename=bsw_ofilename, $
            opath=opath,detector=(*state).d,sfx=(*state).sfx, $
            compress=(*state).compress,stawid=(*state).wix.lstat, $
            topwid=event.top,logunit=(*state).logunit, $
            verbose=(*state).verbose,error=error,debug=(*state).debug
        if error ne 0 then return

        ;;==============================--------------------
        ;;==============================--------------------
        ;; Final setup; treating object data and optional beam switch data
        ;; simultaneously:

        for m=0L,bsw do begin
          if m eq 1L then begin
            extractim=bsw_extract
            ofile=bsw_ofilename
          endif else begin
            extractim=extract
            ofile=ofilename
          endelse

          wset,(*state).wid[d].dleft
          s=size(extractim)
          tcol=s[(*state).daxis]/2L ;; choosing the center wavelength bin

          idx=d*(*state).nblocks;+(*state).block ;; 10.10.2009 /CS
          image=reform((*state).daxis?extractim[tcol,*]:extractim[*,tcol])
          (*state).ccmin[d]=min(image[(*(*state).pos[idx]).fpos])
          (*state).ccmax[d]=max(image[(*(*state).pos[idx]).fpos])

          ;; Updating various widgets:
          widget_control,event.top,update=0L

          widget_control,(*state).wid[d].tobjf,set_value=file_basename(ofile)
          widget_control,(*state).wid[d].lobji,set_value=''
          widget_control,(*state).wid[d].tcmin,set_value=''
          widget_control,(*state).wid[d].tcmax,set_value=''
          widget_control,(*state).wid[d].twave,set_value=''
          widget_control,(*state).wid[d].lwave,set_value=''

          p3d_gui_position_new_file,(*state).nob,file_basename(ofile), $
              extractim,state,error=error,debug=(*state).debug
          if error ne 0 then return

          i=(*state).cur[d,(*state).nob]
          p3d_gui_draw,i,state,event.top,image,/left,/auto

          hdr=headfits(ofile)
          (*state).crvd[d,0L,i]=fxpar(hdr,'CRVAL'+strtrim((*state).daxis,2L))
          (*state).crvd[d,1L,i]=fxpar(hdr,'CDELT'+strtrim((*state).daxis,2L))

          if ~(*state).nf_cancel then begin
            tmp=(*state).file[d,(*state).nob,i]
            widget_control,(*state).wid[d].tobjf,set_value=tmp
            widget_control,(*state).wid[d].tobjf, $
                          set_text_select=[strlen(tmp),0L]
            widget_control,(*state).wid[d].lobji,set_value=strtrim(i,2L)
            widget_control,(*state).wid[d].bobvi[0L],sensitive=i ne -1L
            widget_control,(*state).wid[d].bobvi[1L],sensitive=i ne -1L

            ;; Saving the data:
            p3d_gui_savedata,(*state).nob,i,hdr,state

            ;; The select button is made active if >=2 objects are loaded:
            num=long(total(ptr_valid((*state).data[d,(*state).nob,*])))
            ;; commenting out the following as it doesn't work, 1.2.2010, /CS
;           widget_control,(*state).wid[d].bobse,sensitive=num ge 2L

            ;; Setting the variables pointing at the current column:
            s=size(*(*state).curdata[d,(*state).nob])
            (*state).curb[d]=s[(*state).daxis]/2L
            (*state).column[d,i]=(*state).curb[d]

            if (*state).scalebinsseparately then begin
              tmp=(*state).curb[d]
              image=reform((*state).daxis? $
                           (*(*state).curdata[d,(*state).nob])[tmp,*]: $
                           (*(*state).curdata[d,(*state).nob])[*,tmp])
            endif else begin
              image=reform(*(*state).curdata[d,(*state).nob])
            endelse

            if (*state).slid[d,i] ge 0L then begin
              val=(*state).slidp[d,i]

              ;; Calculating the clipping range:
              p3d_misc_dataclip,image,range,percentage=val, $
                  topwid=event.top,logunit=(*state).logunit, $
                  verbose=(*state).verbose,error=error,debug=(*state).debug
              if error ne 0 then return

              minval=range[0L]
              maxval=range[1L]

              ;; Setting the range widget values:
              widget_control,(*state).wid[d].tcmin, $
                             set_value=strtrim(minval,2L)
              widget_control,(*state).wid[d].tcmax, $
                             set_value=strtrim(maxval,2L)
            endif else begin
              minval=(*state).ccmin[d]
              maxval=(*state).ccmax[d]
            endelse

            (*state).ccmin[d]=minval & (*state).cmin[d,i]=(*state).ccmin[d]
            (*state).ccmax[d]=maxval & (*state).cmax[d,i]=(*state).ccmax[d]

            ;; Updating the left and right object images on the main panel:
            p3d_gui_draw,i,state,event.top,/left,/right

            ;; Updating the uname of the two draw widgets:
            tmp='[object slot '+strtrim(i,2L)+'] '+file_basename(ofile)
            tmp2=' :: right click to change the orientation.'
            widget_control,(*state).wid[d].dobjl,set_uname=tmp+tmp2
            widget_control,(*state).wid[d].dobji[i],set_uname=tmp

            ;; Sensitizing various widgets:
            widget_control,(*state).wid[d].bviei,sensitive=1L
            widget_control,(*state).wid[d].bsett,sensitive=1L

            ;; Updating the wavelength bin slider:
            widget_control,(*state).wid[d].sobjl, $
                           set_value=(*state).curb[(*state).d]
            widget_control,(*state).wid[d].sobjl,set_slider_max= $
                           s[(*state).daxis]-1L
            widget_control,(*state).wid[d].sobjl,sensitive=1L

            ;; Updating the widgets showing the minimum and maximum values:
            tmpstr=strtrim(minval,2L)
            widget_control,(*state).wid[d].tcmin,set_value=tmpstr
            tmpstr=strtrim(maxval,2L)
            widget_control,(*state).wid[d].tcmax,set_value=tmpstr
            tmpstr=strtrim((*state).curb[d],2L)
            widget_control,(*state).wid[d].twave,set_value=tmpstr

            ;; Updating the widget showing the wavelength:
            p3d_gui_setwave,i,state,/updateuname

;              wavelength=(*state).crvd[0L,i]+ $
;                         (*state).crvd[1L,i]*((*state).curb+0.5)
;              tmp=string(wavelength,format='(f8.2,"")')
;              widget_control,(*state).wid[(*state).d].lwave, $
;                set_value=strtrim(tmp,2L)

            widget_control,event.top,update=1L

            wset,(*state).wid[d].dleft
          endif else begin ;; ~(*state).nf_cancel
            wset,(*state).wid[d].dleft

            if i ne -1L then begin

              if (*state).scalebinsseparately then begin
                tmp=(*state).curb[d]
                image=reform((*state).daxis? $
                             (*(*state).curdata[d,(*state).nob])[tmp,*]: $
                             (*(*state).curdata[d,(*state).nob])[*,tmp])
              endif else begin
                image=reform(*(*state).curdata[d,(*state).nob])
              endelse

              if (*state).slid[d,i] ge 0L then begin
                val=(*state).slidp[d,i]

                ;; Calculating the clipping range:
                p3d_misc_dataclip,image,range,percentage=val, $
                    topwid=event.top,logunit=(*state).logunit, $
                    verbose=(*state).verbose,error=error,debug=(*state).debug
                if error ne 0 then return

                minval=range[0L]
                maxval=range[1L]

                ;; Setting the range widget values:
                widget_control,(*state).wid[d].tcmin, $
                               set_value=strtrim(minval,2L)
                widget_control,(*state).wid[d].tcmax, $
                               set_value=strtrim(maxval,2L)
              endif else begin
                minval=(*state).ccmin[d]
                maxval=(*state).ccmax[d]
              endelse

              ;; Updating the left and right object images on the main panel:
              p3d_gui_draw,i,state,event.top,/left

              tmp=exfilename[0L]+' :: wavelength collapsed for bin '+ $
                  strtrim((*state).curb[d],2L)+'.'
              widget_control,(*state).wid[d].dobjl,set_uname=tmp
              widget_control,(*state).wid[d].dobji[i],set_uname=tmp
              widget_control,(*state).wid[d].bobvi[0L],sensitive=i ne -1L
              widget_control,(*state).wid[d].bobvi[1L],sensitive=i ne -1L

              ;; Sensitizing various widgets:
              widget_control,(*state).wid[d].bviei,sensitive=1L
              widget_control,(*state).wid[d].bsett,sensitive=1L

              ;; Updating the wavelength bin slider:
              widget_control,(*state).wid[d].sobjl,set_slider_max= $
                             s[(*state).daxis]-1L
              widget_control,(*state).wid[d].sobjl, $
                             set_value=(*state).curb[d]
              widget_control,(*state).wid[d].sobjl,sensitive=1L

              ;; Updating the widgets showing the minimum and maximum values:
              tmpstr=strtrim(minval,2L)
              widget_control,(*state).wid[d].tcmin,set_value=tmpstr
              tmpstr=strtrim(maxval,2L)
              widget_control,(*state).wid[d].tcmax,set_value=tmpstr
              tmpstr=strtrim((*state).curb[d],2L)
              widget_control,(*state).wid[d].twave,set_value=tmpstr

              ;; Updating the widget showing the wavelength:
              p3d_gui_setwave,i,state,/updateuname

;                wavelength=(*state).crvd[0L,i]+ $
;                           (*state).crvd[1L,i]*((*state).curb+0.5)
;                tmp=string(wavelength,format='(f8.2,"")')
;                widget_control,(*state).wid[(*state).d].lwave, $
;                  set_value=strtrim(tmp,2L)

            endif else begin ;; i ne -1L

              ;; No data - drawing a black box:
              p3d_display_tele,intarr(2L,2L),0,1,(*state).leftsize, $
                  (*state).leftsize,topwid=event.top,error=error, $
                  logunit=(*state).logunit,verbose=(*state).verbose, $
                  debug=(*state).debug
              widget_control,(*state).wid[d].tobjf,set_value=''
              widget_control,(*state).wid[d].lobji,set_value=''
              widget_control,(*state).wid[d].bobvi[0L],sensitive=0L
              widget_control,(*state).wid[d].bobvi[1L],sensitive=0L

              tmp='no file loaded.'
              widget_control,(*state).wid[d].dobjl,set_uname=tmp
              tmp_=(*state).cur[d,(*state).nob]
              widget_control,(*state).wid[d].dobji[tmp_],set_uname=tmp

              ;; Sensitizing various widgets:
              widget_control,(*state).wid[d].bviei,sensitive=0L ; correct?
              widget_control,(*state).wid[d].bsett,sensitive=0L

              ;; Updating the wavelength bin slider:
              widget_control,(*state).wid[d].sobjl,sensitive=0L

            endelse ;; i ne -1L

          endelse ;; ~(*state).nf_cancel
        endfor ;; m=0L,bsw
      endif ;; ex.object
    end ;; 'createobject'
    

    ;;========================================------------------------------
    ;; Browsing for an already reduced file:

    'browseobject': begin
      path=(*state).sepwpath?(*state).outputpath:(*state).inputpath
      sfx=(*state).sfx & str=(*state).obsfx
      filter=['*'+str+'*'+sfx+';*'+str+'*'+sfx+'.gz;*'+str+'*'+sfx+'.Z', $
              '*'+str+'*'+sfx+['','.gz','.Z'], $
              '*'+sfx+';*'+sfx+'.gz;*'+sfx+'.Z','*'+sfx+['','.gz','.Z']]
      filename=dialog_pickfile(path=path,get_path=tmppath,/must_exist, $
          filter=filter,dialog_parent=event.top, $
          title='p3d: please select a reduced object image file')
      if filename ne '' then begin
        (*state).inputpath=tmppath
        d=(*state).d

        ;; Checking if the DETECTOR of the file is the same as the used one:
        p3d_misc_checkfile,filename,(*state).nd,(*state).d, $
            (*state).lkwrdlist,topwid=event.top,logunit=(*state).logunit, $
            verbose=(*state).verbose,error=error,debug=(*state).debug
        if error ne 0 then return

        ;; Reading the data:
        extract=readfits(filename,hdr,silent=(*state).verbose lt 3, $
                         /no_unsigned)

        s=size(extract)
        tcol=s[(*state).daxis]/2L ;; choosing the central wavelength bin

        image=reform((*state).daxis?extract[tcol,*]:extract[*,tcol])
        objectfile=file_basename(filename)

        widget_control,event.top,update=0L
        widget_control,(*state).wid[d].tcmin,set_value=''
        widget_control,(*state).wid[d].tcmax,set_value=''
        widget_control,(*state).wid[d].twave,set_value=''
        widget_control,(*state).wid[d].lwave,set_value=''
        widget_control,(*state).wid[d].tobjf,set_value=objectfile
        widget_control,(*state).wid[d].lobji,set_value=''

        p3d_gui_position_new_file,(*state).nob,objectfile,extract, $
            state,/useinputpath,error=error,debug=(*state).debug
        if error ne 0 then return

        if ~(*state).nf_cancel then begin
          i=(*state).cur[d,(*state).nob]
          tmp=(*state).file[d,(*state).nob,i]
          widget_control,(*state).wid[d].tobjf,set_value=tmp
          widget_control,(*state).wid[d].tobjf,set_text_select=[strlen(tmp),0L]
          widget_control,(*state).wid[d].lobji,set_value=strtrim(i,2L)
          widget_control,(*state).wid[d].bobvi[0L],sensitive=i ne -1L
          widget_control,(*state).wid[d].bobvi[1L],sensitive=i ne -1L

          (*state).crvd[d,0L,i]=fxpar(hdr,'CRVAL'+strtrim((*state).daxis,2L))
          (*state).crvd[d,1L,i]=fxpar(hdr,'CDELT'+strtrim((*state).daxis,2L))

          ;; Saving the data:
          p3d_gui_savedata,(*state).nob,i,hdr,state

          ;; Calculating the clipping range:
          val=(*state).slidp[d,i]
          tmp=(*state).scalebinsseparately?image:extract
          p3d_misc_dataclip,tmp,range,percentage=val, $
              topwid=event.top,logunit=(*state).logunit, $
              verbose=(*state).verbose,error=error,debug=(*state).debug
          if error ne 0 then return

          minval=range[0L]
          maxval=range[1L]
          (*state).ccmin[d]=minval
          (*state).ccmax[d]=maxval

          ;; Updating the left and right object images on the main panel:
          p3d_gui_draw,i,state,event.top,image,/left,/right

          ;; The select button is made active if >=2 objects are loaded:
          num=long(total(ptr_valid((*state).data[d,(*state).nob,*])))
          ;; commenting out the following as it doesn't work, 1.2.2010, /CS
;         widget_control,(*state).wid[d].bobse,sensitive=num ge 2L

          (*state).curb[d]=tcol
          (*state).cmin[d,i]  =(*state).ccmin[d]
          (*state).cmax[d,i]  =(*state).ccmax[d]
          (*state).column[d,i]=(*state).curb[d]

          ;; Setting the range widget values:
          widget_control,(*state).wid[d].tcmin, $
                         set_value=strtrim((*state).ccmin[d],2L)
          widget_control,(*state).wid[d].tcmax, $
                         set_value=strtrim((*state).ccmax[d],2L)
          widget_control,(*state).wid[d].twave, $
                         set_value=strtrim((*state).curb[d],2L)

          ;; Updating the widget showing the wavelength:
          p3d_gui_setwave,i,state,/updateuname

;          wavelength=(*state).crvd[0L,i]+ $
;                     (*state).crvd[1L,i]*((*state).curb+0.5)
;          tmp=string(wavelength,format='(f8.2,"")')
;          widget_control,(*state).wid[d].lwave,set_value=strtrim(tmp,2L)

          wset,(*state).wid[d].dleft

          ;; Sensitizing various widgets:
          widget_control,(*state).wid[d].bviei,sensitive=1L
          widget_control,(*state).wid[d].bsett,sensitive=1L

          ;; Updating the uname for the two draw widgets:
          tmp='[object slot '+ $
              strtrim((*state).cur[d,(*state).nob],2L)+'] '+objectfile
          tmp2=' :: right click to change the orientation.'
          widget_control,(*state).wid[d].dobjl,set_uname=tmp+tmp2
          tmp_=(*state).cur[d,(*state).nob]
          widget_control,(*state).wid[d].dobji[tmp_],set_uname=tmp

          ;; Updating the wavelength bin slider:
          widget_control,(*state).wid[d].sobjl,set_slider_max= $
                         s[(*state).daxis]-1L
          widget_control,(*state).wid[d].sobjl, $
                         set_value=(*state).curb[(*state).d]
          widget_control,(*state).wid[d].sobjl,sensitive=1L

        endif else begin ;; else condition: ~(*state).nf_cancel

          wset,(*state).wid[d].dleft

          i=(*state).cur[d,(*state).nob]
          if i ne -1L then begin

            ;; Updating the left and right object images on the main panel:
            p3d_gui_draw,i,state,event.top,image,/left

            widget_control,(*state).wid[d].tcmin, $
                           set_value=strtrim(minval,2L)
            widget_control,(*state).wid[d].tcmax, $
                           set_value=strtrim(maxval,2L)
            widget_control,(*state).wid[d].twave, $
                           set_value=strtrim((*state).curb[d],2L)

            ;; Updating the widget showing the wavelength:
            p3d_gui_setwave,i,state,/updateuname

            ;; Updating the wavelength bin slider:
            widget_control,(*state).wid[d].sobjl, $
                           set_slider_max=s[(*state).daxis]-1L
            widget_control,(*state).wid[d].sobjl, $
                           set_value=(*state).curb[d]
            widget_control,(*state).wid[d].sobjl,sensitive=1L
          endif else begin

            p3d_display_tele,intarr(2L,2L),0,1,(*state).leftsize, $
                (*state).leftsize,topwid=event.top,logunit=(*state).logunit, $
                verbose=(*state).verbose,error=error,debug=(*state).debug
            if error ne 0 then return

            widget_control,(*state).wid[d].tobjf,set_value=''
            widget_control,(*state).wid[d].lobji,set_value=''

            ;; Updating the wavelength bin slider:
            widget_control,(*state).wid[d].sobjl,sensitive=0L

          endelse ;; (*state).cur[d,(*state).nob] ne -1L
        endelse ;; ~(*state).nf_cancel

        widget_control,event.top,update=1L

      endif ;; filename ne ''

    end ;; case: 'browseobject'


    ;;========================================------------------------------
    ;; Select an already loaded file:

    'selectobject': begin

      p3d_gui_select_file,(*state).nob,state,error=error
      if error ne 0 then return

      d=(*state).d & nob=(*state).nob

      if (*state).cur[d,nob] ne -1L then begin
        i=(*state).cur[d,nob]
        widget_control,(*state).wid[d].tobjf,set_value=(*state).file[d,nob,i]
        widget_control,(*state).wid[d].lobji,set_value=strtrim(i,2L)
        widget_control,(*state).wid[d].tcmin, $
                       set_value=strtrim((*state).ccmin[d],2L)
        widget_control,(*state).wid[d].tcmax, $
                       set_value=strtrim((*state).ccmax[d],2L)
        widget_control,(*state).wid[d].twave, $
                       set_value=strtrim((*state).curb[d],2L)

        ;; Updating the widget showing the wavelength:
        p3d_gui_setwave,i,state,/updateuname

;        image=reform((*state).daxis? $
;                     (*(*state).curdata[(*state).nob])[(*state).curb,*]: $
;                     (*(*state).curdata[(*state).nob])[*,(*state).curb])

        ;; Updating the left and right object images on the main panel:
        p3d_gui_draw,i,state,event.top,/left

      endif else begin
        widget_control,(*state).wid[d].tobjf,set_value=''
        widget_control,(*state).wid[d].lobji,set_value=''
        widget_control,(*state).wid[d].tcmin,set_value=''
        widget_control,(*state).wid[d].tcmax,set_value=''
        widget_control,(*state).wid[d].twave,set_value=''
        widget_control,(*state).wid[d].lwave,set_value=''

        wset,(*state).wid[d].dleft
        p3d_display_tele,intarr(2L,2L),0L,1L, $
            (*state).leftsize,(*state).leftsize,topwid=topwid, $
            verbose=(*state).verbose,error=error,debug=(*state).debug
        if error ne 0 then return

      endelse ;; (*state).cur[(*state).d,(*state).nob] ne -1L
      widget_control,(*state).wid[d].bobvi[0L],sensitive=i ne -1L
      widget_control,(*state).wid[d].bobvi[1L],sensitive=i ne -1L

    end ;; case: 'selectobject'


    ;;========================================------------------------------
    ;; View the object raw data, and with the overlayed trace mask:

    'viewobject': begin

      d=(*state).d
      num=(*state).cur[d,(*state).nob]
      objectfile=reform((*state).path[d,(*state).nob,num] + $
                        (*state).file[d,(*state).nob,num])

      title='[object] "'+objectfile+'"'

      if ~index then begin
        p3d_imv,objectfile,title=title,bottom=(*state).cbottom, $
            xscrsize=xscrsize,yscrsize=yscrsize,group_leader=event.top, $
            verbose=(*state).verbose,error=error,debug=(*state).debug
        if error ne 0 then return

      endif else begin
        data=readfits(objectfile,hdr,silent=(*state).verbose lt 3,/no_unsigned)

        ;; Attempting to read the errors file:
        dff=0L
        dfile=fxpar(hdr,'IMERR')
        if dfile ne '0' then begin
          i=1L & more=1L
          while more do begin
            tmp=fxpar(hdr,'IMERR'+strtrim(i++,2L))
            if tmp ne '0' then dfile+=strtrim(tmp,2L) else more=0L
          endwhile

          dff=file_test(dfile,/read,/regular)
        endif ;; dfile ne '0'

        if ~dff then begin
          ;; Making another attempt, with a slightly different filename:
          sfx=(*state).sfx & str=(*state).ffsfx
          tmp=strpos(objectfile,sfx,/reverse_search)
          fffx=strmid(objectfile,tmp)
          dfile=strmid(objectfile,0L,tmp)+'_err'+fffx
          if file_test(dfile,/read,/regular) then dff=1L
        endif
        if dff then $
           ddata=readfits(dfile,silent=(*state).verbose lt 3,/no_unsigned)

        ;; Starting the spectrum viewer application:
        idx=d*(*state).nblocks;;+(*state).block
;        colorcut=[(*state).ccmin[d],(*state).ccmax[d]]
        htitle='Fits Header: "'+objectfile+'"'

        tvlct,(*state).colors.red,(*state).colors.green,(*state).colors.blue
        p3d_rss,data,(*(*state).pos[idx]).file,dimage=ddata,hdr=hdr, $
            orientation=(*state).orientation[d,num], $
            bottom=(*state).cbottom,cindex=(*state).cindex, $
            cindv=(*state).cindv,title=title,fitsheadertitle=htitle, $
            colortable=(*state).colortable, $
            daxis=(*state).daxis,group_leader=event.top, $
            verbose=(*state).verbose,error=error,debug=(*state).debug
      endelse ;; ~index

    end ;; case: 'viewobject'



    ;;========================================------------------------------
    ;; Other operations:

    'restart': begin

      ret=dialog_message(['Are you sure you want to restart p3d?', $
              'All loaded information will be lost.'],/center, $
               /default_no,dialog_parent=event.top,/question)

      if ret eq 'Yes' then begin
         widget_control,event.top,/iconify

        ;; Starting p3d anew (should maybe add logfile...):
        p3d,debug=(*state).debug,verbose=(*state).verbose, $
            loglevel=(*state).logunit[1L],cwd=(*state).sepwpath,/restart

        ;; Destroying the current instance of p3d:
        widget_control,event.top,/destroy
      endif
    end ;; case: 'restart'

    'exit': begin
      msg='Are you sure that you want to exit p3d?'
      ret=dialog_message(msg,/center,dialog_parent=event.top, $
              /question,/default_no)
      if ret eq 'Yes' then begin
        widget_control,event.top,/destroy
        retall
      endif
    end ;; case: 'exit'

    'minmaxlimits': begin

      d=(*state).d

      ;; Extracting the percentage:
      widget_control,event.id,get_value=val
      val=val[event.index]
      data=*(*state).curdata[d,(*state).nob]
      tmp=(*state).curb[d]
      image=reform((*state).daxis?data[tmp,*]:data[*,tmp])

      i=(*state).cur[d,(*state).nob]
      (*state).slid[d,i]=event.id

      if ~event.id then begin
        minval=min(image,max=maxval)
      endif else begin

        ;; Calculating the clipping range:
        tmp=(*state).scalebinsseparately?image:data
        p3d_misc_dataclip,tmp,range,percentage=val,topwid=event.top, $
            logunit=(*state).logunit,verbose=(*state).verbose,error=error, $
            debug=(*state).debug
        if error ne 0 then return

        (*state).slidp[d,i]=val
        minval=range[0L] & maxval=range[1L]
      endelse

      (*state).ccmin[d]=minval & (*state).cmin[d,i]=(*state).ccmin[d]
      (*state).ccmax[d]=maxval & (*state).cmax[d,i]=(*state).ccmax[d]

      ;; Updating the left and right object images on the main panel:
      p3d_gui_draw,i,state,event.top,/left,/right

      ;; Updating the min- and max-value text widgets:
      widget_control,(*state).wid[d].tcmin, $
                     set_value=strtrim((*state).ccmin[d],2L)
      widget_control,(*state).wid[d].tcmax, $
                     set_value=strtrim((*state).ccmax[d],2L)

    end ;; case: 'minmaxlimits'

    'colminmax': begin
      d=(*state).d

      widget_control,event.id,get_value=val

      i=(*state).cur[d,(*state).nob]
      case index of
         0: begin
            (*state).ccmin[d]=float(val[0L])
            (*state).cmin[d,i]=(*state).ccmin[d]
         end
         1: begin
            (*state).ccmax[d]=float(val[0L])
            (*state).cmax[d,i]=(*state).ccmax[d]
         end
      endcase ;; index

      ;; Setting the droplist values to not be used:
      (*state).slid[d,i]=-1L
      widget_control,(*state).wid[d].bccpe,set_droplist_select=0L

      ;; updating the left and right object images on the main panel:
      p3d_gui_draw,i,state,event.top,/left,/right

    end ;; case: 'colminmax'

    'wavebar': begin
      widget_control,event.id,get_value=val
      tmp=long(val[0L])

      d=(*state).d
      i=(*state).cur[d,(*state).nob]

      (*state).curb[d]=tmp
      (*state).column[d,i]=(*state).curb[d]

      if (*state).scalebinsseparately then begin
        image=reform((*state).daxis? $
                     (*(*state).curdata[d,(*state).nob])[tmp,*]: $
                     (*(*state).curdata[d,(*state).nob])[*,tmp])
      endif else begin
        image=reform((*(*state).curdata[d,(*state).nob]))
      endelse

      s=size(*(*state).curdata[d,(*state).nob])

      if (*state).slid[d,i] ge 0L then begin
        val=(*state).slidp[d,i]

        ;; Calculating the clipping range:
        p3d_misc_dataclip,image,range,percentage=val,topwid=event.top, $
            logunit=(*state).logunit,verbose=(*state).verbose,error=error, $
            debug=(*state).debug
        if error ne 0 then return

        minval=range[0L] & (*state).ccmin[d]=minval
        maxval=range[1L] & (*state).ccmax[d]=maxval

        ;; Setting the range widget values:
        widget_control,(*state).wid[d].tcmin,set_value=strtrim(minval,2L)
        widget_control,(*state).wid[d].tcmax,set_value=strtrim(maxval,2L)
      endif

      ;; Updating the left and right object images on the main panel:
      p3d_gui_draw,i,state,event.top,/left,/right

      ;; Changing the uname of the two draw widgets:
      tmptmp=':: right click to change the orientation.'
      tmp=widget_info((*state).wid[d].dobjl,/uname)
      tmppos=strpos(tmp,'::')
      if tmppos ne -1L then begin
        tmp=strmid(tmp,0L,tmppos)
        widget_control,(*state).wid[d].dobjl,set_uname=tmp+tmptmp
        widget_control,(*state).wid[d].dobji[i],set_uname=tmp
      endif

      ;; Updating the wave text widget:
      widget_control,(*state).wid[d].twave, $
                     set_value=strtrim((*state).curb[d],2L)

      ;; Updating the widget showing the wavelength:
      p3d_gui_setwave,i,state

;      wavelength=(*state).crvd[0L,i]+ $
;                 (*state).crvd[1L,i]*((*state).curb+0.5)
;      tmp=string(wavelength,format='(f8.2,"")')
;      widget_control,(*state).wid[(*state).d].lwave,set_value=strtrim(tmp,2L)

    end ;; case: 'wavebar'

    'wave': begin
      d=(*state).d
      i=(*state).cur[d,(*state).nob]
      if i ne -1L then begin
        widget_control,event.id,get_value=val
        s=size(*(*state).curdata[d,(*state).nob])
        tmp=long(val[0L])

        if tmp ge 0L and tmp le s[(*state).daxis]-1L then begin
          (*state).curb[d]=tmp
          (*state).column[d,i]=(*state).curb[d]

          if (*state).scalebinsseparately then begin
            image=reform((*state).daxis? $
                         (*(*state).curdata[d,(*state).nob])[tmp,*]: $
                         (*(*state).curdata[d,(*state).nob])[*,tmp])
          endif else begin
            image=reform((*(*state).curdata[d,(*state).nob]))
          endelse

          if (*state).slid[d,i] ge 0L then begin
            val=(*state).slidp[d,i]

            ;; Calculating the clipping range:
            p3d_misc_dataclip,image,range,percentage=val,topwid=event.top, $
                logunit=(*state).logunit,verbose=(*state).verbose, $
                error=error,debug=(*state).debug
                              
            if error ne 0 then return

            minval=range[0L]
            maxval=range[1L]

            (*state).ccmin[d]=minval
            (*state).ccmax[d]=maxval

            ;; Setting the range widget values:
            widget_control,(*state).wid[d].tcmin,set_value=strtrim(minval,2L)
            widget_control,(*state).wid[d].tcmax,set_value=strtrim(maxval,2L)
         endif

         ;; Updating the left and right object images on the main panel:
         p3d_gui_draw,i,state,event.top,/left,/right

         ;; Changing the uname of the two draw widgets:
         tmptmp=':: right click to change the orientation.'
         tmp=widget_info((*state).wid[d].dobjl,/uname)
         tmppos=strpos(tmp,'::')
         if tmppos ne -1L then begin
           tmp=strmid(tmp,0L,tmppos)
           widget_control,(*state).wid[d].dobjl,set_uname=tmp+tmptmp
           widget_control,(*state).wid[d].dobji[i],set_uname=tmp
         endif

         ;; Updating the widget showing the wavelength:
         p3d_gui_setwave,i,state

;         ;; Updating the wavelength label
;         wavelength=(*state).crvd[0L,i]+ $
;                    (*state).crvd[1L,i]*((*state).curb+0.5)
;         tmp=string(wavelength,format='(f8.2,"")')
;         widget_control,(*state).wid[(*state).d].lwave,set_value=strtrim(tmp,2L)

         ;; Updating the slider:
         widget_control,(*state).wid[d].sobjl,set_value=(*state).curb[d]

       endif else begin
         widget_control,(*state).wid[d].twave, $
                        set_value=strtrim((*state).curb[d],2L)

         ;; Updating the widget showing the wavelength:
         p3d_gui_setwave,i,state,/updateuname

;         wavelength=(*state).crvd[0L,i]+ $
;                    (*state).crvd[1L,i]*((*state).curb+0.5)
;         tmp=string(wavelength,format='(f8.2,"")')
;         widget_control,(*state).wid[(*state).d].lwave,set_value=strtrim(tmp,2L)
       endelse ;; tmp ge 0L and tmp le s[(*state).daxis]-1L

      endif else begin
        widget_control,(*state).wid[d].twave,set_value=''
        widget_control,(*state).wid[d].lwave,set_value=''
      endelse ;; (*state).cur[(*state).d,(*state).nob] ne -1L

    end ;; case: 'wave'

    ;;===============-----

    'displib': begin
      (*state).displibset=((*state).displibset+1L) mod 2
      if (*state).displibset then begin
        widget_control,(*state).wid[(*state).d].tdisf, $
                       set_value='select from library'
        widget_control,(*state).wid[(*state).d].ldiai,set_value=''
      endif else begin
        i=(*state).cur[(*state).d,(*state).ndi]
        if i ne -1L then begin
          widget_control,(*state).wid[(*state).d].tdisf, $
                         set_value=(*state).file[(*state).d,(*state).ndi,i]
          widget_control,(*state).wid[(*state).d].ldiai,set_value=strtrim(i,2L)
        endif else begin
          widget_control,(*state).wid[(*state).d].tdisf,set_value=''
          widget_control,(*state).wid[(*state).d].ldiai,set_value=''
        endelse
      endelse

      tmp=(1L-(*state).displibset)* $
          ((*state).cur[(*state).d,(*state).ntr] ne -1L)
;     widget_control,(*state).wid[(*state).d].bdibr,sensitive=tmp ;; unnecc?
      widget_control,(*state).wix.bdisp            ,sensitive=tmp
          
      widget_control,displib,set_button=(*state).displibset
      widget_control,(*state).wid[(*state).d].tdisf, $
                     sensitive=~(*state).displibset
      widget_control,(*state).wid[(*state).d].ldiai, $
                     sensitive=~(*state).displibset
;     widget_control,(*state).wid[(*state).d].bdise,
;                    sensitive=~(*state).displibset
      widget_control,(*state).wid[(*state).d].bdibr, $
                     sensitive=~(*state).displibset
      if widget_info((*state).wid[(*state).d].bdiss,/valid_id) then $
         widget_control,(*state).wid[(*state).d].bdiss, $
                        sensitive=~(*state).displibset * $
                        ((*state).cur[(*state).d,(*state).ndi] ge 1L)
      if (*state).displibset then begin
         (*state).pisreadydisp=1L
      endif else begin
        (*state).pisreadydisp= $
           (*state).cur[(*state).d,(*state).ndi] ne -1L?1L:0L
      endelse
    end ;; case: 'displib'

    'shiftinspector': begin
      if (*state).cur[(*state).d,(*state).nob] ne -1L then begin
        if (*state).cur[(*state).d,(*state).ndi] ge 1L and $
           ~(*state).displibset then begin
          dfile=(*state).path[(*state).d,(*state).ndsh, $
                              (*state).cur[(*state).d,(*state).ndi]] + $
                (*state).file[(*state).d,(*state).ndsh, $
                              (*state).cur[(*state).d,(*state).ndi]]
        endif else begin
          difile=0
        endelse

;;        shift_inspector,(*state).path[(*state).ntsh, $
;;                                      (*state).cur[(*state).ntr]] + $
;;                        (*state).file[(*state).ntsh, $
;;                                      (*state).cur[(*state).ntr]], $
;;                        (*state).path[(*state).nosh, $
;;                                      (*state).cur[(*state).nob]] + $
;;                        (*state).file[(*state).nosh, $
;;                                      (*state).cur[(*state).nob]], $
;;                        *(*state).curdata[(*state).nbx],file3=dfile
      endif
    end ;; case: 'shiftinspector'

;    'e3d': begin
;      if (*state).cur[(*state).d,(*state).nob] ne -1L then begin
;        so = size(*(*state).curdata[(*state).d,(*state).nob])
;
;        if ptr_valid((*state).curdata[(*state).d,(*state).nob]) then $
;            ptr_free,(*state).curdata[(*state).d,(*state).nob]
;        (*state).curdata[(*state).d,(*state).nob]= $
;           ptr_new(*(*state).obje[(*state).cur[(*state).d,(*state).nob]])
;
;;;        e3dname=get_basename((*state).path[(*state).nob, $
;;;                                           (*state).cur[(*state).nob]]+ $
;;;                             (*state).file[(*state).nob, $
;;;                                           (*state).cur[(*state).nob]])+ $
;;;                '.e3d'
;;;         ppak2e3d,pathobj(curobje_nr-1)+fileobj(curobje_nr-1),e3dname
;;;        any2e3d,(*state).path[(*state).nob,(*state).cur[(*state).nob]]+ $
;;;                (*state).file[(*state).nob,(*state).cur[(*state).nob]], $
;;;                e3dname,(*state).e3dpostab,so(2)
;        spawn,'ps c | grep tk_e3d.tcl',exist
;        if (exist[0] eq "") then begin
;          spawn,'tk_e3d.tcl &'
;          wait,2
;        endif
;;       e3d_display,e3dname,colcuts=[(*state).ccmin,(*state).ccmax]
;      endif
;    end ;; case: 'e3d'

    'focus': begin
      widget_control,event.top,/iconify
;      imfocus,(*state).path[(*state).ntr,(*state).cur[(*state).ntr]] + $
;              (*state).file[(*state).ntr,(*state).cur[(*state).ntr]]
    end

    'orient': begin
      if ptr_valid((*state).curdata[(*state).d,(*state).nob]) then begin
        ;; updating the left and right object images on the main panel:
        i=(*state).cur[(*state).d,(*state).nob]
        (*state).orientation[(*state).d,i]=index
        p3d_gui_draw,i,state,event.top,/left,/right
      endif
    end ;; case: 'orient'

    'leftdraw': begin
      ;; nothing is done...
    end

    'detectortab': begin
      (*state).d=event.tab
    end

    'seldetector': begin
      d=(*state).d & nd=n_elements((*state).curb)

      case index of
        0: begin
          --d
          if d lt 0L then d=nd-1L
        end
        1: begin
          ++d
          if d ge nd then d=0L
        end
      endcase ;; index

      ;; Storing the newly selected detector:
      (*state).d=d

      ;; Updating widgets:
      widget_control,(*state).wix.btabw,set_tab_current=d

    end ;; case: 'seledetector'

    'right': begin
      nr=index

      d=(*state).d
      if (*state).file[d,(*state).nob,nr] ne '' then begin
        (*state).cur[d,(*state).nob]=nr
        (*state).ccmin[d]=(*state).cmin[d,nr]
        (*state).ccmax[d]=(*state).cmax[d,nr]
        (*state).curb[d]=(*state).column[d,nr]

        ;; Retrieving already loaded data:
        if ptr_valid((*state).curdata[d,(*state).nob]) then $
            ptr_free,(*state).curdata[d,(*state).nob]
        (*state).curdata[d,(*state).nob]= $
           ptr_new(*(*state).data[d,(*state).nob,nr])
        s=size(*(*state).curdata[d,(*state).nob])

        widget_control,(*state).wid[d].tobjf, $
                       set_value=(*state).file[d,(*state).nob,nr]
        widget_control,(*state).wid[d].lobji,set_value=strtrim(nr,2L)
        widget_control,(*state).wid[d].tcmin, $
                       set_value=strtrim((*state).ccmin[d],2L)
        widget_control,(*state).wid[d].tcmax, $
                       set_value=strtrim((*state).ccmax[d],2L)
        widget_control,(*state).wid[d].twave, $
                       set_value=strtrim((*state).curb[d],2L)

        ;; Updating the widget showing the wavelength:
        p3d_gui_setwave,nr,state

;        wavelength=(*state).crvd[0L,nr]+ $
;                   (*state).crvd[1L,nr]*((*state).curb+0.5)
;        tmp=string(wavelength,format='(f8.2,"")')
;        widget_control,(*state).wid[(*state).d].lwave,set_value=strtrim(tmp,2L)

        ;; Updating the wavelength bin slider:
        widget_control,(*state).wid[d].sobjl, $
                       set_slider_max=s[(*state).daxis]-1L
        widget_control,(*state).wid[d].sobjl, $
                       set_value=(*state).curb[d]

        ;; Updating the left and right object images on the main panel:
        p3d_gui_draw,nr,state,event.top,/left

      endif ;; (*state).file[(*state).d,(*state).nob,nr] ne ''
    end ;; case: 'right'

    'colortable': begin
      xloadct,bottom=(*state).cbottom,group=event.top, $
          ncolors=!d.table_size-(*state).cbottom,updatecbdata=state, $
          /use_current,updatecallback='p3d_gui_xloadct'

       ;; Only allowing one instance of xloadct to be run:
       widget_control,(*state).wix.bcolo,sensitive=0L
    end ;; case: 'colortable'

    'colortablesel': begin

      p3d_misc_colortable,index,bottom=(*state).cbottom,colors=colors, $
          topwid=event.top,logunit=(*state).logunit,error=error, $
          verbose=(*state).verbose,debug=(*state).debug
      if error ne 0 then return

      (*state).colortable=index
      (*state).colors=colors

      device,get_decomposed=decomposed
      if decomposed then begin
        if ptr_valid((*state).curdata[(*state).d,(*state).nob]) then begin
          i=(*state).cur[(*state).d,(*state).nob]
          ;; Redrawing the left and right object images on the panel:
          p3d_gui_draw,i,state,(*state).top,image,/left,/right
        endif
      endif ;; decomposed
      
    end ;; case: 'colortable'

    'selcwd': begin
      ;; Setting or unsetting the 'current working directory' option:

      (*state).sepwpath=~(*state).sepwpath
      if (*state).sepwpath then begin
        path=(*state).outputpath eq ''? $
             (*state).inputpath:file_dirname((*state).outputpath)
        directory=dialog_pickfile(path=path,/directory, $
            file=file_basename((*state).outputpath), $
            title='[CWD] Please select a directory to write to.')

        if directory eq '' then begin
          (*state).sepwpath=0L
        endif else if file_test(directory,/regular) then begin
          (*state).sepwpath=0L
           
          ;; Writing information to a message dialog:
          ret=dialog_message(['The chosen directory is a regular file.', $
                  'p3d will continue to write the data in the', $
                  'same directory where the raw data is found.'], $
                  /information,dialog_parent=event.top,/center)
        endif else begin ;; file_test(directory,/regular)

          dircr=0L
          if ~file_test(directory) then begin
            dircr=1L
            file_mkdir,directory
          endif
          (*state).outputpath=directory
          (*state).inputpath=directory

          ;; Writing information to a message dialog:
          msg=['The chosen directory','"'+(*state).outputpath+'"', $
               'is now used to write all output data.']
          if dircr then $
             msg=[msg,'This directory was created as it did not exist.']
          ret=dialog_message(msg,/information,dialog_parent=event.top,/center)

        endelse ;; file_test(directory,/regular)
      endif

      value=(*state).sepwpath?'Using a separate output directory': $
                     'Using the raw data directory'
      uname='Currently writing to "'+(*state).outputpath+'".'
      widget_control,(*state).wix.bcwdi[0L],set_value=value
      widget_control,(*state).wix.bcwdi[1L],sensitive= (*state).sepwpath
      widget_control,(*state).wix.bcwdi[1L],set_uname=uname
      widget_control,(*state).wix.bcwdi[2L],sensitive=~(*state).sepwpath
    end ;; case: 'selcwd'

    'viewinspar': begin
      xdisplayfile,(*state).parfile,done_button='Exit',/grow,width=90L, $
          group=event.top,title=p3d_misc_pathify((*state).parfile)
    end ;; case: 'viewinspar'

    'seleuserpar': begin

      title='Please select a filename.'

      if (*state).userparfile ne '' then begin
        path=file_dirname((*state).userparfile)
        file=file_basename((*state).userparfile)
      endif else begin
        path=(*state).sepwpath?(*state).outputpath:(*state).inputpath
        file='user_p3d.prm'
      endelse
      filename=dialog_pickfile(filter=['*.prm;*.dat'], $
          title=title,path=path,file=file,dialog_parent=event.top)

      if filename ne '' then begin
        (*state).userparfile=filename

        ;; Copying the master user parameter file if it does not exist:
        if ~file_test(filename,/regular) then begin
          mfile=!p3d_path+'data'+path_sep()+'master_userparfile.dat'
          file_copy,mfile,filename,verbose=(*state).verbose ge 3
        endif ;; ~file_test(filename,/regular)

        ;; Sensitizing the user parameter file widget buttons:
        widget_control,(*state).wix.bviup,sensitive=1L
        widget_control,(*state).wix.bedup,sensitive=1L
      end ;; filename ne ''

    end ;; case: 'seleuserpar'

    'viewlogfile': begin
      path=(*state).xlogfile eq ''? $
           (*state).inputpath:file_dirname((*state).xlogfile)
      title='Please select a (text) log file.'
      (*state).xlogfile=dialog_pickfile($
          filter=['*.txt;*.log;*.dat','DP-SUMMARY','*.*'], $
          title=title,path=path,/must_exist,dialog_parent=event.top, $
          file=file_basename((*state).xlogfile))

      if (*state).xlogfile ne '' then begin

        ;; Finding out what the maximum number of columns is:
        nmax=0L
        openr,unit,(*state).xlogfile,/get_lun
        while ~eof(unit) do begin
          tmp=''
          readf,unit,tmp,format='(a)'
          tmp=strlen(strtrim(tmp))
          if tmp gt nmax then nmax=tmp
        endwhile
        free_lun,unit
        nmax=nmax>80L

        xdisplayfile,(*state).xlogfile,done_button='Exit',/grow, $
            group=event.top,width=nmax,title='Viewing the file: "'+ $
            p3d_misc_pathify((*state).xlogfile,/dpath)+'"'
      endif ;; (*state).xlogfile ne ''
    end ;; case: 'viewlogfile'

    'viewdrtimer': begin

      widget_control,(*state).wix.blogt[0L],get_value=tmp
      idx0=strpos(tmp,'[')
      str=strmid(tmp,0L,idx0+1L)

      switch long(index) of
        0: begin
          ;; Turning off the timer:
          (*state).logfileusetimer=0L

          str+='off]'
          break
        end
        else: begin 
          ;; Setting the timer:
          restimer=~(*state).logfileusetimer
          (*state).logfiletimer=index
          (*state).logfileusetimer=1L

          istr=strtrim(long(index),2L)
          str+=(strlen(istr) eq 1L?' ':'')+istr+'s]'

          ;; Issuing a new timer event, in case the timer was switched off:
          if widget_info((*state).logfiletid,/valid_id) and restimer then begin
            widget_control,(*state).wix.blogr,timer=(*state).logfiletimer
            widget_control,(*state).wix.blogt[0L],timer=(*state).logfiletimer
          endif

        end
      endswitch

      widget_control,(*state).wix.blogt[0L],set_value=str

    end ;; case: 'viewdrtimer'

    'viewdrlogfile': begin
      tmp=fstat((*state).logunit[0L])
      if tmp.open then begin

        ;; Reading the text in the file:
        text=p3d_gui_readdr((*state).logfile,offset=offset)
        (*state).logfileoffset=offset

        if ptr_valid((*state).logfiletext) then ptr_free,(*state).logfiletext
        (*state).logfiletext=ptr_new(text)
        (*state).logfilenrows=n_elements(text)

        xdisplayfile,(*state).logfile,done_button='Exit',/grow,text=text, $
            group=event.top,title='Data reduction log file: "'+ $
            p3d_misc_pathify((*state).logfile,/dpath)+'"',wtext=wtextid
        (*state).logfiletid=wtextid

;        ;; Only allowing one instance of the logfile-viewer to be used:
;        widget_control,(*state).wix.blogr,sensitive=0L

        ;; Starting a widget_timer, in order to update the widget...
        widget_control,(*state).wix.blogr,timer=(*state).logfiletimer
        widget_control,(*state).wix.blogt[0L],timer=(*state).logfiletimer

      endif ;; tmp.open
    end ;; case: 'viewdrlogfile'

    'viewuserpar': begin
      xdisplayfile,(*state).userparfile,done_button='Exit',/grow,width=90L, $
          group=event.top,title=p3d_misc_pathify((*state).userparfile,/dpath)
    end ;; case: 'viewuserpar'

    'edituserpar': begin
      xdisplayfile,(*state).userparfile,done_button='Exit',/grow,/editable, $
          group=event.top,width=90L,title= $
          p3d_misc_pathify((*state).userparfile,/dpath)
    end ;; case: 'edituserpar'

    'logopen': begin
      
      path=(*state).sepwpath?(*state).outputpath:(*state).inputpath
      tmpstr=index?'an existing file':'a filename'
      title='Please select '+tmpstr+' for writing.'
      filename=dialog_pickfile(filter='*.log',title=title,path=path,/write, $
          dialog_parent=event.top,overwrite_prompt=~index,must_exist=index)

      if filename ne '' then begin
        ;; Opening a log file:
        on_ioerror,error_handler
        openw,unit,filename,/get_lun,append=index
        on_ioerror,NULL

        (*state).logunit[0L]=unit
        (*state).logfile=filename

        ;; Writing information to the status line:
        tmpstr='Opened the log-file "'+filename+'".'
        widget_control,(*state).wix.lstat,set_value=tmpstr

        ;; Changing the logfile widgets:
        widget_control,(*state).wix.bloga,sensitive=0L
        widget_control,(*state).wix.blogo,sensitive=0L
        widget_control,(*state).wix.blogc,sensitive=1L

        widget_control,(*state).wix.blogr,sensitive=1L
        widget_control,(*state).wix.blogt[0L],sensitive=1L

        widget_control,(*state).wix.llogi,set_value='Log: yes'
        widget_control,(*state).wix.llogi, $
            set_uname='Operations are logged to "'+ $
            p3d_misc_pathify(filename,/dpath)+'".'

        p3d_gui_openedlog,state

      endif ;; filename ne ''
    end ;; case: 'logopen'

    'logclose': begin

      ;; Closing the log file:
      tmp=fstat((*state).logunit[0L])
      if tmp.open then free_lun,(*state).logunit[0L]
      (*state).logunit[0L]=1L

      p3d_gui_openedlog,state,/close

      ;; Writing information to the status line:
      tmpstr='Closed the log-file "'+tmp.name+'".'
      widget_control,(*state).wix.lstat,set_value=tmpstr

      ;; Changing the logfile widgets:
      widget_control,(*state).wix.bloga,sensitive=1L
      widget_control,(*state).wix.blogo,sensitive=1L
      widget_control,(*state).wix.blogc,sensitive=0L

      widget_control,(*state).wix.blogr,sensitive=0L
      widget_control,(*state).wix.blogt[0L],sensitive=0L

      widget_control,(*state).wix.llogi,set_value='Log: no '
      widget_control,(*state).wix.llogi, $
          set_uname='The operations are not logged.'
    end ;; case: 'logclose'

    'dbin': begin
       (*state).dbin=index

       ;; Changing the logfile button widgets:
       tmpstr='Log level ['+strtrim(index,2L)+']'
       case index of
         1L: tmpstr='[create disp.mask]: data are not rebinned on the dis' + $
                    'p. axis.'
         2L: tmpstr='[create disp.mask]: data are rebinned on the disp.ax' + $
                    'is by a factor 2.'
         else: tmpstr='[create disp.mask]: data are rebinned on the disp.' + $
                    'axis until they fit the screen.'
       endcase ;; index
       widget_control,(*state).wix.bdbin,set_uname=tmpstr

       tm2str='D.mask rebin data [dbin='+strtrim(index,2L)+']'
       widget_control,(*state).wix.bdbin,set_value=tm2str

       ;; Writing information to the status line:
       widget_control,(*state).wix.lstat,set_value=tmpstr
    end ;; case: dbin

    'loglevel': begin

       (*state).logunit[1L]=index

       ;; Changing the logfile button widgets:
       tmpstr='Log level ['+strtrim(index,2L)+']'
       widget_control,(*state).wix.blogl,set_value=tmpstr

       ;; Writing information to the status line:
       tmpstr='Set the logging level to '+strtrim(index,2L)+'.'
       widget_control,(*state).wix.lstat,set_value=tmpstr
    end ;; case: loglevel

    'verbosity': begin

       (*state).verbose=byte(index)

       ;; Changing the verbosity button widget:
       tmpstr='Verbose level ['+strtrim(index,2L)+']'
       widget_control,(*state).wix.bverb,set_value=tmpstr

       ;; Writing information to the status line:
       tmpstr='Set the verbose level to '+strtrim(index,2L)+'.'
       widget_control,(*state).wix.lstat,set_value=tmpstr
    end ;; case: verbosity

    'debugging': begin

       (*state).debug=~(*state).debug

       ;; Changing the debug button widget:
       tmpstr='Debug ['+strtrim((*state).debug,2L)+' {='+ $
              ((*state).debug?'yes':'no')+'}]'
       widget_control,(*state).wix.bdebg,set_value=tmpstr

       ;; Writing information to the status line:
       tmpstr='Set the debug mode to '+strtrim((*state).debug,2L)+'.'
       widget_control,(*state).wix.lstat,set_value=tmpstr
    end ;; case: debugging

    'about': begin
      tmpstr=['p3d: a general data-reduction tool for fiber-fed IFSs '+ $
              '(version 1.1.1 - Upperkuchen)', $
              '', $
              'Copyright 2009,2010 by the Astrophysical Institute Potsdam ' + $
               '(AIP), Germany', $
              '', $
              'This is an incomplete excerpt of the README file', $
              '', $
              'Programming and testing:', $
              '  Christer Sandin (AIP)', $
              '  Joris Gerssen (AIP)', $
              '  Peter Weilbacher (AIP)', $
              '', $
              'The project web site is:', $
              '  http://p3d.sourceforge.net', $
              '', $
              'When using this code, please cite the following paper:', $
              '  Sandin C., Becker T., Roth M.M., Gerssen J., Monreal-Ibe' + $
              'ro A., Bhm P.,', $
              '  Weilbacher P. 2010, A&A, in press.', $
              '', $
              'This program is a complete rewrite of an earlier version' + $
              ' that was', $
              '  developed by Thomas Becker and Martin M. Roth (both AIP).']

      ret=dialog_message(tmpstr,/information,/center,dialog_parent=event.top)
    end ;; case: 'about'

    'copying': begin
      xdisplayfile,!p3d_path+path_sep()+'COPYING',done_button='Exit',/grow, $
          width=80L,group=event.top,title='p3d: distribution'
    end ;; case: 'copying'

  endcase ;; uval

  return

error_handler:
  if n_elements(unit) eq 1L then begin
     tmp=fstat(unit)
     if tmp.open then free_lun,unit
     errmsg=!error_state.msg
  endif

  error=p3d_misc_logger(errmsg,(*state).logunit,rname=rname,topwid=event.top, $
            verbose=(*state).verbose,/error)
  return

end ;;; end of p3d_gui_event


pro p3d_gui_cleanup,id
  compile_opt hidden,idl2

  ;; Get the state information:
  widget_control,id,get_uvalue=state

  if (*state).verbose ge 3 then print,'p3d_gui: cleanup.'

  for i=0L,n_elements((*state).curdata)-1L do ptr_free,(*state).curdata[i]
  for i=0L,n_elements((*state).data   )-1L do ptr_free,(*state).data[i]
  for i=0L,n_elements((*state).hdr    )-1L do ptr_free,(*state).hdr[i]
  for i=0L,n_elements((*state).pos    )-1L do ptr_free,(*state).pos[i]

  if ptr_valid((*state).hhdr)     then ptr_free,(*state).hhdr
  if ptr_valid((*state).parname)  then ptr_free,(*state).parname
  if ptr_valid((*state).parvalue) then ptr_free,(*state).parvalue
  if ptr_valid((*state).logfiletext) then ptr_free,(*state).logfiletext

  tmp=fstat((*state).logunit[0L])
  if tmp.open then free_lun,(*state).logunit[0L]
  
  if ptr_valid(state) then ptr_free,state

  return
end ;;; end of: p3d_gui_cleanup


pro p3d_gui,parfile,cwd=cwd,path=path,title=title,detector=detector, $
        userparfile=userparfile,displib=displib,box=box, $
        colortable=colortable,bottom=bottom,cindex=cindex, $
        tracking_events=track,logfile=logfile,mpfit=mpfit,loglevel=loglevel, $
        verbose=verbose,error=error,debug=debug,help=help,restart=restart
  compile_opt hidden,idl2

  on_error,1

  if !version.release lt 6.2 then message,'IDL version <6.2. Cannot continue.'
  error=0 & rname='p3d_gui: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) or n_params() ne 1L then begin
    doc_library,'p3d_gui'
    return
  endif

  ;;========================================------------------------------
  ;; Checking the input arguments:

  ;; Should the current working directory be used to save files?:
  if ~n_elements(path) then begin
    defsysv,'!p3d_data_path',exists=exists
    if exists then inputpath=!p3d_data_path else cd,current=inputpath
  endif else begin
    if ~file_test(path,/directory) then begin
      errmsg='The directory PATH "'+path+'" does not exist, will not continue.'
      goto,error_handler
    endif
    inputpath=path
  endelse

  sepwpath=0
  outputpath=''
  s=size(cwd)
  if s[s[0L]+2L] ne 0L then begin
    sepwpath=1
    if s[s[0L]+2L] eq 1L and s[s[0L]+1L] eq 7L then begin
      if ~file_test(cwd,/directory) then begin
        errmsg='The directory pointed at with CWD ("'+cwd+'") does not exist.'
        goto,error_handler
      endif
      outputpath=cwd
    endif else begin
      cd,current=outputpath
    endelse
    tmp=strpos(outputpath,path_sep(),/reverse_search)
    if tmp ne strlen(outputpath)-1L then outputpath+=path_sep()
  endif ;; s[s[0L]+2L] ne 0L

  if ~file_test(parfile,/regular,/read) then begin
    errmsg='The PARFILE "'+parfile+'" does not exist.'
    goto,error_handler
  endif
  ppath=strmid(parfile,0L,strpos(parfile,path_sep(),/reverse_search)+1L)
  parname='' & parvalue=''
  readcol,parfile,parname,parvalue,format='a,a',delimiter=' ', $
          silent=verbose lt 3,comment=';'

  s=size(logfile)
  if s[s[0L]+2L] gt 1L then begin
    errmsg='LOGFILE must, if set, be a scalar string, of a filename (that ' + $
           'will be overwritten.'
    goto,error_handler
  endif
  if s[s[0L]+2L] eq 1L and s[s[0L]+1L] ne 7L then begin
    errmsg='LOGFILE must, if set, be a scalar string, of a filename (that ' + $
           'will be overwritten.'
    goto,error_handler
  endif

  ;; Opening the log-file:
  logunit=1L
  if s[s[0L]+2L] eq 1L then begin
    openw,logunit,logfile,/get_lun,error=error

    if error ne 0 then begin
      errmsg='Could not open a logfile with the name "'+logfile+'".'
      goto,error_handler
    endif
  endif else logfile=''

  if ~n_elements(loglevel) then loglevel=1L
  s=size(loglevel)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='LOGLEVEL must be a scalar integer; 1<=loglevel<=3.'
    goto,error_handler
  endif
  if loglevel lt 1L or loglevel gt 3L then begin
    errmsg='LOGLEVEL must be a scalar integer; 1<=loglevel<=3.'
    goto,error_handler
  endif

  if ~n_elements(title) then title='p3d data reduction panel'

  if ~n_elements(detector) then detector=0L
  s=size(detector)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='DETECTOR, if set, must be a scalar integer; >=1.'
    goto,error_handler
  endif

  p3d_misc_read_params,parname,parvalue,'ndetector',nd,type='integer', $
        topwid=bbasewid,verbose=verbose,error=error
  if error ne 0 then return
  if ~n_elements(nd) then nd=1L
  detarr=lindgen(nd)+1L
  if detector lt 0L or detector gt nd-1L then begin
    errmsg='1<=DETECTOR<='+strtrim(nd,2L)+ $
        ' [n_elements(DETARR)] must be fulfilled.'
    goto,error_handler
  endif

  if n_elements(userparfile) ne 0L then begin
    s=size(userparfile)
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='USERPARFILE must, if set, be a scalar string pointing at an' + $
             ' existing parameter file.'
      goto,error_handler
    endif
    if ~file_test(userparfile,/regular,/read) then begin
      if keyword_set(cwd) then begin
        userparfile=outputpath+userparfile
        if ~file_test(userparfile,/regular,/read) then begin
          errmsg='Cannot find the USERPARFILE file "'+userparfile+'" (/CWD).'
          goto,error_handler
        endif
      endif else begin
        errmsg='Cannot find the USERPARFILE file "'+userparfile+'".'
        goto,error_handler
      endelse
    endif
  endif else userparfile=''

  usempfit=keyword_set(mpfit)

;;  lampfilename=!p3d_path+'data/instruments/pmas/pmas_lamp_settings.dat'
  track=~n_elements(track)?1L:keyword_set(track)

  displib_active=0L & ldisplib=''
  if n_elements(displib) ne 0L then begin
    ldisplib=displib
    displib_active=1L
  endif

  ;; Setting up colortable-related variables:
  p3d_misc_colortable,colortable,bottom=bottom,index=cindex,indv=indv, $
      colors=colors,/define,verbose=verbose,error=error,debug=debug
  if error ne 0 then return

  ;;===========================================

  leftsize=272L
  rightsize=93L
  if displib_active then begin
    displibset=1L
    dispvalue='Select from library'
  endif else begin
    displibset=0L
    dispvalue = ''
  endelse

  nfil=9L

  ;;========================================------------------------------
  ;; Setting up a widget hierarchy:

  bbasewid=widget_base(title=title,/column,mbar=bbarwid,/base_align_center, $
      tlb_frame_attr=0L,resource_name='p3d')

  wix={bbias:0L,btrcr:0L,bdisp:0L,bflat:0L,bextr:0L,bcwdi:lonarr(3L), $
       bbiao:0L,btrao:0L,bdiso:0L,bflao:0L,bexto:0L,blogt:lonarr(5L), $
       blogf:0L,blogr:0L,bloga:0L,blogo:0L,blogc:0L,blogl:0L,bverb:0L, $
       bdebg:0L,bcolo:0L,bshiv:0L,bdbin:0L, $
       btfoc:0L,bdetp:0L,bdetn:0L, $ ;; be3dv:0L
       bviup:0L,bedup:0L, $
       btabw:0L,bstat:0L,llogi:0L,lstat:0L}

  ;;==============================--------------------
  ;; Setting up a menu bar:

  ;; File:
  file_menu=widget_button(bbarwid,value='File',/menu,uname='With the optio' + $
      'ns of this menu you can create new data files and exit p3d.', $
      tracking_events=track)
  if keyword_set(box) then $
     boxbutton=widget_button(file_menu,value='Create new shift boxes', $
         uvalue='boxes',/align_left,tracking_events=track,accelerator='Ctrl+n')
  wix.bbias=widget_button(file_menu,value='Create new masterbias', $
      uvalue='createmasterbias',uname='Click this button to create a new m' + $
      'asterbias.',tracking_events=track,resource_name='masterbias', $
      accelerator='Ctrl+B')
  wix.btrcr=widget_button(file_menu,value='Create new trace mask', $
      uvalue='createtracemask',uname='Click this button to create a new tr' + $
      'ace mask.',tracking_events=track,resource_name='trace', $
      accelerator='Ctrl+T')
  wix.bdisp=widget_button(file_menu,value='Create new dispersion mask', $
      uvalue='createdispmask',uname='Click this button to create a new dis' + $
      'persion mask.',tracking_events=track,resource_name='createdispmask', $
      accelerator='Ctrl+M',sensitive=0L)
  wix.bflat=widget_button(file_menu,value='Create new fiber flat mask', $
      uvalue='createflatfield',uname='Click this button to create a new fi' + $
      'ber flat mask.',tracking_events=track,resource_name='flat', $
      accelerator='Ctrl+F',sensitive=0L)
  wix.bextr=widget_button(file_menu,value='Extract new object data', $
      uvalue='createobject',uname='Click this button to extract spectra of' + $
      ' a new object.',tracking_events=track,accelerator='Ctrl+O',sensitive=0L)

  if keyword_set(box) then $
     boxbutton=widget_button(file_menu, $
         value='Open existing shift box file', $
         uvalue='boxes',/align_left,tracking_events=track,accelerator='Ctrl+n')
  wix.bbiao=widget_button(file_menu,value='Open existing masterbias', $
      uvalue='browsemasterbias',uname='Click this button to open an existi' + $
      'ng masterbias.',tracking_events=track,resource_name='bias', $
      accelerator='Ctrl+Shift+B',separator=~keyword_set(box))
  wix.btrao=widget_button(file_menu,value='Open existing trace mask', $
      uvalue='browsetracemask',uname='Click this button to open an existin' + $
      'g trace mask.',tracking_events=track,resource_name='trace', $
      accelerator='Ctrl+Shift+T')
  wix.bdiso=widget_button(file_menu,value='Open existing dispersion mask', $
      uvalue='browsedispmask',uname='Click this button to open an existing' + $
      ' dispersion mask.',tracking_events=track,resource_name='dispmask', $
      accelerator='Ctrl+Shift+M')
  wix.bflao=widget_button(file_menu,value='Open existing fiber flat mask', $
      uvalue='browseflatfield',uname='Click this button to open an existin' + $
      'g fiber flat mask.',tracking_events=track,resource_name='flat', $
      accelerator='Ctrl+Shift+F')
  wix.bexto=widget_button(file_menu,value='Open existing extracted object', $
      uvalue='browseobject',uname='Click this button to open an existing '+ $
      'file with extracted data.',tracking_events=track, $
      accelerator='Ctrl+Shift+O')

  bmnewwid=widget_button(file_menu,/separator,value='Restart p3d', $
      uvalue='restart',uname='Click this button to restart p3d and select ' + $
      'another instrument setup.',tracking_events=track)

  bmexiwid=widget_button(file_menu,value='Exit',uvalue='exit', $
      uname='Click this button to exit p3d.',tracking_events=track, $
      accelerator='Ctrl+Q')

  ;; View:
  mviewwid=widget_button(bbarwid,value='View',/menu,uname='With the tools' + $
      ' of this menu you can view both raw and reduced data.', $
      tracking_events=track)
  if keyword_set(box) then $
    wix.bshiv=widget_button(mviewwid,value='Shift inspector', $
        uvalue='shiftinspector',tracking_events=track)

;  wix.be3dv=widget_button(mviewwid,value='E3d',uvalue='trace', $
;      tracking_events=track)
  wix.btfoc=widget_button(mviewwid, $
      value='Evaluate telescope focus series',tracking_events=track)

  wix.blogf=widget_button(mviewwid,/separator,tracking_events=track, $
      value='View a data logfile',uvalue='viewlogfile', $
      uname='Click to view the currently used instrument par.file.')
  wix.blogr=widget_button(mviewwid,tracking_events=track,sensitive=0L, $
      value='View the data reduction log file',uvalue='viewdrlogfile', $
      uname='Click to view the currently open data reduction log file.')
  wix.blogt[0L]=widget_button(mviewwid,/menu,tracking_events=track, $
      sensitive=0L,uvalue='viewdrtimer', $
      value='Set the log view timer [off]', $
      uname='Click to set the update timer for the log file viewer.')
  wix.blogt[1L]=widget_button(wix.blogt[0L],tracking_events=track, $
      uvalue='viewdrtimer_0',value='turn off timer', $
      uname='Click to turn off the timer.')
  wix.blogt[2L]=widget_button(wix.blogt[0L],tracking_events=track, $
      uvalue='viewdrtimer_5',value='5 seconds', $
      uname='Click to set the timer to 5 seconds.')
  wix.blogt[3L]=widget_button(wix.blogt[0L],tracking_events=track, $
      uvalue='viewdrtimer_20',value='20 seconds', $
      uname='Click to set the timer to 20 seconds.')
  wix.blogt[4L]=widget_button(wix.blogt[0L],tracking_events=track, $
      uvalue='viewdrtimer_60',value='60 seconds', $
      uname='Click to set the timer to 60 seconds.')

  wix.bdetp=widget_button(mviewwid,/separator,accelerator='Ctrl+R', $
      value='Select previous detector',uvalue='seldetector_0', $
      uname='Click to select the previous detector element', $
      tracking_events=track,sensitive=nd gt 1L)
  wix.bdetn=widget_button(mviewwid,accelerator='Ctrl+E', $
      value='Select next detector',uvalue='seldetector_1', $
      uname='Click to select the next detector element', $
      tracking_events=track,sensitive=nd gt 1L)

  ;; Color tables:
  bcolmwid=widget_button(mviewwid,/menu,/separator,value='Select colortable',$
      uvalue='colortablemenu',uname='Click to select a colortable', $
      tracking_events=track)

  loadct,get_names=names
  wix.bcolo=widget_button(bcolmwid,value='xloadct',uvalue='colortable', $
      uname='Click to select a colortable using xloadct', $
      tracking_events=track)

  names=['Sauron',names]
  nnames=n_elements(names)
  nbuttons=nnames/12 & mbuttons=nnames mod 12
  bcolwwid=lonarr(nnames)
  imax=nbuttons+(mbuttons gt 0L?1L:0L)
  iimax=lonarr(imax)+11L
  iimax[imax-1L]=mbuttons-1L

  k=0L
  for i=0L,imax-1L do begin
    bcolqwid=widget_button(bcolmwid,/menu,separator=~i,value='colortables '+ $
        strtrim(i+1L,2L)+'/'+strtrim(imax,2L),uvalue='colortablemenu', $
        uname='Click to choose a colortable in this group', $
        tracking_events=track)

    for j=0L,iimax[i] do begin
      bcolwwid[k]=widget_button(bcolqwid,value=names[k], $
          uvalue='colortablesel_'+strtrim(k-1L,2L),uname='Click to select '+ $
          'the '+names[k]+' colortable',tracking_events=track)
      k++
    endfor
  endfor

  ;; Options:
  boptiwid=widget_button(bbarwid,value='Options',/menu,uname='Here you' + $
      ' can view and edit various parameters.',tracking_events=track)

  value=sepwpath?'Using a separate output directory': $
                 'Using the raw data directory'
  wix.bcwdi[0L]=widget_button(boptiwid,/menu,uvalue='selcwd',value=value, $
      uname='Select the target directory of the output data.', $
      tracking_events=track)
  uname='Currently writing to "'+outputpath+'".'
  wix.bcwdi[1L]=widget_button(wix.bcwdi[0L],uvalue='selcwd_0',value='Write' + $
      ' output to the raw data directory',uname=uname,tracking_events=track, $
      sensitive= sepwpath)
  wix.bcwdi[2L]=widget_button(wix.bcwdi[0L],uvalue='selcwd_1',value='Use a' + $
      ' separate directory for the output',uname='Currently writing output' + $
      ' to the (selected) raw data directory.',tracking_events=track, $
      sensitive=~sepwpath)

  if keyword_set(cwd) then begin
    sepwpath=1
    cd,current=outputpath
    tmp=strpos(outputpath,path_sep(),/reverse_search)
    if tmp ne strlen(outputpath)-1L then outputpath+=path_sep()
  endif


  bviipwid=widget_button(boptiwid,uvalue='viewinspar',value='View instrume' + $
      'nt parameter file',uname='Click to view the currently used instrumen'+ $
      't par.file.',tracking_events=track,accelerator='Ctrl+I',/separator)
  sens=userparfile eq ''?0L:1L
  buserwid=widget_button(boptiwid,uvalue='seleuserpar',value='Select a use' + $
      'r parameter file',uname='Click to select a user parameter file.', $
      tracking_events=track,accelerator='Ctrl+O')
  wix.bviup=widget_button(boptiwid,uvalue='viewuserpar',sensitive=sens, $
      value='View the user parameter file',uname='Click to view the curren' + $
      'tly selected user parameter file.',tracking_events=track, $
      accelerator='Ctrl+U')
  wix.bedup=widget_button(boptiwid,uvalue='edituserpar',sensitive=sens, $
      value='Edit the user parameter file',uname='Click to edit the curren' + $
      'tly selected user parameter file.',tracking_events=track, $
      accelerator='Ctrl+A')

  dbin=1L
  tmp=strtrim(dbin,2L)
  tmpstr='[create disp.mask]: data are not rebinned on the disp. axis.'
  wix.bdbin=widget_button(boptiwid,/separator,value='D.mask rebin data [db' + $
      'in='+tmp+']',uvalue='dbin',/menu,uname=tmpstr,tracking_events=track)
  bdbinwid=lonarr(3L)
  lstr=['1 - Data are not rebinned on the dispersion axis.', $
        '2 - Data are rebinned on the disp.axis by a factor 2.', $
        '3 - Data are rebinned on the disp.axis until they fit the screen']
  for i=0L,n_elements(bdbinwid)-1L do begin
    istr=strtrim(i+1L,2L)
    bdbinwid[i]=widget_button(wix.bdbin,value=lstr[i], $
        uvalue='dbin_'+istr,tracking_events=track, $
        uname='[create disp.mask] click to select the dispersion axis rebi' + $
        'nning mode '+istr+'.')
  endfor ;; i=0L,n_elements(bdbinwid)-1L


  tmp=fstat(logunit)
  wix.bloga=widget_button(boptiwid,/separator,value='Append to a log file', $
      uvalue='logopen_1',uname='Click this button to append to a log file.', $
      tracking_events=track,accelerator='Ctrl+J',sensitive=~tmp.open)
  wix.blogo=widget_button(boptiwid,value='Open a log file', $
      uvalue='logopen_0',uname='Click this button to open a log file.', $
      tracking_events=track,accelerator='Ctrl+L',sensitive=~tmp.open)
  wix.blogc=widget_button(boptiwid,value='Close the log file', $
      uvalue='logclose',uname='Click this button to close the log file.', $
      tracking_events=track,accelerator='Ctrl+K',sensitive=tmp.open)
  wix.blogl=widget_button(boptiwid,/menu,uvalue='loglevel', $
      value='Log level ['+strtrim(loglevel,2L)+']', $
      uname='Click to change the message log level',tracking_events=track)
  bloglwid=lonarr(3)
  lstr=['1 - Log some information', $
        '2 - Log more information', $
        '3 - Log  all information']
  for i=0L,n_elements(bloglwid)-1L do begin
    istr=strtrim(i+1L,2L)
    bloglwid[i]=widget_button(wix.blogl,value=lstr[i], $
        uvalue='loglevel_'+istr,tracking_events=track, $
        uname='Click to select the message log level '+istr+'.')
  endfor ;; i=0L,n_elements(bloglwid)-1L

  wix.bverb=widget_button(boptiwid,/menu,uvalue='verbosity', $
      value='Verbose level ['+string(verbose,format='(i1)')+']', $
      tracking_events=track, $
      uname='Click to change the stdout message verbosity level')
  bverbwid=lonarr(4)
  lstr=['0 - Write   no information to stdout', $
        '1 - Write some information to stdout', $
        '2 - Write more information to stdout', $
        '3 - Write  all information to stdout']
  for i=0L,n_elements(bverbwid)-1L do begin
     istr=strtrim(i,2L)
     bverbwid[i]=widget_button(wix.bverb,value=lstr[i], $
         uvalue='verbosity_'+istr,tracking_events=track, $
         uname='Click to select the verbosity level '+istr+'.')
  endfor

  wix.bdebg=widget_button(boptiwid,/separator,uvalue='debugging', $
      value='Debug ['+string(debug,format='(i1)')+' {='+ $
      (debug?'yes':'no')+'}]',tracking_events=track, $
      uname='Click to change the debug mode flag.')

  ;; Help:
  help_menu=widget_button(bbarwid,value='Help',/menu)
  help_info=widget_button(help_menu,value='About',uvalue='about')
  dist_info=widget_button(help_menu,value='Copying',uvalue='copying')


  ;;==============================--------------------
  ;; Setting up the file selector part:

  infostr=['Master bias:','Trace mask:','Disp. mask:','Fiber flat:','Object:']
  wwid=max(strlen(infostr))*!d.x_ch_size+6L
  shifstr=['set trace ref.','set shift ref.']
  shwid=max(strlen(shifstr))*!d.x_ch_size+10L
  twid=46L & swid=!d.x_ch_size+6L
  sstr='This field shows which slot the current data occupies; out of ' + $
       strtrim(nfil,2L) +' available slots.'

  wid={tboxf:0L,lboxi:0L, $
       tbiaf:0L,lbiai:0L,bbicr:0L,bbibr:0L,bbise:0L,bbivi:0L, $
       ttraf:0L,ltrai:0L,btrcr:0L,btrbr:0L,btrse:0L,btrvi:0L,btras:0L, $
       tdisf:0L,ldiai:0L,bdicr:0L,bdibr:0L,bdise:0L,bdivi:lonarr(2L),bdiss:0L,$
       tflaf:0L,lflai:0L,bflcr:0L,bflbr:0L,bflse:0L,bflvi:lonarr(2L), $
       tobjf:0L,lobji:0L,bobcr:0L,bobbr:0L,bobse:0L,bobvi:lonarr(2L), $
       sobjl:0L,dobjl:0L,bdrco:0L,bdrcc:lonarr(4L), $
       bsett:0L,twave:0L,lwave:0L,tcmin:0L,tcmax:0L,bccpe:0L, $
       bviei:0L,dobji:lonarr(nfil), $
       dleft:0L,dright:lonarr(nfil)}
  wid=replicate(wid,nd)

  if nd gt 1L then begin
    wix.btabw=widget_tab(bbasewid,location=3L,uvalue='detectortab', $
        uname='Click to select which detector element to work with.', $
        tracking_events=track)
    bbas2wid=lonarr(nd)
  endif else begin
    bbas2wid=bbasewid
  endelse

  bmnfile=filepath(subdir=['resource','bitmaps'],'new.bmp')
  bmofile=filepath(subdir=['resource','bitmaps'],'open.bmp')
  bmsfile=filepath(subdir=['resource','bitmaps'],'hand.bmp')
  bmrfile=filepath(subdir=['resource','bitmaps'],'demo.bmp')
  bmvfile=filepath(subdir=['resource','bitmaps'],'profile.bmp')

  for ii=0L,nd-1L do begin
    if nd gt 1L then begin
      p3d_misc_read_params,parname,parvalue,'detectit_'+strtrim(ii+1L,2L), $
         /nou,title,topwid=bbasewid,verbose=verbose,error=error
      if error ne 0 then return

      if title eq '-1' then title='Detector '+strtrim(detarr[ii],2L) else begin
        tmp=strsplit(title,'_',/extract)
        title=strjoin(tmp,' ')
      endelse

      bbas2wid[ii]=widget_base(wix.btabw,/column, $
          space=0L,xpad=0L,ypad=0L,title=title)
    endif


    binfowid=widget_base(bbas2wid[ii],/column,/grid_layout,/align_left, $
        space=5L,xpad=0L,ypad=0L)

    ;; Shift box:
    if keyword_set(box) then begin
      boxinfo=widget_base(binfowid,/row,space=5L,xpad=2L,ypad=0L)
      boxlabel=widget_label(boxinfo,value='Shift boxes:',xsize=100,/align_left)
      wid[ii].tboxf=widget_text(boxinfo,value=' ',xsize=twid,uname='This f' + $
          'ield shows the filename of the currently selected box file.', $
          tracking_events=track)
      wid[ii].lboxi=widget_label(boxinfo,value='-',xsize=swid,uname=sstr, $
          tracking_events=track,/sunken_frame)
      boxbrowse=widget_button(boxinfo,value=bmfile,uvalue='browsebox', $
          uname='Click to open an already created shift box file.', $
          tracking_events=track,/bitmap)
      boxselect=widget_button(boxinfo,value=bmsfile,uvalue='selectbox', $
          uname='Click to select an already loaded shift box file.', $
          tracking_events=track,sensitive=0L,/bitmap)
    endif

    ;; Master bias:
    binf3wid=widget_base(binfowid,/row,space=5L,xpad=2L,ypad=0L)
    lbiaswid=widget_label(binf3wid,value=infostr[0L],xsize=wwid,/align_left)
    wid[ii].tbiaf=widget_text(binf3wid,value=' ',xsize=twid,uname='This fi' + $
        'eld shows the filename of the currently selected master bias file.', $
        tracking_events=track,resource_name='field')
    wid[ii].lbiai=widget_label(binf3wid,value='-',xsize=swid,uname=sstr, $
        tracking_events=track,/sunken_frame)
    wid[ii].bbicr=widget_button(binf3wid,value=bmnfile, $
        uvalue='createmasterbias',uname='Click this button to create a new' + $
        ' master bias.',tracking_events=track,/bitmap)
    wid[ii].bbibr=widget_button(binf3wid,value=bmofile, $
        uvalue='browsemasterbias',uname='Click this button to open an exis' + $
        'ting master bias.',tracking_events=track,/bitmap)
    wid[ii].bbise=widget_button(binf3wid,value=bmsfile, $
        uvalue='selectmasterbias',uname='Click to select between already l' + $
        'oaded master bias files.',tracking_events=track,sensitive=0L,/bitmap)
    wid[ii].bbivi=widget_button(binf3wid,value=bmrfile,uvalue='viewbias', $
        uname='Click to view the bias raw data, and the result.', $
        tracking_events=track,sensitive=0L,/bitmap)


    ;; Trace:
    binf2wid=widget_base(binfowid,/row,space=5L,xpad=2L,ypad=0L)
    ltracwid=widget_label(binf2wid,value=infostr[1L],xsize=wwid,/align_left)
    wid[ii].ttraf=widget_text(binf2wid,value='',xsize=twid,uname='This fie' + $
        'ld shows the filename of the currently selected trace mask file.', $
        tracking_events=track,resource_name='field')
    wid[ii].ltrai=widget_label(binf2wid,value='-',xsize=swid,uname=sstr, $
        tracking_events=track,/sunken_frame)
    wid[ii].btrcr=widget_button(binf2wid,value=bmnfile,/bitmap, $
        uvalue='createtracemask',uname='Click this button to create a new ' + $
        'trace mask.',tracking_events=track)
    wid[ii].btrbr=widget_button(binf2wid,value=bmofile,/bitmap, $
        uvalue='browsetracemask',uname='Click this button to open an exist' + $
        'ing trace mask.',tracking_events=track)
    wid[ii].btrse=widget_button(binf2wid,value=bmsfile,/bitmap, $
        uvalue='selecttracemask',uname='Click to select between already lo' + $
        'aded trace masks.',tracking_events=track,sensitive=0L)
    wid[ii].btrvi=widget_button(binf2wid,value=bmrfile,uvalue='viewtrace', $
        uname='Click to view the trace raw data, and the overplotted trace' + $
        ' mask.',tracking_events=track,sensitive=0L,/bitmap)
    if keyword_set(box) then begin
      wid[ii].btras=widget_button(binf2wid,value=shifstr[0L], $
          uvalue='trcshift',xsize=shwid,sensitive=0L)
    endif

    ;; Dispersion mask:
    binf4wid=widget_base(binfowid,/row,space=5L,xpad=2L,ypad=0L)
    ldispwid=widget_label(binf4wid,value=infostr[2L],xsize=wwid,/align_left)
    wid[ii].tdisf=widget_text(binf4wid,value=dispvalue,xsize=twid, $
        uname='This field shows the filename of the currently selected dis' + $
        'persion mask file.',tracking_events=track,resource_name='field')
    wid[ii].ldiai=widget_label(binf4wid,value='-',xsize=swid,uname=sstr, $
        tracking_events=track,/sunken_frame)
    wid[ii].bdicr=widget_button(binf4wid,value=bmnfile,/bitmap, $
        uvalue='createdispmask',tracking_events=track,sensitive=0L, $
        uname='Click this button to create a new dispersion mask.')
    wid[ii].bdibr=widget_button(binf4wid,value=bmofile,/bitmap, $
        uvalue='browsedispmask',uname='Click this button to open an existi' + $
        'ng dispersion mask.',tracking_events=track)
    wid[ii].bdise=widget_button(binf4wid,value=bmsfile,/bitmap, $
        uvalue='selectdispmask',uname='Click to select between already loa' + $
        'ded dispersion masks.',tracking_events=track,sensitive=0L)
    wid[ii].bdivi[0L]=widget_button(binf4wid,value=bmrfile,/bitmap, $
        uvalue='viewdispmask_0',uname='Click to view the raw data, and the' + $
        ' overlayed trace mask.',tracking_events=track,sensitive=0L)
    wid[ii].bdivi[1L]=widget_button(binf4wid,value=bmvfile,/bitmap, $
        uvalue='viewdispmask_1',tracking_events=track,sensitive=0L, $
        uname='Click to view the extracted spectrum data.')
 
    if keyword_set(box) then begin
      wid[ii].bdiss=widget_button(binf4wid,value=shifstr[1L], $
          uvalue='dispshift',xsize=shwid,sensitive=0L)
    endif
    if displib_active then begin
      displibbase=widget_base(binf4wid,/row,/nonexclusive)
      displibbut=widget_button(displibbase,value=' ',uvalue='displib')
    endif

    ;; Flat field:
    binf4wid=widget_base(binfowid,/row,space=5L,xpad=2L,ypad=0L)
    lflatwid=widget_label(binf4wid,value=infostr[3L],xsize=wwid,/align_left)
    wid[ii].tflaf=widget_text(binf4wid,value='',xsize=twid,uname='This fie' + $
        'ld shows the filename of the currently selected flat field file.', $
        tracking_events=track,resource_name='field')
    wid[ii].lflai=widget_label(binf4wid,value='-',xsize=swid,uname=sstr, $
        tracking_events=track,/sunken_frame)
    wid[ii].bflcr=widget_button(binf4wid,value=bmnfile,/bitmap, $
        uvalue='createflatfield',uname='Click this button to create a new ' + $
        'fiber flat mask.',tracking_events=track,sensitive=0L)
    wid[ii].bflbr=widget_button(binf4wid,value=bmofile,/bitmap, $
        uvalue='browseflatfield',uname='Click this button to open an exist' + $
        'ing fiber flat mask.',tracking_events=track)
    wid[ii].bflse=widget_button(binf4wid,value=bmsfile,/bitmap, $
        uvalue='selectflatfield',uname='Click to select between already lo' + $
        'aded flat field masks.',tracking_events=track,sensitive=0L)
    wid[ii].bflvi[0L]=widget_button(binf4wid,value=bmrfile,/bitmap, $
        uvalue='viewflatfield_0',tracking_events=track,sensitive=0L, $
        uname='Click to view the raw data, and the overlayed trace mask.')
    wid[ii].bflvi[1L]=widget_button(binf4wid,value=bmvfile,/bitmap, $
        uvalue='viewflatfield_1',tracking_events=track,sensitive=0L, $
        uname='Click to view the extracted spectrum data.')

    ;; Object:
    binf5wid=widget_base(binfowid,/row,space=5L,xpad=2L,ypad=0L)
    lobjewid=widget_label(binf5wid,value=infostr[4L],xsize=wwid,/align_left)
    wid[ii].tobjf=widget_text(binf5wid,value='',xsize=twid,uname='This fie' + $
        'ld shows the filename of the currently selected object data file.', $
        tracking_events=track,resource_name='field')
    wid[ii].lobji=widget_label(binf5wid,value='-',xsize=swid,uname=sstr, $
        tracking_events=track,/sunken_frame)
    wid[ii].bobcr=widget_button(binf5wid,value=bmnfile,/bitmap, $
        uvalue='createobject',tracking_events=track,sensitive=0L, $
        uname='Click this button to extract spectra of a new object.')
    wid[ii].bobbr=widget_button(binf5wid,value=bmofile,/bitmap, $
        uvalue='browseobject',tracking_events=track,uname='Click this butt' + $
        'on to open an existing file with extracted data.')
    wid[ii].bobse=widget_button(binf5wid,value=bmsfile,/bitmap, $
        uvalue='selectobject',uname='Click to select between already loade' + $
        'd object files.',tracking_events=track,sensitive=0L)
    wid[ii].bobvi[0L]=widget_button(binf5wid,value=bmrfile,/bitmap, $
        uvalue='viewobject_0',tracking_events=track,sensitive=0L, $
        uname='Click to view the raw data, and the overlayed trace mask.')
    wid[ii].bobvi[1L]=widget_button(binf5wid,value=bmvfile,/bitmap, $
        uvalue='viewobject_1',tracking_events=track,sensitive=0L, $
        uname='Click to view the extracted spectrum data.')

    ;;==============================--------------------
    ;; Setting up the object view panels:

    bviewwid=widget_base(bbas2wid[ii],column=2L,space=5L,xpad=0L,ypad=0L, $
        /grid_layout)

    ;; Left panel:
    bvielwid=widget_base(bviewwid,/column,/frame,/base_align_center, $
        space=2L,xpad=2L,ypad=2L,resource_name='field3')
    llablwid=widget_label(bvielwid,value='Display of extracted objects')

    bleftwid=widget_base(bvielwid,/row,/base_align_center, $
                         space=0L,xpad=0L,ypad=0L)
    wid[ii].sobjl=widget_slider(bleftwid,/suppress_value,/vertical, $
        ysize=leftsize+4L,sensitive=0L,uvalue='wavebar', $
        uname='Use this slider to quickly chan' + $
        'ge the shown wavelength bin.',tracking_events=track,minimum=0L,/drag)
    wid[ii].dobjl=widget_draw(bleftwid,xsize=leftsize,ysize=leftsize, $
        retain=2L,/frame,uvalue='leftdraw',uname='No data has been loaded.', $
        tracking_events=track,/button_events)
    ;; Setting up a context menu for this draw widget:
    wid[ii].bdrco=widget_base(wid[ii].dobjl,/context_menu)
    wid[ii].bdrcc[0L]=widget_button(wid[ii].bdrco, $
        value='Set: North [right], East [up]',uvalue='orient_0', $
        uname='Click this button to set the orientation: right-North, up-E' + $
        'ast',tracking_events=track)
    wid[ii].bdrcc[1L]=widget_button(wid[ii].bdrco, $
        value='Set: North [up]   , East [left]',uvalue='orient_1', $
        uname='click this button to set the orientation: up-North, left-Ea' + $
        'st',tracking_events=track)
    wid[ii].bdrcc[2L]=widget_button(wid[ii].bdrco, $
        value='Set: North [left] , East [down]',uvalue='orient_2', $
        uname='Click this button to set the orientation: left-North, down-' + $
        'east',tracking_events=track)
    wid[ii].bdrcc[3L]=widget_button(wid[ii].bdrco, $
        value='Set: North [down] , East [right]',uvalue='orient_3', $
        uname='Click this button to set the orientation: down-North, right' + $
        '-East',tracking_events=track)


    lxsize=9L*!d.x_ch_size
    wid[ii].bsett=widget_base(bvielwid,/row,/base_align_center, $
        space=0L,xpad=0L,ypad=0L,sensitive=0L)
    wid[ii].twave=widget_text(wid[ii].bsett,value='',uvalue='wave', $
        /editable,xsize=5L,uname='This is the value of the displayed wavel' + $
        'ength bin.',tracking_events=track,resource_name='field', $
        /kbrd_focus_events)
    wid[ii].lwave=widget_label(wid[ii].bsett,value=' ',xsize=lxsize, $
        uname='This field shows the wavelength of the currently selected w' + $
        'avelength bin.',tracking_events=track,/sunken_frame)
    wid[ii].tcmin=widget_text(wid[ii].bsett,value='',uvalue='colminmax_0', $
        /editable,xsize=7L,uname='This is the minimum value that is used i' + $
        'n the color scaling.',tracking_events=track,resource_name='field', $
        /kbrd_focus_events)
    wid[ii].tcmax=widget_text(wid[ii].bsett,value='',uvalue='colminmax_1', $
        /editable,xsize=7L,uname='This is the maximum value that is used i' + $
        'n the color scaling.',tracking_events=track,resource_name='field', $
        /kbrd_focus_events)
    droplistvalue=['100%','97%','95%','90%','80%']
    wid[ii].bccpe=widget_droplist(wid[ii].bsett,value=droplistvalue, $
        uvalue='minmaxlimits',uname='Selects the range of values which are' + $
        ' used when determining the minimum and maximum to plot.', $
        tracking_events=track)

    ;; Right panel:
    bvierwid=widget_base(bviewwid,/column,/frame,/base_align_center, $
        /align_top,space=2L,xpad=4L,ypad=2L,resource_name='field3')
    llabrwid=widget_label(bvierwid,value='Comparison images')

    wid[ii].bviei=widget_base(bvierwid,row=3L,/base_align_center, $
        space=5L,xpad=0L,ypad=0L,sensitive=0L)
    for k=0L,nfil-1L do begin
      kstr=strtrim(k,2L) & uvalue='right_'+kstr
      wid[ii].dobji[k]=widget_draw(wid[ii].bviei, $
          xsize=rightsize,ysize=rightsize,/frame, $
          retain=2L,/button_events,uvalue=uvalue,uname='[object slot '+kstr+ $
          '/'+strtrim(nfil,2L)+'] no data has been loaded.', $
          tracking_events=track)
    endfor

  endfor ;; ii=0L,nd-1L

  wix.bstat=widget_base(bbasewid,/row,/base_align_left, $
      space=5L,xpad=0L,ypad=0L)

  tmp=fstat(logunit) & value=tmp.open?'Log: yes':'Log: no '
  uname=tmp.open?'Operations are logged to "'+ $
      p3d_misc_pathify(logfile,/dpath)+'".':'This field shows ' + $
      'if the reduction process is being logged.'
  wix.llogi=widget_label(wix.bstat,value=value,uname=uname, $
      tracking_events=track,resource_name='field4')
  wix.lstat=widget_label(wix.bstat,/sunken_frame,value='Welcome to p3d!', $
      uname='This line shows information on what is going on in this tool.', $
      tracking_events=track,resource_name='field')

  ;;==============================--------------------
  ;; Realizing the application and sensitizing various widgets:

  widget_control,bbasewid,update=0L
  widget_control,bbasewid,/realize
; widget_control,btraswid,sensitive=0L

  for ii=0L,nd-1L do begin
    widget_control,wid[ii].tdisf,sensitive=1L-displibset
    widget_control,wid[ii].bdise,sensitive=0L
    widget_control,wid[ii].bdibr,sensitive=1L-displibset
    widget_control,wid[ii].dobjl,get_value=index
    wid[ii].dleft=index
    widget_control,wid[ii].bccpe,set_droplist_select=2L ;; default is 95%
  endfor

  if displib_active then widget_control,displibbut,set_button=displibset
;  widget_control,bdisswid,sensitive=(1L-displibset)*0;(curdisp_nr ge 1)
  if keyword_set(box) then widget_control,wix.bshiv,sensitive=0L
  widget_control,wix.bdisp,sensitive=displibset
;  widget_control,wix.be3dv,sensitive=0L
  widget_control,wix.btfoc,sensitive=0L
  if nd gt 1L then widget_control,wix.btabw,set_tab_current=detector

  ;; setting the horizontal size of the status label widget:
  tmp =widget_info(bbasewid,/geometry)
  tmp2=widget_info(wix.bstat,/geometry)
  tmp3=widget_info(wix.llogi,/geometry)
  tmpval=long(tmp.xsize-4*tmp.xoffset-tmp3.xsize-tmp2.space)
  widget_control,wix.lstat,xsize=tmpval

  tmp=fstat(logunit)
  if tmp.open then widget_control,wix.blogr,sensitive=1L
  if tmp.open then widget_control,wix.blogt[0L],sensitive=1L

  widget_control,bbasewid,update=1L

  ;;==============================--------------------
  ;; Reading instrument specific data:

  p3d_misc_read_params,parname,parvalue,'daxis',daxis,/must_exist, $
      type='integer',topwid=bbasewid,verbose=verbose,error=error
  if error ne 0 then goto,error_handler2

  p3d_misc_read_params,parname,parvalue,'nblocks',nblocks, $
      type='integer',topwid=bbasewid,verbose=verbose,error=error
  if error ne 0 then goto,error_handler2
  if ~n_elements(nblocks) then nblocks=1L
  block=nblocks gt 1L?1L:0L ;; defaulting to the 'b' block, for vimos

  ndat=nd;*nblocks ;; Commented 10.10.2009 /CS

  ;; Retrieving the name(s) of the keywords file:
  p3d_misc_read_params,parname,parvalue,'keywordsfile',/must_exist, $
      lkwrd_list,topwid=bbasewid,verbose=verbose,error=error
  if error ne 0 then goto,error_handler2
  lkwrd_list=ppath+lkwrd_list
  if ~file_test(lkwrd_list,/regular,/read) then begin
    errmsg='The keywords file "'+lkwrd_list+'" does not exist.'
    goto,error_handler
  endif

  ;; Retrieving the name(s) of the position tables:
  posfile=strarr(ndat)
  for i=0L,ndat-1L do begin
    str=ndat eq 1L?'':'_'+strtrim(i+1L,2L)
    p3d_misc_read_params,parname,parvalue,'postable'+str,/must_exist,/nou, $
        postable,topwid=bbasewid,verbose=verbose,error=error
    if error ne 0 then goto,error_handler2
    posfile[i]=postable
  endfor
  posfile=ppath+posfile ;; adding the path of the parameter file

  ;; Reading the position table(s):
  pos=ptrarr(n_elements(posfile))
  for i=0L,n_elements(posfile)-1L do begin
    p3d_misc_read_postable,postable=posfile[i],rownum=rownum,id=id, $
        xpos=xpos,ypos=ypos,lens_size=lenssize,shape=shape, $
        nvertices=nvertices,/science, $
        verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,error_handler2
    if ~n_elements(lenssize) then lenssize=1.0 ;; temporary!!

    pos[i]=ptr_new({file:posfile[i],shape:shape,n:nvertices, $
                    fpos:rownum,fid:id,fpx:xpos,fpy:ypos,fles:lenssize})
  endfor

  ;;==============================--------------------
  ;; Saving information about the widget in a structure:

   tmp=strarr(nd,nfil)
  ftmp=fltarr(nd,nfil)
  itmp=lonarr(nd,nfil)

  for ii=0L,nd-1L do begin
    for k=0L,nfil-1L do begin
      widget_control,wid[ii].dobji[k],get_value=index
      wid[ii].dright[k]=index
    endfor
  endfor

  tmp=strarr(nd,9L,nfil)
  cur=lonarr(nd,9L)-1L
  stype=['Master bias','Trace','Disp.mask','Flat field','Object','Box']
  crvd=fltarr(nd,2,9L)
  orientation=lonarr(nd,nfil)+1L

  state={nbi:0L,ntr:1L,ndi:2L,nfl:3L,nob:4L,nbx:5L,ntsh:6L,ndsh:7L,nosh:8L, $
         n:nfil,cur:cur,curb:lonarr(nd),crvd:crvd,usempfit:usempfit, $
         file:tmp,path:tmp,stype:stype, $
         cmin:ftmp,cmax:ftmp,slid:itmp+3L,slidp:ftmp+95.0, $
         ccmin:fltarr(nd),ccmax:fltarr(nd),column:itmp, $
         curdata:ptrarr(nd,9L),data:ptrarr(nd,6L,nfil), $
         hdr:ptrarr(nd,6L,nfil),hhdr:ptr_new(nd), $
         lkwrdlist:lkwrd_list, $
         pos:pos,nblocks:nblocks,block:block, $
         leftsize:leftsize,rightsize:rightsize,orientation:orientation, $
         skyalign:'',docosmicsub:0L,displibset:displibset,displib:ldisplib, $
         dotraceshif:0L,dodispshift:0L, $
         pisreadybias:0L,pisreadyflat:0L,pisreadydisp:0L, $
         pisreadybox:0L,scalebinsseparately:0L, $
         parfile:parfile,userparfile:userparfile, $
         parname:ptr_new(parname,/no_copy), $
         parvalue:ptr_new(parvalue,/no_copy), $
         daxis:daxis,track:track,wid:wid,wix:wix,colors:colors, $
         nd:nd,d:detector,darr:detarr,logfile:logfile,logfilenrows:0L, $
         logfileusetimer:0L,logfiletimer:2.0,dbin:dbin, $
         logfileoffset:0L,logfiletext:ptr_new(''),logfiletid:0L,xlogfile:'', $
         nf_cancel:0L,sf_cancel:0L,logunit:[logunit,loglevel],compress:0L, $
         xxsfx:'',crsfx:'',sfx:'.fits',icsfx:'_imcmb',mbsfx:'_mbias', $
         trsfx:'_trsfx',dmsfx:'_dmask',ffsfx:'_flatf',obsfx:'_oextr', $
         inputpath:inputpath,sepwpath:sepwpath,outputpath:outputpath, $
         cbottom:bottom,cindex:cindex,cindv:indv,colortable:colortable, $
         verbose:verbose,top:bbasewid,debug:debug,rname:rname}

  ;; Storing the state information in the widget tree:
  pstate=ptr_new(state,/no_copy)
  widget_control,bbasewid,set_uvalue=pstate

  tmp=fstat(logunit)
  if tmp.open then p3d_gui_openedlog,pstate

  if keyword_set(restart) then begin
    ret=dialog_message('p3d has been restarted.',/center,/information, $
            dialog_parent=bbasewid,title='User information')
  endif

  ;;==============================--------------------
  ;; Registering the widget application with xmanager:

  xmanager,'p3d_gui',bbasewid,/no_block, $
           cleanup='p3d_gui_cleanup'

  return

error_handler:
  message,errmsg,/informational

error_handler2:
  error=-1

  ;; Closing the widget, if it is open:
  if n_elements(bbasewid) eq 1L then begin
    if widget_info(bbasewid,/valid_id) then $
      widget_control,bbasewid,/destroy
  endif

  ;; Closing the log file (if it is open):
  tmp=fstat(logunit)
  if tmp.open then free_lun,logunit

  return
END ;;; procedure: p3d_gui
