;+
; NAME:
;         p3d_misc_extract_method
;
;         $Id: p3d_misc_extract_method.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         This routine determines which spectrum extraction method is to be
;         used. If no user parameter file is specified then the default value
;         'tophat' is returned. If a user parameter file is specified (as the
;         first argument) then it is searched for the parameter name
;         KEYWORD ['methodextract'] and thereafter reads its value. Allowed
;         values of KEYWORD are 'tophat' and 'optimal'.
;
;         In the creation of a trace mask it might be desirable to calculate
;         cross-dispersion line profiles even though the extraction method is
;         'tophat'. If this is the case then use the keyword /TRACEMASK
;         (see below).
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_extract_method,userparfile,tracemask=,keyword=, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; OPTIONAL INPUTS:
;         userparfile     - A scalar string specifying the name of an optional
;                           user parameter file, that may contain the keyword
;                           KEYWORD (the default of KEYWORD is
;                           'methodextract'). This keyword may, in turn, be set
;                           to one of the two following options:
;                             'tophat' :: default standard spectrum extraction
;                             'optimal':: to use the optimal
;                                         extraction method of Horne (1986).
;
; KEYWORD PARAMETERS:
;         keyword ['methodextract']
;                         - USERPARFILE is scanned for this keyword.
;         tracemask       - If set, then USERPARFILE is scanned for the entry
;                           'trace_calcprofs', that can be set to either 'yes'
;                           or 'no'. If it is 'yes' then method is set to
;                           optimal. The idea with this option is to allow
;                           cross-dispersion line profiles to be calculated
;                           even though the method of extraction is set to
;                           'tophat'.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         17.07.2009 - Created routine. /CS
;-
FUNCTION p3d_misc_extract_method,userparfile,tracemask=tracemask, $
             keyword=keyword,topwid=topwid,logunit=logunit,verbose=verbose, $
             error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_misc_extract_method: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_misc_extract_method'
    return,0
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return,error
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  useuserpar=0
  if n_elements(userparfile) ne 0L then begin
    s=size(userparfile)
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='USERPARFILE, if set, must be a scalar string.'
      goto,error_handler
    endif
    if userparfile ne '' then begin
      if ~file_test(userparfile,/regular,/read) then begin
        errmsg='The file USERPARFILE "'+userparfile+'" does not exist.'
        goto,error_handler
      endif
      useuserpar=1
    endif
  endif ;; n_elements(userparfile) ne 0L

  if ~n_elements(keyword) then keyword='methodextract'
  s=size(keyword)
  if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
    errmsg='KEYWORD must be set to a scalar string.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; The default spectrum extraction method is 'tophat':

  method='tophat'

  ;; Checking the user parameter file for the extraction method:
  if useuserpar then begin
    uparname='' & uparvalue=''
    readcol,userparfile,uparname,uparvalue,format='a,a', $
            silent=verbose lt 3,delimiter=' ',comment=';'
    if uparname[0L] ne '' then begin

      p3d_misc_read_params,uparname,uparvalue,keyword,methodextract,/upo, $
          found=found2,topwid=topwid,verbose=verbose,error=error
      if error ne 0 then return,error

      if found2 then begin
        if strlowcase(methodextract) ne 'tophat' and $
           strlowcase(methodextract) ne 'optimal' then begin
          errmsg='[user parameter file] The spectrum extraction meth' + $
                 'od must be either ''average'' or ''median''; not ' + $
                 ''''+methodextract+'''.'
          goto,error_handler
        endif
        method=methodextract
      endif ;; found2

      ;; Creating a trace mask there is a special option to calculate cross-
      ;; dispersion line profiles even though the method of extraction is not
      ;; optimal:
      if keyword_set(tracemask) then begin
        p3d_misc_read_params,uparname,uparvalue,'trace_calcprofs',/upo, $
            profiles__,found=found,topwid=topwid,verbose=verbose,error=error
        if error ne 0 then return,error

        if found then begin
          if strlowcase(profiles__) ne 'yes' and $
             strlowcase(profiles__) ne 'no' then begin
            errmsg='[user parameter file] The option to calculate cros' + $
                   's-dispersion line profiles when calculating a trac' + $
                   'e mask must be either ''yes'' or ''no''; not '''+ $
                   profiles__+'''.'
            goto,error_handler
          endif
          if strlowcase(profiles__) eq 'yes' then method='optimal'
        endif ;; found
      endif ;; keyword_set(trace)

    endif ;; uparname[0L] ne ''
  endif ;; userparfile

  return,method

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return,error
END ;;; function: p3d_misc_extract_method
