;+
; NAME:
;         p3d_misc_imcombine
;
;         $Id: p3d_misc_imcombine.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         This routine combines input data in two senses:
;
;         1) A combined image is created by taking the average [default],
;          median, or min/max-average of the available values for every pixel,
;          using three or more input files. The method can only be set using
;          the user parameter file (see below on how to do this). If
;          METHOD=='minmax' is used, then the minimum and maximum values of
;          every pixel are thrown away; the header keyword NCOMBINE is
;          decremented by 2 to reflect this filtering.
;           'average':: The average is calculated for every pixel through
;                       all images. 
;           'median' :: The median is calculated for every pixel through
;                       all images. 
;           ['minmax']::For every additional number of 10 input images
;                       the lowest and highest pixel values are thrown
;                            away. This method is the default. 
;
;         2) If the input array of filenames has a second dimension then the
;           data is combined across that second dimension in order to create a
;           larger final output image. This way data is handled which has been
;           read out in different blocks from the CCD. The data is combined
;           using information in the DETSEC input array. No checking is done
;           here if data of different blocks belong to each other.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_imcombine,filename,out,detsec=,daxis=,/spatialflip, $
;             userparfile=,stawid=,topwid=,logunit=,verbose=, $
;             error=,/debug,/help
;
; INPUTS:
;         filename        - A one- or two-dimensional array of filenames that
;                           are to be combined. If FILENAME is an array then
;                           all files are combined as is. If FILENAME is a two-
;                           dimensional array then the data is first combined
;                           across the first dimension. Thereafter it is
;                           combined to create a larger image across the second
;                           dimension. It is a requirement to specify DETSEC if
;                           FILENAME is specified as a two-dimensional array.
;
; KEYWORD PARAMETERS:
;         detsec          - A four-element (columns) -by- number of blocks
;                           (rows) integer array that specifies the detector
;                           region to use on the CCD for each block. For each
;                           row the first two elements are used with the x-axis,
;                           and the second two elements with the y-axis. DBIAS,
;                           GAIN, and RDNOISE must have as many elements as
;                           DETSEC has rows. DETSEC is only used if FILENAME is
;                           a two-dimensional array.
;         daxis [1]       - Defines the dimension of the dispersion direction
;                           in the spectrum image; must be either 1 or 2.
;         spatialflip [0] - If this keyword is set then the combined data
;                           (when there are several blocks of data) is flipped
;                           in the spatial direction before the data is
;                           returned.
;         userparfile     - A scalar string with the name of a file with user-
;                           defined parameters. Here the parameters
;                           'methodcombine' and 'slowimcombine' are used.
;                           methodcombine can be set to:
;                             'average':: using an average.
;                             'median'::  using a median.
;                             'minmax':: using a min/max-filtered average.
;                           slowimcombine can be either 'yes' or 'no'. If
;                             slowimcombine=='yes' then the minimum and
;                             minmax-array is calculated by looping, instead of
;                             using the IDL-intrinsic min().
;         stawid          - If set to a valid ID then a log message is written
;                           using this ID for relevant actions.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - The combined output array. If FILENAME has two
;                           dimensions then OUT is larger than the input DATA.
;                           The output size then depends on DETSEC.
;         nval            - A scalar integer with the final number of values
;                           used in the combination.
;         rstr            - A scalar string with information on which image
;                           combination method was used.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         18.11.2008 - The first version of this routine was written by
;                      Matthias Reis. /Matthias Reis
;         19.11.2008 - Added the AVERAGE option. /CS
;-
PRO p3d_misc_imcombine,filename,out,nval,rstr,detsec=detsec, $
        daxis=daxis,spatialflip=spatialflip,userparfile=userparfile, $
        stawid=stawid,topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_misc_imcombine: '
  if ~n_elements(verbose) then verbose=0
  usestawid=~n_elements(stawid)?0L:widget_info(stawid,/valid_id)
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_misc_imcombine'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  nval=1L
  rstr='n/a'

  if ~n_elements(daxis) then daxis=1L
  sd=size(daxis)
  if sd[sd[0L]+2L] ne 1L or $
    (sd[sd[0L]+1L] ge 4L and sd[sd[0L]+1L] le 11L) then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  sid=daxis?2L:1L

  s=size(filename) & n=s[1L] & nblocks=s[0L] le 1L?1L:s[2L]
  if n ne 1L and n lt 3L or s[s[0L]+1L] ne 7L then begin
    errmsg='FILENAME [1] must be set; to a string array with at least 3 na' + $
           'mes of existing files.'
    goto,error_handler
  endif

  for i=0L,n-1L do begin
    for j=0L,nblocks-1L do begin
      if ~file_test(filename[i,j],/regular,/read) then begin
        errmsg='FILENAME [1] {'+strtrim(i,2L)+'}, the input file with the ' + $
             'NAme "'+filename[i,j]+'" does not exist.'
        goto,error_handler
      endif
    endfor
  endfor

  ;; Nothing to do here if the number of files is 1 and there is only 1 block:
  if n eq 1L and nblocks eq 1L then begin
    out=readfits(filename[0L],h,silent=verbose lt 3,/no_unsigned)
    nval=1L
    return
  endif

  if nblocks gt 1L then begin
    s=size(detsec)
    if ~s[s[0L]+2L] then begin
      errmsg='DETSEC must be specified when NBLOCKS>1, it is not.'
      goto,error_handler
    endif
    if s[1L] ne 4L or s[2L] ne nblocks or $
      (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
      errmsg='DETSEC must be set to a two-dimensional array with 4,NBLOCKS='+ $
             strtrim(nblocks,2L)+' columns,rows (not '+strtrim(s[1L],2L)+','+ $
             strtrim(s[2L,2L])+').'
      goto,error_handler
    endif
  endif ;; nblocks gt 1L

  useopath=0L
  if n_elements(opath_) ne 0L then begin
    if ~file_test(opath_,/directory) then begin
      errmsg='The output directory OPATH does not exist.'
      goto,error_handler
    endif
    opath=opath_
    tmp=strpos(opath,path_sep(),/reverse_search)
    if tmp ne strlen(opath)-1L then opath+=path_sep()
    useopath=1L
  endif

  if ~n_elements(spatialflip) then spatialflip=0L
  sd=size(spatialflip)
  if sd[sd[0L]+1L] ge 4L and sd[sd[0L]+1L] le 11L then begin
    errmsg='SPATIALFLIP must be of (any) integer type; SPATIALFLIP=0||1.'
    goto,error_handler
  endif
  if spatialflip lt 0L or spatialflip gt 1L then begin
    errmsg='SPATIALFLIP must be of (any) integer type; SPATIALFLIP=0||1.'
    goto,error_handler
  endif

  useuserpar=0
  if n_elements(userparfile) ne 0L then begin
    s=size(userparfile)
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='USERPARFILE, if set, must be a scalar string.'
      goto,error_handler
    endif
    if userparfile ne '' then begin
      if ~file_test(userparfile,/regular,/read) then begin
        errmsg='The file USERPARFILE "'+userparfile+'" does not exist.'
        goto,error_handler
      endif
      useuserpar=1
    endif
  endif ;; n_elements(userparfile) ne 0L

  ;;========================================------------------------------
  ;; Only combining images if there are more than 1 file in the first
  ;; dimension of FILENAMES:

  if n ge 3L then begin

    ;;========================================------------------------------
    ;; Reading the method parameter from the USERPARFILE, if it is defined:

    method='minmax' & slowmode=1L
    if useuserpar then begin
      readcol,userparfile,parname,parvalue,format='a,a',comment=';', $
              silent=verbose lt 3,delimiter=' '
      if n_elements(parname) ne 0L then begin
        idx=strpos(strlowcase(parname),'methodimcombine')
        tmp=where(idx ge 0L,count)

        if count gt 1L then begin
          errmsg='[user parameter file] There are several entries of ''met' + $
                 'hodimcombine''. There can only be one!'
          goto,error_handler
        endif else if count eq 1L then begin
          method=strtrim(parvalue[tmp[0L]],2L)

          switch method of
            'average':
            'median':
            'minmax': break
            else: begin
              errmsg='[user parameter file] methodimcombine must be either' + $
                     ' ''average'' or ''median''; '''+ method+''' is not a' + $
                     'llowed.'
              goto,error_handler
            end
          endswitch ;; method
        endif ;; count eq 1L

        idx=strpos(strlowcase(parname),'slowimcombine')
        tmp=where(idx ge 0L,count)
        if count ge 1L then begin
          slowmode=strtrim(parvalue[tmp[0L]],2L)
          slowmode=strlowcase(slowmode) eq 'yes'?1L:0L
        endif

      endif ;; n_elements(parname) ne 0L
    endif ;; useuserpar

    ;;========================================------------------------------
    ;; Writing information to the status line:

    case method of
      'average': tmp='n average'
      'median': tmp=' median'
      'minmax': tmp='n average min/max-filtered'
    endcase

    msg='Creating a'+tmp+' data image from '+strtrim(n,2L)+' input images.'
    if nblocks gt 1L then begin
      msg=[msg,'Combining data of '+strtrim(nblocks,2L)+' offset blocks.']
      if spatialflip then msg=[msg,'Flipping the combined image in the spa' + $
                               'tial direction.']
    endif

    if slowmode then msg=[msg,'Looping over all pixels (this is slow!).']

    nval=n
    if logunit[1L] ge 2L then begin
      nslen=strtrim(strlen(nval),2L)
      for i=0L,n-1L do $
         msg=[msg,' Input file '+string(format='(i'+nslen+')',i+1L)+'/'+ $
              strtrim(nval,2L)+': '+file_basename(filename[i])]
    endif ;; logunit[1L] ge 2L

    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
    if error ne 0 then return

    if usestawid then begin
      tmpstr='[imcombine] '+msg[0L]
      widget_control,stawid,set_value=tmpstr
    endif

    ;;=======================================------------------------------
    ;; First it is checked that all images have the same size, using one
    ;; block at a time:

    hdr=ptrarr(n,nblocks)
    for i=0L,n-1L do begin
      for j=0L,nblocks-1L do begin
        tmp=headfits(filename[i,j],silent=verbose lt 3,errmsg=errmsg)
        hdr[i,j]=ptr_new(tmp,/no_copy)
        if errmsg ne '' then goto,error_handler
      endfor
    endfor

    naxis1=lonarr(n,j)
    naxis2=lonarr(n,j)

    for i=0L,n-1L do begin
      for j=0L,nblocks-1L do begin
        naxis1[i,j]=fxpar(*hdr[i,j],'NAXIS1',0L)
        naxis2[i,j]=fxpar(*hdr[i,j],'NAXIS2',0L)
      endfor
    endfor

    tmpstr='of the same block'
    for j=0L,nblocks-1L do begin
      for i=0L,n-1L do begin
        if naxis1[0L,j] ne naxis1[i,j] or $
           naxis2[0L,j] ne naxis2[i,j] then begin
          errmsg='All input files '+tmpstr+' must have the same dimensions' + $
                 '; {0} ['+strtrim(naxis1[0L,j],2L)+','+ $
                 strtrim(naxis2[1L,j],2L)+'], not {'+strtrim(i,2L)+'} ['+ $
                 strtrim(naxis1[i,j],2L)+','+strtrim(naxis2[i,j],2L)+'].'
          goto,error_handler
        endif
      endfor ;; i=0L,n-1L
    endfor ;; j=0L,nblocks-1L

    for i=0L,n-1L do for j=0L,nblocks-1L do $
      if ptr_valid(hdr[i,j]) then ptr_free,hdr[i,j]

    ;;========================================------------------------------
    ;; Looping over all blocks:

    if nblocks gt 1L then pout=ptrarr(nblocks)

    for j=0L,nblocks-1L do begin

      msg='Combining data of block '+strtrim(j+1L,2L)+'/'+ $
          strtrim(nblocks,2L)+'.'
      error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

      ;;========================================------------------------------
      ;; Reading the data files, and saving them to the same array:

      data=fltarr(n,naxis1[0L,j],naxis2[0L,j])
      for i=0L,n-1L do begin
        datatmp=readfits(filename[i,j],bhdr,silent=verbose lt 3,/no_unsigned)
        data[i,*,*]=temporary(datatmp)
      endfor ;; i=0L,n-1L

      ;;========================================------------------------------
      ;; Performing either a median, an average, or a min/max-average:

      case method of
        'average': begin

          ;; Performing a median on all given files:
          out=float(total(data,dimension=1L,/double)/n)
          cmsg=' Average; used '+strtrim(n,2L)+' files.'
          rstr=' Average'

        end ;; case: 'average'

        'minmax': begin

          ilp=n/20L
          naway=n-2L*(ilp+1L) & nval=naway

          if ~ilp then begin

            ;; Performing an average on all given files:
            if slowmode then begin
              out=fltarr(naxis1[0L,0L],naxis2[0L,0L])
              for i=0L,naxis1[0L,j]-1L do begin
                for k=0L,naxis2[0L,j]-1L do begin
                  arr=reform(data[*,i,k])
                  tmp=min(arr,idxmin) & arr[idxmin]=0.0
                  tmp=max(arr,idxmax) & arr[idxmax]=0.0
                  out[i,k]=total(arr)/naway
                endfor ;; k=0L,naxis2[0L,j]-1L
              endfor ;; i=0L,naxis1[0L,j]-1L
            endif else begin
              tmp=min(data,idxmin,dimension=1L,subscript_max=idxmax)

              ;; Filtering out the minimum and maximum values of every pixel:
              data[idxmin]=0.0 & data[idxmax]=0.0

              ;; Calculating the average:
              out=float(total(data,1L,/double)/naway)
            endelse ;; slowmode

          endif else begin ;; ~ilp

            ;; THIS DOESN'T QUITE WORK YET (OF SOME REASON)

            for i=0L,ilp do begin

              biasidx=bytarr(n-i*2L,naxis1[0L],naxis2[0L])+1b

              ;; Performing an average on all given files:
              tmp=min(data,idxmin,dimension=1L,subscript_max=idxmax)

              ;; Marking the minimum and maximum values of every pixel:
              biasidx[idxmin]=0b & biasidx[idxmax]=0b

              ;; Removing the marked elements:
              data=data[where(biasidx,/l64)]

              data=reform(data,n-(i+1L)*2L,naxis1[0L],naxis2[0L])
            endfor ;; i=0L,ilp

            ;; Calculating the average:
            out=float(total(data,1L,/double)/naway)

          endelse ;; ~ilp

          cmsg=' Minmax; used an average of '+strtrim(nval,2L)+'/'+ $
               strtrim(n,2L)+ $
               ' values/pixel; min/max-filtered [*'+strtrim(ilp+1L,2L)+'].'
          rstr=' Minmax; out of '+strtrim(n,2L)+' images'

        end ;; case: 'minmax'

        else: begin

          ;; Performing a median on all given files:
          out=median(data,dimension=1L)
          cmsg=' Median; used '+strtrim(n,2L)+' files.'
          rstr=' Median'

        end ;; case: else
      endcase ;; method

      error=p3d_misc_logger(cmsg,logunit,rname=rname,verbose=verbose ge 1)

      if nblocks gt 1L then pout[j]=ptr_new(out,/no_copy)
    endfor ;; j=0L,nblocks-1L
  endif else if nblocks gt 1L then begin

    ;; Only reading the data at this point when the data is not combined first:
    pout=ptrarr(nblocks)
    naxis1=lonarr(1L,nblocks)
    naxis2=lonarr(1L,nblocks)

    for j=0L,nblocks-1L do begin
      d=readfits(filename[0L,j],h,silent=verbose lt 3,/no_unsigned)
      pout[j]=ptr_new(d,/no_copy)
      naxis1[0L,j]=fxpar(h,'NAXIS1')
      naxis2[0L,j]=fxpar(h,'NAXIS2')
    endfor

  endif ;; n ge 1L

  ;;========================================------------------------------
  ;; Merging the data of the separate blocks (iff NBLOCKS>1):
  ;;
  ;; Note! All overscan regions are removed.

  if nblocks gt 1L then begin

    msg=['Merging the data of all blocks.']

    ;; Determining the extent of each block in the combined image:
    xmin=lonarr(nblocks) & xmax=lonarr(nblocks) & xoff=lonarr(nblocks)
    ymin=lonarr(nblocks) & ymax=lonarr(nblocks) & yoff=lonarr(nblocks)

    mini=min(detsec[0L:1L,*],dimension=1L) & tmp=min(mini,minjx)
    mini=min(detsec[2L:3L,*],dimension=1L) & tmp=min(mini,minjy)
    for j=0L,nblocks-1L do begin
      ;; x:
      cvar=min(detsec[0L:1L,j]) eq min(detsec[0L:1L,*])
      xmin[j]=cvar?0L:(max(detsec[0L:1L,minjx])+1L)
      xoff[j]=cvar?0L:(naxis1[0L,j]-xmin[j])

      xmax[j]=max(detsec[0L:1L,j])

      ;; y:
      cvar=min(detsec[2L:3L,j]) eq min(detsec[2L:3L,*])
      ymin[j]=cvar?0L:(max(detsec[2L:3L,minjy])+1L)
      yoff[j]=cvar?0L:(naxis2[0L,j]-ymin[j])

      ymax[j]=max(detsec[2L:3L,j])
    endfor ;; j=0L,nblocks-1L

    ;; Determining the combined image size and defining the output array:
    out=fltarr(max(xmax)+1L,max(ymax)+1L)

    ;; Copying the data of the separate blocks to the output array:
    for j=0L,nblocks-1L do begin
      tmp=*pout[j] & ptr_free,pout[j]

      ;; Flipping the orientation of the data, where necessary:
      if detsec[0L,j] gt detsec[1L,j] then begin
        msg=[msg,'  block '+strtrim(j+1L,2L)+'/'+strtrim(nblocks,2L)+ $
                 ' :: Flipped the input data in the x-direction.']
        tmp=reverse(tmp,1L)
      endif
      if detsec[2L,j] gt detsec[3L,j] then begin
        msg=[msg,'  block '+strtrim(j+1L,2L)+'/'+strtrim(nblocks,2L)+ $
                 ' :: Flipped the input data in the y-direction.']
        tmp=reverse(tmp,2L)
      endif

      out[xmin[j]:xmax[j],ymin[j]:ymax[j]]= $
         tmp[xoff[j]:xoff[j]+xmax[j]-xmin[j], $
             yoff[j]:yoff[j]+ymax[j]-ymin[j]]

      tmp1=string(format='(i4,":",i4,",",i4,":",i4)', $
               xoff[j]+1L,xoff[j]+xmax[j]-xmin[j]+1L, $
               yoff[j]+1L,yoff[j]+ymax[j]-ymin[j]+1L)
      tmp2=string(format='(i4,":",i4,",",i4,":",i4)', $
               xmin[j]+1L,xmax[j]+1L,ymin[j]+1L,ymax[j]+1L)
      msg=[msg,'  block '+strtrim(j+1L,2L)+'/'+strtrim(nblocks,2L)+ $
           ' :: ['+tmp1+'] => ['+tmp2+']']

    endfor ;; j=0L,nblocks-1L

    ;; Flipping the data on the spatial axis, if required to do so:
    if spatialflip then out=reverse(out,sid)

    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
  endif ;; nblocks gt 1L

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  if n_elements(pout) gt 0L then begin
    if total(ptr_valid(pout)) ge 1L then begin
      n=n_elements(pout)
      for i=0L,n-1L do if ptr_valid(pout[i]) then ptr_free,pout[i]
    endif
  endif

  return
END ;;; procedure: p3d_misc_imcombine
