;+
; NAME:
;         p3d_misc_mean_smooth
;
;         $Id: p3d_misc_mean_smooth.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         This routine calculates a two-dimensional array smoothing. A
;         rectangular box of the size [2*XBOX+1,2*YBOX+1] is moved across the
;         array and the center of the box is replaced by the average of the
;         box.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_mean_smooth,image,out,xbox,ybox,topwid=,logunit=,
;             verbose=,error=,/debug,/help
;
; INPUTS:
;         image           - A two-dimensional array.
;         xbox            - The half x-width of the filter smoothing box.
;         ybox            - The half y-width of the filter smoothing box.
;
; KEYWORD PARAMETERS:
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - 2D image
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         07.10.2008 - Converted from the original routines mean_smo of
;                      Thomas Becker. /CS
;-
PRO p3d_misc_mean_smooth,image,out,xbox,ybox,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  error=0 & rname='p3d_misc_mean_smooth: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_misc_mean_smooth'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking input and output arguments:

  s=size(image)
  if s[0L] ne 2L then begin
    errmsg='IMAGE must be set; to a two-dimensional array.'
    goto,error_handler
  endif

  if n_elements(xbox) ne 1L then begin
    errmsg='XBOX must be set; to a scalar integer.'
    goto,error_handler
  endif

  if n_elements(ybox) ne 1L then ybox=xbox

  msg='XBOX,YBOX = '+strtrim(xbox,2L)+','+strtrim(ybox,2L)+'.'
  error=p3d_misc_logger(msg,logunit,loglevel=3L,rname=rname, $
      verbose=verbose ge 2)
  if error ne 0 then return

  if ~xbox and ~ybox then begin
    out=image
    return
  endif

  ;;========================================------------------------------
  ;; Performing the smoothing:

  time1=systime(1) 

  if xbox eq ybox then begin

    ;;====================----------
    ;; For a quadratic box:

    borderless=fltarr(s[1L]+2L*xbox,s[2L]+2L*ybox)
    borderless[(xbox):(s[1L]+xbox-1L),(ybox):(s[2L]+ybox-1L)]=image

    out=smooth(borderless,2L*xbox+1L)
    out=out[(xbox):(s[1L]+xbox-1L),(ybox):(s[2L]+ybox-1L)]

    points1=((lindgen(s[1L],s[2L]) mod s[1L])+xbox+1L) < (2L*xbox+1L) < $
            (s[1L]-1L-(lindgen(s[1L],s[2L]) mod s[1L])+xbox+1L)
    points2=(lindgen(s[1L],s[2L])/s[1L]+ybox+1L) < (2L*ybox+1L) < $
            (s[2L]-1L-(lindgen(s[1L],s[2L])/s[1L])+ybox+1L)
    out*=(2L*xbox+1L)*(2L*ybox+1L)/points1/points2   

  endif else begin

    ;;====================----------
    ;; For a rectangular box:

    out=fltarr(s[1L],s[2L])
    for k=-xbox,xbox do begin
      xlow =k>0L     & xup =s[1L]+(k<0L)-1L
      xlow2=-(k<0L)  & xup2=s[1L]-(k>0L)-1L
      for L=-ybox,ybox do begin
        ylow =  L>0L  & yup =s[2L]+(L<0L)-1L
        ylow2=-(L<0L) & yup2=s[2L]-(L>0L)-1L
        out[xlow:xup,ylow:yup]+=image[xlow2:xup2,ylow2:yup2]
      endfor
    endfor

    points1=((lindgen(s[1L],s[2L]) mod s[1L])+xbox+1L) < (2L*xbox+1L) < $
            (s[1L]-1L-(lindgen(s[1L],s[2L]) mod s[1L])+xbox+1L) 
    points2=(lindgen(s[1L],s[2L])/s[1L]+ybox+1L) < (2L*ybox+1L) < $
            (s[2L]-1L-(lindgen(s[1L],s[2L])/s[1L])+ybox+1L)
    out/=points1*points2

  endelse

  time2=systime(1)
  msg='Processing time: '+strtrim(time2-time1,2L) +' sec.'
  error=p3d_misc_logger(msg,logunit,loglevel=3L,rname=rname, $
      verbose=verbose ge 3)

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_misc_mean_smooth
