;+
; NAME:
;         p3d_misc_read_postable
;
;         $Id: p3d_misc_read_postable.pro 127 2010-03-22 16:41:40Z christersandin $
;
; PURPOSE:
;         Reads a spatial element position file, and returns the contents. The
;         returning array XPOS always returns coordinates West->East and YPOS
;         South->North.
;
;         The format of the position table is well defined and must be followed
;         in order to provide a correctly defined output. The format of a
;         position table file is as follows:
;
;         header ::
;          All entries must be preceeded by a '; ' in the columns 1 & 2. The
;          parameter value is specified next, and then the parameter name.
;          Required parameters are (default values in []):
;
;          SHAPE  [0],1 ::
;            0:: square-shaped elements
;            1:: circular elements
;            2:: hexagon-shaped elements; with vertice 1 at angle=0
;            3:: hexagon-shaped elements; with vertice 1 at angle=30
;
;          XISW2E [0],1 ::
;            0:: Column 1, out of 2, in the data gives the S->N positions.
;            1:: Column 1, out of 2, in the data gives the W->E positions.
;
;          XFAC -1,[1] ::
;            -1:: The first column positions will be multiplied with -1 to make
;                 the coordinates go from West to East (not from East to West).
;                 This multiplication takes place -before- the x and y columns
;                 are swapped (iff XISW2E==0).
;            1:: The first column positions already give coordinates as West to
;                East.
;
;          YFAC -1,[1] ::
;           -1:: The first column positions will be multiplied with -1 to make
;                the coordinates go from South to North (not from North to
;                South). This multiplication takes place -before- the x and y
;                columns are swapped (iff XISW2E==0).
;            1:: The first column positions already give coordinates as South
;                to North.
;
;
;         data ::
;          The positional data must be present in five columns, and the unit of
;          the values must be the same for all elements in columns 3-5:
;
;         Column 1 ::
;          The image row (cross-dispersion direction). This number will be
;          decremented by 1 inside the program in order to account for IDL's
;          0-based array numbering.
;
;          Note! Rows, which have an ID higher than, or equal to, ELEMNOUSE
;               will be deleted and hence the row numbering will be decremented
;               by one for rows ahead of every such row.
;
;         Column 2 :: This column specifies the spatial element ID. Element
;          IDs, which are different from the image row number, are used by,
;          e.g., the PPAK-IFU.
;
;          Note! Spatial elements marked with an ID of a science, calibration
;                (ELEMCALIB), sky (ELEMSKY), and not to be used (ELEMNOUSE) all
;                use a different numbering (see below).
;
;         Column 3 :: This is the x-coordinate column. If the data of this
;          column specify W-E coordinates, then XISW2E must be set to 1 in the
;          table header. 
;
;         Column 4 :: This is the y-coordinate column. If the data of this
;          column specify S-N coordinates, then XISW2E must be set to 1 in the
;          table header.
;
;         Column 5 :: This column specifies the size of every spatial element,
;          in the same unit as the coordinates. If SHAPE==1-3 then this number
;          specifies the circular element diameter. It is not used if SHAPE==0.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_read_postable,postable=,rownum=,id=,xpos=,ypos=, $
;             lens_size=,shape=,nvertices=,science=,sky=,calibration=, $
;             elemnouse=,elemsky=,elemcalib=,topwid=,logunit=,verbose=, $
;             error=,/debug,/help
;
; KEYWORD PARAMETERS:
;         postable        - Position table filename (containing the full path).
;         rownum          - An array of element row numbers [output].
;         id              - An array of element  id numbers [output].
;         xpos            - An array of element x positions [output].
;         ypos            - An array of element y positions [output].
;         lens_size       - An array of element lens sizes [output].
;         shape [0]       - An integer specifying the shape of the spatial
;                           elements; 0-3 [output].
;         nvertices [15]  - An integer specifying the number of vertices of
;                           circularly shaped elements (SHAPE==1||2||3).
;         science         - Read science elements.
;         sky             - Read sky elements.
;         calibration     - Read calibration elements.
;         elemnouse [1000000L] - Elements with an index larger than this value
;                           are considered to not be used. 
;         elemsky [800000L] - Elements with an index larger than this value are
;                           considered to be sky elements.
;                           Note! ELEMSKY<ELEMCALIB !!
;         elemcalib [900000L] - Elements with an index larger than this value
;                           are considered to be calibration elements.
;                           Note! ELEMCALIB<ELEMNOUSE !!
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         02.09.2008 - Converted from original routine postable.pro of
;                      Thomas Becker, Martin Roth, Petra Bhm. /CS
;         05.02.2009 - Introduced a proper routine and table format
;                      description.
;-
PRO p3d_misc_read_postable,postable=postable,rownum=rownum,id=id, $
        xpos=xpos,ypos=ypos,lens_size=lens_size,shape=shape, $
        nvertices=nvertices,science=science,sky=sky, $
        calibration=calibration,elemnouse=elements_nouse, $
        elemsky=elements_sky,elemcalib=elements_calibration, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error, $
        debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_misc_read_postable: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_misc_read_postable'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking input arguments:

  s=size(postable)
  if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L  then begin
    errmsg='POSTABLE must be defined; and set to the name of an existing f' + $
           'ile.'
    goto,error_handler
  endif
  if ~file_test(postable,/regular,/read) then begin
    errmsg='POSTABLE must be defined; and set to the name of an existing f' + $
           'ile.'
    goto,error_handler
  endif

  if ~n_elements(elements_nouse) then elements_nouse=1000000L
  sb=size(elements_nouse)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0L]+1L] ge 4L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='ELEMNOUSE must be set to a scalar integer.'
    goto,error_handler
  endif

  if ~n_elements(elements_sky) then elements_sky=800000L
  sb=size(elements_sky)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0L]+1L] ge 4L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='ELEMSKY must be set to a scalar integer.'
    goto,error_handler
  endif

  if ~n_elements(elements_calibration) then elements_calibration=900000L
  sb=size(elements_calibration)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0L]+1L] ge 4L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='ELEMCALIB must be set to a scalar integer.'
    goto,error_handler
  endif

  if elements_sky ge elements_calibration then begin
    errmsg='ELEMSKY must be < ELEMCALIB.'
    goto,error_handler
  endif
  if elements_calibration ge elements_nouse then begin
    errmsg='ELEMCALIB must be < ELEMNOUSE.'
    goto,error_handler
  endif
  elements_science=min([elements_calibration,elements_sky])-1L

  ;;========================================------------------------------
  ;; Reading various setup parameters related to the position table:

  errmsg='Error reading the comment header in the fiber position table file.'
  on_ioerror,format_error
  openr,unit,postable,/get_lun

  ok=1L
  while ok do begin 
    tmp=' '
    readf,unit,tmp
    if strmid(tmp,0L,2L) eq ';;' then begin
      ;; A comment line...skipping to the next line.
    endif else if strmid(tmp,0L,2L) ne '; ' then begin
      ok=0L
    endif else begin
      ;; Reading the shape:
      idx=strpos(tmp,'shape')
      if idx gt 0L then shape=strmid(tmp,1L,idx-1L)

      ;; Reading the number of vertices:
      idx=strpos(tmp,'nvertices')
      if idx gt 0L then nvertices=strmid(tmp,1L,idx-1L)

      ;; Reading the scale:
      idx=strpos(tmp,'scale')
      if idx gt 0L then scale=strmid(tmp,1L,idx-1L)

      ;; Reading the xisw2e:
      idx=strpos(tmp,'xisw2e')
      if idx gt 0L then xisw2e=strmid(tmp,1L,idx-1L)

      ;; Reading the xfac:
      idx=strpos(tmp,'xfac')
      if idx gt 0L then xfac=strmid(tmp,1L,idx-1L)

      ;; Reading the yfac:
      idx=strpos(tmp,'yfac')
      if idx gt 0L then yfac=strmid(tmp,1L,idx-1L)
    endelse
  endwhile

  free_lun,unit

  ;; Checking the data:
  if ~n_elements(shape) then shape=0L
  shape=long(shape)
  if shape lt 0L or shape gt 3L then begin
    errmsg='SHAPE must be 0, 1, 2, or 3, not "'+strtrim(shape,2L)+'".'
    goto,error_handler
  endif

  if ~n_elements(nvertices) then begin
    case shape of
      2L: nvertices=6L
      3L: nvertices=6L
      else: nvertices=15L
    endcase
  endif
  nvertices=long(nvertices)
  if nvertices lt 4L or nvertices gt 100L then begin
    errmsg='NVERTICES must be >=4, and <=100, not "'+strtrim(nvertices,2L)+'".'
    goto,error_handler
  endif

  if ~n_elements(scale) then scale=1.0
  scale=double(scale)
  if scale le 0d0 then begin
    errmsg='SCALE must be >0.0., not "'+strtrim(scale,2L)+'".'
    goto,error_handler
  endif
  
  if ~n_elements(xisw2e) then xisw2e=0L
  xisw2e=long(xisw2e)
  if xisw2e ne 0L and xisw2e ne 1L then begin
    errmsg='XISW2E must be 0 or 1, not "'+strtrim(xisw2e,2L)+'".'
    goto,error_handler
  endif

  if ~n_elements(xfac) then xfac=1L
  xfac=long(xfac)
  if xfac ne -1L and xfac ne 1L then begin
    errmsg='XFAC must be -1 or 1, not "'+strtrim(xfac,2L)+'".'
    goto,error_handler
  endif

  if ~n_elements(yfac) then yfac=1L
  yfac=long(yfac)
  if yfac ne -1L and yfac ne 1L then begin
    errmsg='YFAC must be -1 or 1, not "'+strtrim(yfac,2L)+'".'
    goto,error_handler
  endif

  msg='shape, nvertices, scale, xisw2e, xfac, yfac :: '+ $
      string(format='(i1,",",i3,",",f7.2,",",i2,",",i3,",",i3)', $
             shape,nvertices,scale,xisw2e,xfac,yfac)
  error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose ge 1)

  ;;========================================------------------------------
  ;; Reading data:

  tmp=read_ascii(postable,comment_symbol=';') & tmp=tmp.field1
  rownum =  long(tmp[0L,*])-1L
  id     =  long(tmp[1L,*])
  xpos   = float(tmp[2L,*])
  ypos   = float(tmp[3L,*])
  lens_size = float(tmp[4L,*])

  msg=['no use element id:s :: '       +strtrim(elements_nouse,2L), $
       'sky element id:s :: > '        +strtrim(elements_sky  ,2L), $
       'calibration element id:s :: > '+strtrim(elements_calibration,2L)]
  error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose ge 1)

  ;; Handling calibration & sky elements:
  gappos=where(id eq elements_nouse,count)
  if count gt 0L then begin
    nrows=n_elements(rownum)
    for k=0L,n_elements(gappos)-1L do rownum[gappos[k]:nrows-1L]--
    rownum[gappos]=-1L
  endif

  lscience=keyword_set(science)
  lsky    =keyword_set(sky)
  lcalib  =keyword_set(calibration)
  if ~lscience and ~lsky and ~lcalib then begin
    ;; Nothing is done, not an option (see next statement):
    lscience=1L & lsky=1L & lcalib=1L
  endif

  if lscience and lsky and lcalib then begin
    errmsg='Set to do all SCIENCE, SKY, and CALIBRATION elements, returning.'
    goto,error_handler
  endif

  msg='Saving positions of'
  msg+=lscience?' science':''
  msg+=lcalib?' calibration':''
  msg+=lsky?' sky':''
  msg+=' fibers.'
  error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose ge 1)

  ;; Science elements:
  science_found=0L
  if lscience then begin
    sciencefibers=where(id le elements_science,count)
    if count gt 0L then begin
      scienceid        =        id[sciencefibers]
      sciencerownum    =    rownum[sciencefibers]
      sciencexpos      =      xpos[sciencefibers]
      scienceypos      =      ypos[sciencefibers]
      sciencelensesize = lens_size[sciencefibers]
      science_found=1L

      msg='Found '+strtrim(count,2L)+' science elements in the' + $
          ' position table file.'
      error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
          verbose=verbose ge 1)
    endif
  endif ;; science

  ;; Sky elements:
  sky_found=0L
  if lsky then begin
    skyfibers=where(id ge elements_sky and id lt elements_calibration,count)
    if count gt 0L then begin
      skyid        =        id[skyfibers]
      skyrownum    =    rownum[skyfibers]
      skyxpos      =      xpos[skyfibers]
      skyypos      =      ypos[skyfibers]
      skylensesize = lens_size[skyfibers]
      sky_found=1L

      msg='Found '+strtrim(count,2L)+' sky elements in the position' + $
          ' table file.'
      error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
          verbose=verbose ge 1)
    endif
  endif ;, lsky

  ;; Calibration elements:
  calib_found=0L
  if lcalib then begin
    calibfibers=where(id ge elements_calibration and id lt elements_nouse, $
                      count)
    if count gt 0L then begin
      calibid     =     id[calibfibers]
      calibrownum = rownum[calibfibers]
      calibxpos   =   xpos[calibfibers]
      calibypos   =   ypos[calibfibers]
      calib_found = 1L

      msg='Found '+strtrim(count,2L)+' calibration elements in the po' + $
          'sition table file.'
      error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
          verbose=verbose ge 1)
    endif
  endif ;; lcalib

  ;; Creating the final output arrays:
  rownum=-1L & id=-1L & xpos=-1L & ypos=-1L & lens_size=-1L

  if lscience then begin
    if science_found then begin
      rownum = [rownum,sciencerownum]
      id     = [id,scienceid]
      xpos   = [xpos,sciencexpos]
      ypos   = [ypos,scienceypos]
      lens_size = [lens_size,sciencelensesize]
    endif
  endif
  if lsky then begin
    if sky_found then begin
      rownum = [rownum,skyrownum]
      id     = [id,skyid]
      xpos   = [xpos,skyxpos]
      ypos   = [ypos,skyypos]
      lens_size = [lens_size,skylensesize]
    endif
  endif
  if lcalib then begin
    if calib_found then begin
      rownum = [rownum,calibrownum]
      id     = [id,calibid]
      xpos   = [xpos,calibxpos]
      ypos   = [ypos,calibypos]
    endif
  endif

  n=n_elements(rownum)-1L
  if n_elements(   rownum) gt 1L then    rownum=   rownum[1L:n]
  if n_elements(       id) gt 1L then        id=       id[1L:n]
  if n_elements(     xpos) gt 1L then      xpos=     xpos[1L:n]
  if n_elements(     ypos) gt 1L then      ypos=     ypos[1L:n]
  if n_elements(lens_size) gt 1L then lens_size=lens_size[1L:n]

  xpos*=scale*xfac
  ypos*=scale*yfac
  lens_size*=scale

  ;; Swapping positions, if necessary, to always have x as W->E, and y as S->N:
  if ~xisw2e then begin
    tmp=ypos & ypos=xpos & xpos=tmp
  endif

  msg='The output arrays contain totally '+strtrim(n,2L)+' elements.'
  error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose ge 1)

  return

format_error:
  tmp=fstat(unit)
  if tmp.open then free_lun,unit

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_misc_read_postable
