;+
; NAME:
;         p3d_misc_sauron_colormap
;
;         $Id: p3d_misc_sauron_colormap.pro 83 2010-03-05 11:33:06Z olestreicher $
;
; PURPOSE:
;         This routine sets up a "Sauron"-style color table using the recipe
;         of: Michele Cappellari & Eric Emsellem, Leiden, 10 July 2001
;
;         Start with 7 equally spaced coordinates, then adds 4 additional
;         points: x=findgen(7)*255/6.+1
;         [1.0,  43.5,  86.0,  128.5,  171.0,  213.5,  256.0]
;
;         This colormap formulation is, with permission of its original
;         authors used in the p3d package. The routine has been adapted to
;         allow the use of different numbers of colors and offsets.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_sauron_colormap,ncolors=,bottom=
;
; KEYWORD PARAMETERS:
;         ncolors [!d.table_size] - 
;                           A scalar integer specifying the number of
;                           colors to use.
;         bottom [max(index)+1] - 
;                           A scalar integer that specifies the lower
;                           index of the colortable that is used in the
;                           scaling of the colormap. The upper index is always
;                           the maximum value (!d.table_size). This value
;                           should be set >=2 (5 is the optimum choice) to
;                           allow some room for the reserved colors of P3D.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         Modifies the colortable.
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         17.11.2009 - Converted into a p3d GPL routine. /CS
;-
PRO p3d_misc_sauron_colormap,ncolors=ncolors,bottom=bottom
  compile_opt hidden,IDL2

  if ~n_elements(ncolors) then ncolors=!d.table_size
  if ~n_elements(bottom) then bottom=0L

  ncolors=ncolors<(!d.table_size-bottom) ;; no wrapping is allowed

  ;; scaling x to use NCOLOR colors:
  x=[1.0, 43.5, 86.0, 86.0+20.0, 128.5-10.0, 128.5, $
     128.5+10.0, 171.0-20.0, 171.0, 213.5, 256.0]/256.0*ncolors

  r=[0.0, 0.0, 0.4 , 0.5, 0.3, 0.0, 0.7, 1.0,  1.0, 1.0, 0.9]
  g=[0.0, 0.0, 0.85, 1.0, 1.0, 0.9, 1.0, 1.0, 0.85, 0.0, 0.9]
  b=[0.0, 1.0, 1.0 , 1.0, 0.7, 0.0, 0.0, 0.0,  0.0, 0.0, 0.9]

  xnew=findgen(ncolors)+1.0
  rV=interpol(r,x,xnew)
  gV=interpol(g,x,xnew)
  bV=interpol(b,x,xnew)

  ;; Load the SAURON colormap:
  tvlct,rV*255,gV*255,bV*255,bottom

  return
END ;;; procedure: p3d_misc_sauron_colormap
