;+
; NAME:
;         p3d_tracing_findspec0
;
;         $Id: p3d_tracing_findspec0.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         For the value COLUMN in the dispersion direction of ARRAY this
;         routine finds the location of all spectra (it can find) in the cross-
;         dispersion direction.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: tracing of spectra on the CCD
;
; CALLING SEQUENCE:
;         p3d_tracing_findspec0,array,column,width,cut,pos, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         array           - A two-dimensional array [not checked for its
;                           properties in this routine].
;         column          - A scalar integer specifying on which pixel in the
;                           dispersion direction the data in the cut out
;                           wavelength range is centered.
;         width           - A scalar integer specifying the half wavelength
;                           range, which is used to identify the spectra.
;         cut             - A decimal scalar specifying the fraction of the
;                           wavelength bins which must have a maximum in order
;                           for a spectrum to be indicated as such;
;                           0.0<=CUT<=1.0.
;
; KEYWORD PARAMETERS:
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         pos             - A one-dimensional array of floating point type
;                           which specifies the fractional positions of all
;                           found spectra in the cross-dispersion direction. If
;                           none was found, then pos=-1L.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         06.10.2008 - Converted from the original routine findspec0 of
;                      Thomas Becker. /CS
;-
PRO p3d_tracing_findspec0,array,column,width,cut,pos,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_tracing_findspec0: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_tracing_findspec0'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(array)
  if ~s[s[0L]+2L] then begin
    errmsg='ARRAY must be set.'
    goto,error_handler
  endif

  if ~n_elements(column) then begin
    errmsg='COLUMN must be set.'
    goto,error_handler
  endif

  if ~n_elements(width) then begin
    errmsg='WIDTH must be set.'
    goto,error_handler
  endif

  if ~n_elements(cut) then begin
    errmsg='CUT must be set.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Locating the local maxima (i.e. the spectra) in the data along the cross-
  ;; dispersion direction:

  width=width<((s[1L]-column-1L)<column)
  arr=array[(column-width):(column+width),*]
  peakarr=(arr[*,1L:s[2L]-2L] ge arr[*,0L:s[2L]-3L]) and $
          (arr[*,1L:s[2L]-2L] ge arr[*,2L:s[2L]-1L])

  totalposarr=fltarr(s[2L])
  totalposarr[1L:s[2L]-2L]=total(peakarr,1L)

  ;; POSARR is 1 for pixels where a local maximum was found for at least
  ;; CUT*(number of local max. pixels in the wavelength range) for a 2-pixel
  ;; wide stripe. A too small value on CUT has the potential to identify noise
  ;; as a spectrum: 
  ncol=(float(2L*width+1L)*(cut<1.0))
  posarr=totalposarr[0L:s[2L]-2L]+totalposarr[1L:s[2L]-1L] gt ncol

  pos0=where(posarr,count)
  if count ne 0L then begin
    ;; Two consecutive rows of pixels in the cross-dispersion direction, which
    ;; are both marked as OK [1], is not possible. Hence the next row is
    ;; marked as not OK.
    n=n_elements(pos0)
    ok=bytarr(n)+1b
    for k=0L,n-2L do if pos0[k+1L] eq pos0[k]+1L then ok[k+1L]=0b
    pos0=pos0[where(ok)]

    ;; Finding a sub-pixel offset accounting for those wavelength bins where
    ;; the local maximum was found in the consecutive pixel:
    n=n_elements(pos0)
    pos=float(pos0)
    for k=0L,n-1L do begin
      tmp=pos[k]
      pos[k]+=totalposarr[tmp+1L]/total(totalposarr[tmp:(tmp+1L)])
    endfor
  endif else begin
    pos=-1L
  endelse ;; count ne 0L

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_tracing_findspec0
