;+
; NAME:
;         p3d_wavecal_correct_maskpos
;
;         $Id: p3d_wavecal_correct_maskpos.pro 140 2010-03-28 17:51:09Z christersandin $
;
; PURPOSE:
;         This routine calculates corrected positions for a list of calibration
;         lines in every spectrum of a supplied spectrum image. The method is
;         either 'Gaussian' (the default), or 'Weighted'. The results should be
;         much more accurate using Gaussian line centering than using a data-
;         weighted centering. The approach using Gaussian fitting is, however,
;         much more time-consuming.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: wavelength calibration
;
; CALLING SEQUENCE:
;         p3d_wavecal_correct_maskpos,refimage,linepos,lwid,out,inlines, $
;             niterat=,offsets=,dooffsets=,refrow=,fwhm=,method=,/monitor, $
;             stawid=,topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         refimage        - A two-dimensional array; of floating point type. It
;                           is assumed that the dispersion axis is the first
;                           dimension in REFIMAGE.
;         linepos         - A one-dimensional array.
;         lwid            - The width of the region, which is used to integrate
;                           over the emission lines; given in pixels.
;
; KEYWORD PARAMETERS:
;         niterat [5]     - The maximum number of iterations; a scalar
;                           integer<=100. Only used with METHOD=='Weighted'.
;         offsets         - Specifies an array of offsets between the different
;                           spectra.
;         dooffsets       - Calculates the offset array between the different
;                           spectra anew.
;         refrow [s[2]/2] - The starting spectrum.
;         fwhm            - A scalar decimal value with the instrumental line
;                           width in the dispersion dimension. Only used with
;                           METHOD=='Gaussian'.
;         method          - A scalar string that defines the method to be used
;                           when calculating more precise line center
;                           positions: ['Gaussian'] or 'Weighted'.
;         monitor         - If set to 1 then Gaussian fits are shown for the
;                           reference spectrum. If MONITOR==2 then the fit is
;                           shown for all spectra.
;         stawid          - If set to a valid ID then a log message is written
;                           using this ID for relevant actions.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - A two-dimensional array of corrected positions for
;                           every calibration line in every spectrum.
;         inlines         - A one-dimensional array specifying for which lines
;                           in LINEPOS that corrected positions could be
;                           calculated. If none, then inlines==-1L.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         17.10.2008 - Converted from original routine correct_lines of
;                      Thomas Becker. /CS
;-
PRO p3d_wavecal_correct_maskpos_gauss,x,a,fx
  compile_opt hidden,IDL2

  tmp1=-0.5d0*((x-a[2L])/(a[3L]>1d-20))^2
  tmp2=a[4L]/(a[3L]*sqrt(2d0*!DPI))
  fx=tmp2*exp(tmp1)+a[0L]+a[1L]*x

  return
END ;;; procedure: p3d_wavecal_correct_maskpos_gauss

PRO p3d_wavecal_correct_maskpos,refimage,linepos,lwid,out,inlines, $
        niterat=nit,offsets=offsets_,dooffsets=dooffsets,refrow=rwfrow, $
        fwhm=fwhm,method=method,monitor=monitor_,stawid=stawid,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_wavecal_correct_maskpos: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)
  usestawid=~n_elements(stawid)?0L:widget_info(stawid,/valid_id)

  if keyword_set(help) then begin
    doc_library,'p3d_wavecal_correct_maskpos'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(refimage)
  if s[0L] ne 2L or (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='REFIMAGE [1] must be set; to a two-dimensional array of floati' + $
           'ng point type.'
    goto,error_handler
  endif

  sl=size(linepos)
  if sl[0L] ne 1L or $
    (sl[sl[0L]+1L] ge 6L and sl[sl[0L]+1L] le 11L) then begin
    errmsg='LINEPOS [2] must be set; to a one-dimensional array of floatin' + $
           'g point type.'
    goto,error_handler
  endif
  sl=sl[3L] ;; the number of elements
  ssl=strtrim(sl,2L)
  lpos=linepos

  sb=size(lwid)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='LINEWIDTH [3] must be set; to a scalar integer.'
    goto,error_handler
  endif
  lwid=long(lwid)

  if ~n_elements(nit) then nit=5L
  sb=size(nit)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0L]+1L] ge 4L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='NITERAT must, if set, be a scalar integer; 1<=NITERAT<=100.'
    goto,error_handler
  endif
  nit=long(nit)
  if nit le 0L or nit gt 100L then begin
    errmsg='NITERAT must, if set, be a scalar integer; 1<=NITERAT<=100.'
    goto,error_handler
  endif

  if n_elements(offsets_) ne 0L then begin
    sb=size(offsets_)
    if sb[0L] ne 2L or $
      (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
      errmsg='OFFSETS, when set, must be a two-dimensional array of decima' + $
             'l type.'
      goto,error_handler
    endif
    if sb[2L] ne s[2L] or sb[1L] ne sl then begin
      errmsg='OFFSETS ['+strtrim(sb[1L],2L)+','+strtrim(sb[2L],2L)+ $
             '] must have as many columns as there are elements in LINEPOS' + $
             ' ['+strtrim(sl,2L)+'], and as many rows as there are rows in' + $
             ' REFIMAGE ['+strtrim(s[2L],2L)+'].'
      goto,error_handler
    endif
    offsets=float(offsets_)
  endif ;; n_elements(offsets) ne 0L

  if ~n_elements(method) then method='Gaussian'

  if ~n_elements(refrow) then refrow=s[2L]/2L
  sr=size(refrow)
  if sr[sr[0L]+2L] ne 1L or $
    (sr[sr[0L]+1L] ge 4L and sr[sr[0L]+1L] le 11L) then begin
    errmsg='REFROW must, if set, be a scalar integer.'
    goto,error_handler
  endif

  sw=size(fwhm)
  if sw[sw[0L]+2L] ne 1L or $
    (sw[sw[0L]+1L] ge 6L and sw[sw[0L]+1L] le 11L) then begin
    errmsg='FWHM must be set to a scalar decimal value; FWHM>0.0.'
    goto,error_handler
  endif
  if fwhm lt 0d0 then begin
    errmsg='FWHM must be set to a scalar decimal value; FWHM>0.0.'
    goto,error_handler
  endif

  monitor=~n_elements(monitor_)?0L:monitor_
  sw=size(monitor)
  if sw[sw[0L]+2L] ne 1L or $
    (sw[sw[0L]+1L] ge 4L and sw[sw[0L]+1L] le 11L) then begin
    errmsg='MONITOR must, if set, be a scalar integer; 0<=MONITOR<=2.'
    goto,error_handler
  endif
  if monitor lt 0L or monitor gt 2L then begin
    errmsg='MONITOR must, if set, be a scalar integer; 0<=MONITOR<=2.'
    goto,error_handler
  endif

  if monitor ge 1L then begin
    device,get_decomposed=decomposed
    device,decomposed=0L
  endif

  ;;========================================------------------------------
  ;; Offsets between the reference spectrum are either calculated (/DOOFFSETS)
  ;; or can be input (OFFSETS) or ignored:

  pos0=fltarr(sl)
  lpos=round(lpos)
 
  if keyword_set(dooffsets) then begin
    loffsets=fltarr(s[2L])

    ;; Looping from the middle spectrum to the last:
    spec0=refimage[*,refrow]
    for k=refrow+1L,s[2L]-1L do begin
      spec=refimage[*,k]
      offset=p3d_misc_correlate_arrays(spec0,spec,topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug)
      if error ne 0 then return
      loffsets[k]=offset
    endfor

    ;; Looping from the middle spectrum to the first:
    for k=refrow-1L,0L,-1L do begin
      spec=refimage[*,k]
      offset=p3d_misc_correlate_arrays(spec0,spec,topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug)
      if error ne 0 then return
      loffsets[k]=offset
    endfor

    ;; Smoothing the offset array:
    p3d_misc_smooth_1d,loffsets,1L,med,/median,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return

    loffsets=med
    loffsets=-loffsets

    ;; In this case no change of the geometry is assumed across the
    ;; dispersion axis:
    loffsets=rebin(transpose(loffsets),sl,s[2L])

  endif else begin ;; keyword_set(dooffsets)

    loffsets=~n_elements(offsets)?fltarr(sl,s[2L]):offsets

  endelse ;; keyword_set(dooffsets)

  ;;========================================------------------------------
  ;; Normalizing the offset array with the values in the center spectrum:

  tmp=loffsets[*,refrow]
  loffsets-=rebin(tmp,sl,s[2L])

  ;;========================================------------------------------
  ;; Looping over all entries in the list of calibration lines:

  ok=lonarr(sl)
  for k=0L,sl-1L do $
     ok[k]=lpos[k]+min(loffsets[k,*]) ge lwid and $
           lpos[k]+max(loffsets[k,*]) lt s[1L]-lwid

  out=dblarr(sl,s[2L])-1d0
  x=dindgen(4L*lwid+1L)-2L*lwid

  stride=[1L,-1L]
  lstart=stride+refrow
  lfinal=[s[2L]-1L,0L]

  ;;========================================------------------------------
  ;;========================================------------------------------
  ;;========================================------------------------------
  ;; Using a Gaussian:

  if strlowcase(method) eq 'gaussian' then begin

    ;; Constrain parameters (no parameter can be <0):
    a=[0d0,0d0,0d0,fwhm/(sqrt(8d0*alog(2d0))),1d0]
    parinfo=replicate({fixed:0L,limited:[1L,0L],limits:[0d0,0d0]},5L)
    parinfo[1L].limited=[0L,0L]
    parinfo[2L].limited=[0L,0L]
    parinfo[3L].limited=[1L,1L]
    parinfo[3L].limits=[0.7,1.3]*a[3L]

    ;; 0  - zero-level
    ;; 1  - slope
    ;; 2  - center
    ;; 3  - sigma
    ;; 4  - intensity

    ;; Looping over all entries in the line list:
    for k=0L,sl-1L do begin

      if usestawid then begin
        msg='Calculating cross-dispersion line center positions for arc li' + $
            'nes using a Gaussian profile; spectrum ' + $
            strtrim(k+1L,2L)+'/'+strtrim(sl,2L)+'.'
        widget_control,stawid,set_value=msg
      endif ;; usestawid

      ;;====================----------
      ;; First calculating the position of the center spectrum; pos0:

      y=reform(double(refimage[(lpos[k]-2L*lwid)   >0L: $
                               (lpos[k]+2L*lwid)   <(s[1L]-1L),refrow] - $
                  min(refimage[(lpos[k]-2L*lwid-5L)>0L: $
                               (lpos[k]+2L*lwid+5L)<(s[1L]-1L),refrow])))
      dy=dblarr(n_elements(y))+1d0

      ;; Calling the line fitting routine MPCURVEFIT:
      a[0L:2L]=0d0 & a[4L]=total(y)
      yfit=mpcurvefit(x,y,1/dy,a,sigpar,/autoderivative,chisq=chisq, $
               dof=dof,errmsg=errmsg, $
               function_name='p3d_wavecal_correct_maskpos_gauss', $
               iter=nit,itmax=20L,nfev=nfev,parinfo=parinfo, $
               quiet=verbose lt 3,status=status)

      if status le 0L then begin
        msg='Failure when executing MPCURVEFIT.'
        errmsg=[msg,errmsg]
        goto,error_handler
      endif

      pos0=lpos[k]+a[2L]
      if pos0 ge lwid and pos0 le s[1L]-lwid then begin
        out[k,refrow]=pos0
      endif else begin
        ok[k]=0L
      endelse

      if monitor ge 1L then begin
        plot,x,y,psym=-4,xrange=[-1,1]*8L*lwid,xstyle=1, $
             xtitle='offset position [pixels]'
        oplot,x,yfit,thick=2,psym=-7,color=3b
        xpos=0.10*(!x.crange[1L]-!x.crange[0L])+!x.crange[0L]
        ypos=0.85*(!y.crange[1L]-!y.crange[0L])+!y.crange[0L]
        tmp='arc line '+strtrim(k+1L,2L)+'/'+ssl+' :: spec='+ $
            strtrim(refrow+1L,2L)+'/'+strtrim(s[2L],2L)
        xyouts,xpos,ypos,tmp
        wait,0.2
      endif ;; monitor ge 1L

      ;;====================----------
      ;; Calculating the position of the remaining spectra; combined loop to
      ;;   first increasing, and then decreasing, spectrum number:

      for j=0,1 do begin
        pos=pos0
        for L=lstart[j],lfinal[j],stride[j] do begin
          if ok[k] then begin
            rpos=(round(pos+loffsets[k,L]-loffsets[k,L-stride[j]]) $
                  >0L)<(s[1L]-1L)
            y=reform(double(refimage[(rpos-2L*lwid)   >0L: $
                                     (rpos+2L*lwid)   <(s[1L]-1L),L] - $
                        min(refimage[(rpos-2L*lwid-5L)>0L: $
                                     (rpos+2L*lwid+5L)<(s[1L]-1L),L])))

            ;; Calling the fitting routine MPCURVEFIT:
            a[0L:2L]=0d0 & a[4L]=total(y)
            yfit=mpcurvefit(x,y,1/dy,a,sigpar,/autoderivative,chisq=chisq, $
                     dof=dof,errmsg=errmsg, $
                     function_name='p3d_wavecal_correct_maskpos_gauss', $
                     iter=nit,itmax=20L,nfev=nfev,parinfo=parinfo, $
                     quiet=verbose lt 3,status=status)

            if status le 0L then begin
              msg='Failure when executing MPCURVEFIT.'
              errmsg=[msg,errmsg]
              goto,error_handler
            endif

            if ~(monitor-2L) then begin
              plot,x,y,psym=-4,xrange=[-1,1]*8L*lwid,xstyle=1, $
                   xtitle='offset position [pixels]'
              oplot,x,yfit,thick=2,psym=-7,color=3b
              xpos=0.10*(!x.crange[1L]-!x.crange[0L])+!x.crange[0L]
              ypos=0.85*(!y.crange[1L]-!y.crange[0L])+!y.crange[0L]
              tmp__=j?'[decrementing counter]':'[incrementing counter]'
              tmp='arc line '+strtrim(k+1L,2L)+'/'+ssl+' :: spec='+ $
                strtrim(L+1L,2L)+'/'+strtrim(s[2L],2L)+' :: '+tmp__
              xyouts,xpos,ypos,tmp
              wait,0.1
            endif ;; ~(monitor-2L)

            pos=rpos+a[2L] & rpos=round(pos)
            if rpos ge lwid and rpos le s[1L]-lwid then $
               out[k,L]=pos else ok[k]=0L

          endif ;; ok[k]
        endfor ;; L=lstart[j],lfinal[j],stride[j]
      endfor ;; j=0,1
    endfor ;; k=0L,sl-1L

    ;;========================================------------------------------
    ;;========================================------------------------------
    ;;========================================------------------------------

  endif else begin ;; strlowcase(method) eq 'gaussian'

    ;;========================================------------------------------
    ;;========================================------------------------------
    ;;========================================------------------------------
    ;; Using linear weighting:

    for k=0L,sl-1L do begin

    ;;====================----------
    ;; First calculating the position of the center spectrum; pos0:

      if ok[k] then begin
        m=0L
        repeat begin
          m++
             sum=total(refimage[(lpos[k]-2L*lwid)>0L: $
                                (lpos[k]+2L*lwid)<(s[1L]-1L),refrow])
          weight=total(refimage[(lpos[k]-2L*lwid)>0L: $
                                (lpos[k]+2L*lwid)<(s[1L]-1L),refrow]*x)
          pos0=lpos[k]+weight/sum
        endrep until m eq nit or pos0 lt lwid or pos0 ge s[1L]-lwid

        if pos0 ge lwid and pos0 le s[1L]-lwid then out[k,refrow]=pos0 else $
           ok[k]=0L
      endif ;; ok[k]

      ;;====================----------
      ;; Calculating the position of the remaining spectra; combined loop to
      ;;   first increasing, and then decreasing, spectrum #:

      for j=0,1 do begin
        pos=pos0
        for L=lstart[j],lfinal[j],stride[j] do begin
          if ok[k] then begin
            rpos=round(pos+loffsets[k,L]-loffsets[k,L-stride[j]])
            m=0L
            while m++ lt nit and rpos ge lwid and rpos lt s[1L]-lwid-2 do begin
              spec=refimage[(rpos-2L*lwid)>0L:(rpos+2L*lwid)<(s[1L]-1L),L] - $
                             min(refimage[(rpos-2L*lwid-5L)>0L: $
                                          (rpos+2L*lwid+5L)<(s[1L]-1L),L])
                 sum=total(spec)
              weight=total(spec*x)
            
              if sum gt 0d0 then begin
                pos=rpos+weight/sum
                rpos=round(pos)
              endif
            endwhile ;; m++ lt nit and rpos ge lwid and rpos lt s[1L]-lwid

            if rpos ge lwid and rpos lt s[1L]-lwid-2L then out[k,L]=pos else $
               ok[k]=0L
          endif ;; ok[k]
        endfor ;; L=lstart[j],lfinal[j],stride[j]
      endfor ;; j=0,1
    endfor ;; k=0L,sl-1L
  endelse ;; strlowcase(method) eq 'gaussian'

  if monitor ge 1L then device,decomposed=decomposed

  inlines=where(ok)

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_wavecal_correct_maskpos
