;+
; NAME:
;         p3d_wavecal_dispmask_add_spec
;
;         $Id: p3d_wavecal_dispmask_add_spec.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         This routine adds numbered rows to a dispersion mask, that is given
;         in parameter form. Each added row holds the same parameters as the
;         preceeding row. A purpose of this routine is to add calibration
;         spectra for instance, which were not used in the creation of the
;         dispersion mask.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: wavelength calibration
;
; CALLING SEQUENCE:
;         p3d_wavecal_dispmask_add_spec,inparams,rownum,outparams, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         inparams        - A two-dimensional array with the parameters of the
;                           polynomial fit (2nd dimension) of the wavelength
;                           array. Different spectra are stacked in the first
;                           dimension.
;         rownum          - A one-dimensional array of integer indices
;                           indicating where to add lines.
;
; KEYWORD PARAMETERS:
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         outparams       - A two-dimensional array with the parameters of the
;                           polynomial fits - withe added rows (according to
;                           ROWNUM).
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         15.10.2008 - Converted from original routine dispmask_add_spec of
;                      Thomas Becker. /CS
;-

PRO p3d_wavecal_dispmask_add_spec,inparams,rownum,outparams,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_wavecal_dispmask_add_spec: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_wavecal_dispmask_add_spec'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(inparams)
  if s[0L] ne 2L or (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='INPARAMS [1] must be set; to a two-dimensional array of decima' + $
           'l type.'
    goto,error_handler
  endif

  sb=size(rownum)
  nrownum=sb[sb[0L]+2L]
  if ~nrownum then begin
    outparams=inparams
    return
  endif else if sb[0L] gt 1L or $
     (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='ROWNUM [2] must be set to a one-dimensional array of decimal type.'
    goto,error_handler
  endif
  if nrownum eq 1L and rownum[0L] eq -1L then begin
    msg='No spectra were added.'
    error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
              verbose=verbose ge 2)
    outparams=inparams
    return
  endif

  ;;========================================------------------------------
  ;; Adding the rows indicated by ROWNUM:

  nspec=s[1L]+nrownum

  specid=lonarr(nspec)+1L
  k=-1L
  while ++k lt nrownum do $
     if rownum[k] ge 0L and rownum[k] le nspec then specid[rownum[k]]=0L

  outparams=dblarr(nspec,s[2L])
  specpos=where(specid,count)
  if count eq s[1L] then outparams[specpos,*]=inparams

  calibpos=where(1L-specid,count)
  if count ne 0L then begin
    k=-1L
    while ++k lt count do outparams[calibpos[k],*]=~calibpos[k]? $
                              inparams[0L,*]:outparams[calibpos[k]-1L,*]
  endif ;; count ne 0L

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
                        verbose=verbose,/error)
  return
END ;;; procedure: p3d_wavecal_dispmask_add_spec
